;;;; TCP Connection Manager
;;;;
;;;; This code is an addition to the Allegro CL IPC package, which
;;;; facilitates writing TCP servers.  It consists to three primary
;;;; functions.
;;;;
;;;; (start-connection-manager port)
;;;; (accept-connection) => stream
;;;; (stop-connection-manager)
;;;;
;;;; The application *must* stop the connection manager when it is
;;;; finished (unwind-protect highly recommended), and it is an error
;;;; to try to start the connection manager when it is already running.
;;;;

(in-package 'ipc)
(export '(start-connection-manager 
	  accept-connection 
	  stop-connection-manager)) 

;;;
;;; Global variables used to maintain the state of an open connection
;;;

(defvar *cm-running* nil)
(defvar *cm-listen-socket-fd*)
(defvar *cm-listen-sockaddr*)


;;;
;;; Create a socket, bind it to the given port, and listen for
;;; connections on that socket.
;;;
(defun start-connection-manager (inet-port)
  (when *cm-running*
    (error "Connection Manager already running"))
  (let ((initialization-error t))
    (setf *cm-listen-sockaddr*
      (let ((sin (ff:make-cstruct 'sockaddr-in)))
	(bzero sin (ff::cstruct-length 'sockaddr-in))
	sin))
    (unwind-protect
	(progn
	  ;; Create a socket
	  (setq *cm-listen-socket-fd* (socket *af-inet* *sock-stream* 0))
	  (when (< *cm-listen-socket-fd* 0)
	    (perror "socket")
	    (setq *cm-listen-socket-fd* nil)
	    (error "with-connection-on-port failed"))
	  
	  ;; Bind the socket to the connection port
	  (setf (sockaddr-in-family *cm-listen-sockaddr*) *af-inet*)
	  (setf (sockaddr-in-port *cm-listen-sockaddr*)
	    (lisp_htons inet-port))
	  (unless (zerop (bind *cm-listen-socket-fd*
			       *cm-listen-sockaddr*
			       (ff::cstruct-length 'sockaddr-in)))
	    (perror "bind")
	    (error "with-connection-on-port failed"))
	  
	  ;; Enable listening for connections on that socket
	  (unless (zerop (unix-listen *cm-listen-socket-fd* 5))
	    (perror "listen")
	    (error "with-connection-on-port failed"))
	  
	  ;; Done with initialization
	  (setf initialization-error nil)
	  (setf *cm-running* t))
	
      ;; This is the unwind-protect condition handler
      (when initialization-error
	;; Always close socket when leaving
	(when *cm-listen-socket-fd*
	  (unix-close *cm-listen-socket-fd*))
	(setf *cm-listen-socket-fd* nil)
	(setf *cm-running* nil)
	))))


(defun accept-connection ()
  (let ((int (ff:make-cstruct 'unsigned-long))
	(timeval (ff:make-cstruct 'timeval))
	(mask-obj (ff:make-cstruct 'unsigned-long))
	(mask (ash 1 *cm-listen-socket-fd*))
	stream
	fd)
    
    (setf (timeval-sec timeval) 0
	  (timeval-usec timeval) 0)
    (setf (unsigned-long-unsigned-long mask-obj) mask)
    (if (zerop (select 32 mask-obj 0 0 timeval))
	
	;; no connection requests, return nil
	nil
      
      ;; accept the connection and return the stream
      (progn
	;; Accept connection
	(setf (unsigned-long-unsigned-long int)
	  (ff::cstruct-length 'sockaddr-in))
	(setq fd (accept *cm-listen-socket-fd* *cm-listen-sockaddr* int))
	(when (< fd 0)
	  (perror "accept")
	  (error "accept-connection failed"))
	    
	;; Turn the new socket into a stream, and call the function
	(setq stream (make-ipc-terminal-stream fd))
	
	;; return the stream
	stream
	))))


(defun stop-connection-manager ()
  (when *cm-running*
    (when *cm-listen-socket-fd*
      (unix-close *cm-listen-socket-fd*))
    (setf *cm-listen-socket-fd* nil)
    (setf *cm-running* nil)))

