;;;;
;;;; Support for command/control channel pairs
;;;;


;;;
;;; The channel structure
;;;
(defclass channel 
    ()
    ((io-stream :accessor io-stream :initform nil)
     (control-messages :accessor control-messages :initform nil)
     (command-messages :accessor command-messages :initform nil)))

;;;
;;; Public methods, used to manipulate channels
;;;

(defmethod read-command ((self channel))
  (read-command-or-control self 'command-messages))
  
(defmethod read-control ((self channel))
  (read-command-or-control self 'control-messages))

(defmethod send-command ((self channel) message)
  (send-command-or-control self message 'COMMAND))

(defmethod send-control ((self channel) message)
  (send-command-or-control self message 'CONTROL))

(defmethod command-message-available? ((self channel))
  (read-messages self)
  (not (null (command-messages self))))

(defmethod control-message-available? ((self channel))
  (read-messages self)
  (not (null (control-messages self))))

(defmethod any-message-available? ((self channel))
  (read-messages self)
  (not (null (or (control-messages self) (command-messages self)))))

(defmethod close-channel ((self channel))
  (if (io-stream self)
      (close (io-stream self))))

;;;
;;; Internal methods, specific to channel implementation
;;;

(defmethod send-command-or-control ((self channel) message c-or-c)
  (let ((stream (io-stream self)))
    (format stream "~S " (cons c-or-c message))
    (finish-output stream)))


(defmethod read-command-or-control ((self channel) c-or-c)
  (loop
    (when (slot-value self c-or-c)
      (return))
    (read-messages self))
  (let ((message (first (slot-value self c-or-c))))
    (setf (slot-value self c-or-c) (rest (slot-value self c-or-c)))
    (values message)))

;;;
;;; Read any messages from the stream and put them in the appropriate
;;; message queue.
;;;
(defmethod read-messages ((self channel))
  (let ((io-stream (io-stream self)))
    (loop
      (when (not (listen io-stream))
	(return))
      (let ((message (read io-stream)))
	(cond ((eql (first message) 'COMMAND)
	       (setf (command-messages self) (append (command-messages self)
						     (list (rest message)))))
	      ((eql (first message) 'CONTROL)
	       (setf (control-messages self) (append (control-messages self)
						     (list (rest message)))))
	      (t
	       (error "malformed message received: ~d" message)))))))


	      
  
  