;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; object SIPHON
;;;
;;; A siphon is an object that removes liquid from a vessel into
;;; the fuel tank of the truck that sets it, over time.
;;; The siphon has a RATE property that controls how fast the liquid
;;; flows.
;;;
;;; To use: Grasp with an arm, move the arm to a vessel, and SET the
;;; siphon.  It will then start transfering the liquid in the vessel
;;; into the fuel-tank.  It will stop if the fuel tank, the vessel,
;;; or the siphon is moved, or when there is no more liquid (whichever
;;; comes first).
;;;

(defvar *default-siphon-rate* 2)	; flow in fluid units/time

(defobject siphon (thingoid)
  (rate *default-siphon-rate*))

(defmethod set-object ((self siphon) setter &rest args)
  (let ((vessel (thing-at-arm (query self 'container))))
    (when (and (typep setter 'truck)
	       (typep vessel 'vessel))
      (start-siphon-process self vessel (fuel-tank setter)))))

(defun start-siphon-process (siphon from to)
  (let* ((start (actual-time))
	 (prev-time start))
    
    (start-process
     nil
     #'(lambda (tok time why)
	 (case why

	   (ADVANCE
	    ;; Initialize the process
	    (when (compare-times time '= start)
	      ;; If either the TO or FROM vessels move, siphoning stops
	      (add-condition tok to 'movement 'TO-MOVED)
	      (add-condition tok from 'movement 'FROM-MOVED)
	      (add-condition tok siphon 'movement 'SIPHON-MOVED)
	      (add-output-condition tok to 'amount-held 'TO-AMOUNT)
	      (add-output-condition tok from 'amount-held 'FROM-AMOUNT))
	   
	    ;; Move liquid
	    (update-siphon-process tok siphon prev-time time from to))
	   
	   (CONDITION
	    (case (name why)
	      ;; If either vessel or the siphon moves, siphoning stops
	      ((TO-MOVED FROM-MOVED SIPHON-MOVED)  (stop-process tok))
	      ;; update the process if someone asks about amount-held
	      ((TO-AMOUNT FROM-AMOUNT) (update-siphon-process tok
							      siphon
							      prev-time
							      time
							      from
							      to)))))
	 (setf prev-time time)))))

(defun update-siphon-process (tok siphon prev-time time from to)
  (when (compare-times time '/= prev-time)
    ;; During this period, the AMOUNT output conditions should be disabled,
    ;; because the POUR operations query the amount-from properties.
    ;; If they weren't disabled, there would be a lot of warnings
    ;; about circularities in the query update graph for AMOUNT HELD.
    
    (disable-output-condition tok 'FROM-AMOUNT)
    (disable-output-condition tok 'TO-AMOUNT)

    (let* ((stuff (query from 'composition))
	   (amount (min (* (query siphon 'rate)
			   (- time prev-time))
			(space-empty to)
			(query from 'amount-held)))
	   (amt-out (pour-out from amount))
	   (amt-in (pour-in to stuff amt-out)))
      
      (when (or (<= amt-out 0) (<= amt-in 0))
	(stop-process tok)))
    
    (enable-output-condition tok 'FROM-AMOUNT)
    (enable-output-condition tok 'TO-AMOUNT)))
