
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; object RADIO
;;;
;;; A RADIO is an active sensor that can be tuned to a frequency,
;;; and sends messages to the RECEIVER of other radios tuned to that
;;; frequency.  A RADIO then generates a report when it receives a message.
;;;
;;; The radio is operated with SET operations, with arguments as follows:
;;;
;;;   ON/OFF/nothing : Turns radio ON, OFF, or toggles, as with all
;;;                    active sensors
;;;   FREQUENCY <freq>  : Sets the radio's broadcasting and receiving
;;;                       frequency to <freq>
;;;   SEND <message args> : sends the list of message args to other
;;;                         activated radios tuned to the same frequency
;;;
;;; Radios sorta' skirt the normal containment/visibility concepts
;;; of normal objects, since they explicitly choose what objects receive
;;; the messages.
;;;

(defasensor radio
    :extra-properties ((receiver)
		       (frequency (random 100)))
    :sensing-method    'radio-waves	; Irrelevant, since radios aren't
					;   implemented by sensing each other,
					;   only themselves.
    :watched-properties '(receiver)	; When the RECEIVER
    :watched-objects   #'myself		; of the radio changes state,
    :sensed-properties '(receiver)	; Report message received by
    :sensing-scope     #'myself)	; the radio.


;;;
;;; This variable keeps track of every radio ever used (even from
;;; simulation to simulation).  So it may get cluttered, and slow
;;; performance.  It may be a good idea to set it to NIL after
;;; a simuation ends.
;;;

(defvar *RADIOS-IN-WORLD* nil)

;;;
;;; The SET-OBJECT method allows the usual active sensor semantics,
;;; and adds it's own protocol for radios.
;;;

(defmethod set-object :after ((self radio) setter &rest args)
  (case (first args)
    (FREQUENCY   (setp self 'frequency (second args))
		 (register-radio self))
    (SEND        (when (eq (query self 'power) 'ON)
		   (send-radio-message self (cdr args))))))

(defun register-radio (radio)
  (when (not (member radio *radios-in-world*))
    (push radio *radios-in-world*)))

(defun send-radio-message (radio message)
  (let ((f (query radio 'frequency)))
    (mapc
     #'(lambda (r)
	 (when (and (not (eq radio r))
		    (eq f   (query r 'frequency))
		    (eq 'ON (query r 'power)))
	   (setp r 'receiver message)))
     *radios-in-world*)))

;;;
;;; DESTROY-OBJECT
;;;
;;; When a radio is destroyed, clean it out of the global radio list
;;;

(defmethod destroy-object :before ((self radio))
  (setf *radios-in-world* (delete self *radios-in-world*)))

  
  

