
;;;
;;; START-MOTION-WATCHING-PROCESS watcher container action
;;;
;;; WATCHER : some SIM-OBJECT
;;; CONTAINER : A locality function of one argument: the WATCHER
;;; ACTION : A function of one argument: the WATCHER
;;;
;;; This creates a process that executes ACTION (a function that takes
;;; the WATCHER as an argument), whenever the WATCHER moves,
;;; or anything moves in or out of the CONTAINER (a function that
;;; returns a container object when passed the WATCHER).
;;; Note that the actual container object may depend on the WATCHER's
;;; own location, as CONTAINER is re-evaluated whenever the WATCHER
;;; moves.
;;;
;;; The token of the resulting process is returned.
;;;
;;; Example: invoke the SET action of object FOOBAR whenever FOOBAR
;;; moves, or anything in FOOBAR's enclosing map-node enters or exits:
;;;
;;;   (start-motion-watching-process foobar 
;;;	 	 	             #'my-map-node
;;;			             #'(lambda (the-watcher)
;;;				          (setp the-watcher 'set t)))
;;;

(defun start-motion-watching-process (watcher container action)
  (let ((start t))
    (start-process
     nil
     #'(lambda (tok time why)
	 (case why
	   (ADVANCE
	    (cond
	     
	     ;; On startup, link to all conditionals
	     (start
	      
	      ;; Make a condition that gets tripped when the sensor is moved
	      ;; When an object is moved (either directly, or by virtue
	      ;; of an enclosing container being moved), it's MOVEMENT
	      ;; property gets SETP'ed
	      
	      (add-condition tok watcher 'movement 'movement-condition)

	      ;; Make all conditions that get tripped when the watched
	      ;; object changes state: note: because we set current-watched
	      ;; to nil, make-all... will automatically set up necessary
	      ;; maintainance conditions.

	      (make-all-motion-conditions tok watcher container)
	      (setf start nil))))
	     
	   ;; Something about the world has changed: respond to it
	   
	   (CONDITION
	    (respond-to-motion-condition tok watcher container (name why) 
					 action)))))))

(defun make-all-motion-conditions (tok watcher container)
  (let ((watchedobj (take-first (funcall container watcher))))
    
    ;; If the object the sensor is watching doesn't match
    ;; the object it should be watching, then remove all conditionals
    ;; to the former object, and link conditionals to the latter.
    
    (when (typep watchedobj 'container)
      (delete-condition tok 'CONTENTS)
      (add-condition tok watchedobj 'CONTENTS 'CONTENTS))))

(defun respond-to-motion-condition (tok watcher container name action)
  ;; Something has moved: either the watcher, or something in the watcher's
  ;; watched vicinity: invoke the action
  (funcall action watcher)
   
  (cond

   ;; The sensor has moved: make sure the object it should be watching
   ;; is still the object it is watching.

   ((eq name 'MOVEMENT-CONDITION)
    (make-all-motion-conditions tok watcher container))
    
   ;; something else has moved: do nothing
   ((eq name 'CONTENTS)  nil)
   
   (t
    (format t "WARNING: Watching process for ~S got an unknown signal ~S~%"
	    watcher name))))
