;;**************************************************************
;;;  Definition and methods for TRUCK-DISPLAYER
;;;**************************************************************

(defstruct (td (:constructor really-make-td)
               (:print-function print-td))
  displayed-object
  subdisplayers 
  cnt-loc-disp 
  map-displayer
  arm-displayers
  data)

(defun td-truck (td)
  (td-displayed-object td))

(defun print-td (self stream depth)
  (declare (ignore depth))
  (format stream "#{Displayer for ~s}" (td-truck self)))

;;;************************************************************
;;;  Constructor

(defun make-truck-displayer (truck 
                             world 
                             time-controller 
                             display-name
                             &optional (origin (make-point :x 0 :y 0)))
  (let* ((td            (really-make-td))
         (gauges        (td-make-gauges truck time-controller origin))
         (bays          (td-make-bays truck origin))
         (arms          (td-make-arms td truck bays))
         (time-displayer (find-displayer-in-list time-controller gauges))
         (time-max-x    (rectangle-max-x (display-region time-displayer)))
         (external      (make-loc-displayer  (make-point :x (+ time-max-x 10) :y 22)
                                             6 3))
         (map           (td-make-map truck 
                                     world 
                                     external 
                                     (find-displayer-in-list (second (arms truck)) 
                                                             bays)))
         (external-max-x (+ (rectangle-max-x (display-region external)) 10))
         (map-max-y (+ (rectangle-max-y (display-region map)) 10)))
    (setf (td-displayed-object td) truck)
    (setf (td-subdisplayers td) (append bays arms gauges (list external map)))
    (setf (td-cnt-loc-disp td) external)
    (setf (td-map-displayer td) map)
    (setf (td-arm-displayers td) arms)
    (setf (td-data td) 
      (disp.init td 
                 (make-rectangle :origin origin
                                 :other-point 
                                 (make-point :x external-max-x
                                             :y map-max-y))
                 display-name))
    td))

;;;********************************************************************

(defun td-make-bays (truck origin)
  (let ((left-x (+ (point-x origin) 10))
        (top-y  (+ (point-y origin) 5))
        (displayers '()))
    (push (make-bay-displayer (tire-bay truck)  
                              "Tires" 
                              (make-point :x left-x :y top-y) 
                              1 1
                              '(top left) 'horizontal)
          displayers)
    (push (make-bay-displayer (weapon-bay truck) 
                              "Weapons" 
                              (make-point :x (+ left-x 90) :y top-y)
                              1 1
                              '(top left) 'horizontal)
          displayers)
    (setf top-y (rectangle-max-y (display-region (car displayers))))
    (push (make-bay-displayer  (first (bays truck)) 
                               NIL
                               (make-point :x left-x :y (+ top-y 10))
                               1 9
                               '(top left) 'horizontal)
          displayers)
    (setf top-y (rectangle-max-y (display-region (car displayers))))
    (push (make-bay-displayer  (second (bays truck)) 
                               NIL
                               (make-point :x left-x  :y (+ top-y 10))
                               1 9
                               '(top left) 'horizontal)
          displayers)
    (setf top-y (rectangle-max-y (display-region (car displayers))))
    (push (make-bay-displayer (first (arms truck)) 
                              NIL
                              (make-point :x left-x :y (+ top-y 10))
                              1 8
                              '(top right) 'horizontal)
          displayers)
    (setf top-y (rectangle-max-y (display-region (car displayers))))
    (push (make-bay-displayer (second (arms truck)) 
                              NIL
                              (make-point :x left-x :y (+ top-y 10))
                              1 8
                              '(top right) 'horizontal)
          displayers)
    displayers))

;;;*******************************************************

(defun td-make-arms (td truck bay-displayers)
  (mapcar #'(lambda (arm string) 
              (make-arm-displayer arm 
                                  (find-displayer-in-list arm bay-displayers)
                                  td
                                  string))
          (arms truck) 
          (list "1" "2")))

;;;*******************************************************

(defun td-make-gauges (truck time-controller origin)
  (list  
   (make-fuel-displayer (fuel-tank truck) (add-point origin :x 190))
   (make-heading-displayer (heading-setting truck) (add-point origin :x 250))
   (make-speed-displayer (speed-setting truck) (add-point origin :x 320))
   (make-info-displayer time-controller "Time"    (add-point origin :x 400))
   (make-info-displayer (status-gauge truck) 
                        "Status" 
                        (add-point origin :x 400 :y 40))))

;;;*******************************************************

(defun td-make-map (truck world external-display arm2-display)
  (let* ((top-x 2)
         (top-y (+ 10 (rectangle-max-y (display-region arm2-display))))
         (width (rectangle-max-x (display-region external-display))))
    (make-map-displayer truck 
                        (world-map world) 
                        (make-point :x top-x :y top-y)
                        width)))

;;;****************************************************************
;;;  Finding displayers

(defun find-displayer-in-list (obj disp-list)
  (find-if #'(lambda (disp) (equal (displayed-object disp) obj)) disp-list))

;;;****************************************************************
;;;****************************************************************
;;;  Method handlers

(defmethod draw ((self td))
  (with-active-display (td-data self)
    (mapc #'draw (td-subdisplayers self))
    (let ((env (query (td-displayed-object self) 'container)))
      (when env 
        (loc-displayer-set-containment (td-cnt-loc-disp self) env)))))

(defmethod terminate  ((self td))
  (disp.terminate (td-data self))
  (setf (td-data self) NIL)
  (values))

(defmethod displayed-object ((self td))
  (td-displayed-objec self))

;;;  Updating
;;;
;;;  The truck will be the original receiver of all update requests.
;;;  It has a couple of roles:
;;;     1.  To set the display so all its subdisplayers write to the 
;;;         right screen.
;;;     2.  To handle the case where the truck's current location has 
;;;         changed.
;;;     4.  Otherwise, to dispatch the update request to its subdisplayers.

(defmethod update-display ((self td) object)
  (with-active-display (td-data self)
    (let ((the-truck (td-truck self)))
      (cond
       ((and (member the-truck (holdings object))
             (not (eq object (displayed-object (td-cnt-loc-disp self)))))
        ;; Here the current location of the truck has been changed.
        ;; We need to notify the displayer especially, because it can't 
        ;; distinguish between changes to its *contents* and changes to 
        ;; what container it is *displaying*
        (loc-displayer-set-containment (td-cnt-loc-disp self)
                                       object)
        (update-display (td-map-displayer self) object))
       ((and (not (member the-truck (holdings object)))
             (eq object (displayed-object (td-cnt-loc-disp self))))
        ;; This is the opposite.  Need to unset the current 
        ;; location displayer
        (loc-displayer-unset-containment (td-cnt-loc-disp self))
        (update-display (td-map-displayer self) object))
       (t ;; Otherwise a simple dispatch to the subdisplayers
        (mapc #'(lambda (subd) (update-display subd object))
              (td-subdisplayers self))))))
  NIL)

;;;*************************************

(defmethod display-gripper ((self td) arm container position)
  (let* ((truck (td-displayed-object self))
         (tank (fuel-tank truck)))
    (with-active-display (td-data self)
      (cond
       ((and (eq truck container)
             (eq tank (nth-contents truck position)))
        (display-gripper (find-displayer-in-list tank (td-subdisplayers self))
                         arm
                         tank
                         0))
       (t (mapc #'(lambda (subd) (display-gripper subd arm container position))
                (td-subdisplayers self)))))))

(defmethod undisplay-gripper ((self td) arm)
  (with-active-display (td-data self)
    (mapc #'(lambda (subd) (undisplay-gripper subd arm))
          (td-subdisplayers self))))

