(shadow '(color display *version* screen))  ;; to avoid name conflicts

(require :clx)
(use-package :xlib)

(defvar *default-truckworld-background* "Gainsboro")
(defvar *default-truckworld-foreground* "Black")

;*****************************************************;
;*    X globals                                      *;
;*****************************************************;

(defstruct x-display
  window gcontext display
  x0 y0 w h scale font color fonts
  color? foreground background top-shadow bottom-shadow)

(defun grafp.set-origin (disp x0 y0)
  (setf (x-display-x0 disp) x0)
  (setf (x-display-y0 disp) y0)
  disp)

(defun grafp.get-origin (disp)
  (values (x-display-x0 disp) (x-display-y0 disp)))

(defun ->display-coordinates (disp x y &optional w h)
  (let ( (x0    (x-display-x0 disp))
         (y0    (x-display-y0 disp))
         (scale (x-display-scale disp)) )
    (values (round (* scale (+ x x0)))
            (round (* scale (+ y y0)))
            (if w (round (* scale w)) 0)
            (if h (round (* scale h)) 0))))

(defun grafp.set-drawing-region (disp x y w h)
  (setf (x-display-x0 disp) x)
  (setf (x-display-y0 disp) y)
  (setf (x-display-w  disp) w)
  (setf (x-display-h  disp) h)
  (multiple-value-bind (cx cy cw ch) 
                       (->display-coordinates disp 0 0 w h)
    (let ( (gc (x-display-gcontext disp)) )
      (setf (gcontext-clip-x gc) cx)
      (setf (gcontext-clip-y gc) cy)
      (setf (gcontext-clip-mask gc) (list 0 0 (+ cw 1) (+ ch 1)))))
  (values))

(defun grafp.get-drawing-region (disp)
  (values (x-display-x0 disp) (x-display-y0 disp)
          (x-display-w  disp) (x-display-h  disp)))
 
;**********************************************************************;
;**********************************************************************;
;***       Generic drawers for lines, circles, and polygons           *;
;**********************************************************************;
;**********************************************************************;

(defun grafp.draw-line (disp x1 y1 x2 y2)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (draw-line (x-display-window disp) (x-display-gcontext disp)
                 x1 y1 x2 y2)))
  (values))

(defun grafp.draw-circle (disp x y radius filled?)
  (multiple-value-bind (x y radius)
                       (->display-coordinates disp x y radius)
    (draw-arc (x-display-window disp) (x-display-gcontext disp)
              (- x radius) (- y radius) 
              (* 2 radius) (* 2 radius) 0 (* pi 2.0) filled?))
  (values))

(defun grafp.draw-3d-circle (disp up/down x y radius)
  (multiple-value-bind (x y radius)
                       (->display-coordinates disp x y radius)
    (let ( (x1 (- x radius)) (y1 (- y radius)) (w (* 2 radius))
           (window (x-display-window disp))
           (gcontext (x-display-gcontext disp)) 
           (top-color (if (eq up/down :up)
                          (x-display-top-shadow disp)
                          (x-display-bottom-shadow disp)))
           (bot-color (if (eq up/down :up)
                          (x-display-bottom-shadow disp)
                          (x-display-top-shadow disp))) )
      (unwind-protect
        (progn
          (setf (gcontext-foreground gcontext) top-color)
          (draw-arc window gcontext x1 y1 w w  (* pi 0.35) (* pi 1.01) nil)
          (setf (gcontext-foreground gcontext) bot-color)
          (draw-arc window gcontext x1 y1 w w  (* pi 1.35) (* pi 1.01) nil))
        (setf (gcontext-foreground gcontext) (x-display-foreground disp)))))
  (values))

(defun grafp.draw-polygon (disp points filled?)
  (let ( (real-points (polygon-point-list disp points)) )
    (draw-lines (x-display-window disp) (x-display-gcontext disp)
                real-points :fill-p filled?))
  (values))

(defun polygon-point-list (disp points)
  (multiple-value-bind (x0 y0)
                       (->display-coordinates disp (car points) (cadr points))
    (let ( (point-list (list x0 y0)) )
      (do ( (points (cddr points) (cddr points)) )
          ((null points) point-list)
        (multiple-value-bind (x y)
                             (->display-coordinates disp 
                                                    (car points) (cadr points))
          (setf point-list (cons x (cons y point-list))))))))

(defun grafp.draw-rectangle (disp x1 y1 x2 y2 filled?)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (draw-rectangle (x-display-window disp) (x-display-gcontext disp)
                      (min x1 x2) (min y1 y2)
                      (abs (- x2 x1)) (abs (- y2 y1))
                      filled?)))
  (values))

(defun grafp.draw-3d-rectangle (disp up/down x1 y1 x2 y2)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (let ( (window (x-display-window disp))
             (gcontext (x-display-gcontext disp)) 
             (top-color (if (eq up/down :up)
                            (x-display-top-shadow disp)
                            (x-display-bottom-shadow disp)))
             (bot-color (if (eq up/down :up)
                            (x-display-bottom-shadow disp)
                            (x-display-top-shadow disp))) )
        (unwind-protect
          (progn 
            (setf (gcontext-foreground gcontext) top-color)
            (draw-line window gcontext x1 y2 x1 y1)
            (draw-line window gcontext x1 y1 x2 y1)
            (setf (gcontext-foreground gcontext) bot-color)
            (draw-line window gcontext x2 y1 x2 y2)
            (draw-line window gcontext x2 y2 x1 y2))
          (setf (gcontext-foreground gcontext) (x-display-foreground disp))))))
  (values))

(defun grafp.draw-rectangle-interior (disp x1 y1 x2 y2 filled?)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (let ( (x (+ (min x1 x2) 1)) (y (+ (min y1 y2) 1))
             (w (abs (- x2 x1))) (h (abs (- y2 y1))) )
        (if (and (> w 2) (> h 2))
          (draw-rectangle (x-display-window disp) (x-display-gcontext disp)
                          x y (- w 1) (- h 1) filled?)))))
  (values))

;**********************************************************************;
;**********************************************************************;
;***       Clearing rectangles, display                               *;
;**********************************************************************;
;**********************************************************************;

(defun grafp.clear-rectangle (disp x1 y1 x2 y2)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (let ( (gcontext (x-display-gcontext disp))
             (x1 (min x1 x2)) (y1 (min y1 y2))
             (x2 (+ (max x1 x2) 1)) (y2 (+ (max y1 y2) 1)) )
        (unwind-protect
          (progn
            (setf (gcontext-foreground gcontext) (x-display-background disp))
            (draw-rectangle (x-display-window disp) gcontext
                            x1 y1 (- x2 x1) (- y2 y1) T))
          (setf (gcontext-foreground gcontext) (x-display-foreground disp))))))
  (values))

(defun grafp.clear-rectangle-interior (disp x1 y1 x2 y2)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (let ( (gcontext (x-display-gcontext disp))
             (x (+ (min x1 x2) 1)) (y (+ (min y1 y2) 1))
             (w (abs (- x2 x1))) (h (abs (- y2 y1))) )
        (if (and (> w 2) (> h 2))
         (unwind-protect
          (progn
            (setf (gcontext-foreground gcontext) (x-display-background disp))
            (draw-rectangle (x-display-window disp) gcontext
                            x y (- w 1) (- h 1) T))
          (setf (gcontext-foreground gcontext) (x-display-foreground disp)))))))
 (values))

(defun grafp.clear-display (disp)
  (let ( (gcontext (x-display-gcontext disp))
         (window   (x-display-window disp)) )
    (unwind-protect
      (progn
        (setf (gcontext-foreground gcontext) (x-display-background disp))
        (draw-rectangle window gcontext 0 0
                        (drawable-width window) (drawable-height window) T))
      (setf (gcontext-foreground gcontext) (x-display-foreground disp))))
  (values))

;**********************************************************************;
;**********************************************************************;
;***       Text and Fonts                                             *;
;**********************************************************************;
;**********************************************************************;

(defun grafp.set-font (disp font)
  (when font
    (setf (gcontext-font (x-display-gcontext disp)) font))
  (setf (x-display-font disp) font))

(defun grafp.get-font (disp)
  (x-display-font disp))

(defun grafp.text-extent (disp str)
  (multiple-value-bind (width ascent descent)
                       (text-extents (x-display-font disp) str)
    (values width
            (+ ascent descent)
            descent)))

(defun grafp.draw-text (disp str x y)
  (multiple-value-bind (x y)
                       (->display-coordinates disp x y)
    (draw-glyphs (x-display-window disp) (x-display-gcontext disp)
                 x y str)
    (values)))

(defun grafp.font (disp key)
  (let ( (x-font (cdr (assoc key (x-display-fonts disp) :test #'eq))) )
    (cond ((null x-font)
           (let ( (x-font (open-font (x-display-display disp)
                                     (case key ((:tiny)   "5x8")
                                               ((:small)  "6x10")
                                               ((:medium) "8x13")
                                               ((:large)  "9x15")
                                               ((:huge)   "10x20")
                                               (otherwise "8x13")))) )
               (when (null x-font)
                  (error "(GRAFP.FONT) Can't find X font ~s." key))
               (setf (x-display-fonts disp)
                     (cons (cons key x-font) (x-display-fonts disp)))
               x-font))
          (T x-font))))

;**********************************************************************;
;**********************************************************************;
;***       Colors                                                     *;
;**********************************************************************;
;**********************************************************************;

(defstruct grafp-color
   red green blue
   internal)

(defun grafp.color (disp red green blue)
  (declare (ignore disp))
  (make-grafp-color :red red :green green :blue blue 
                    :internal (make-color :red red :green green :blue blue)))

(defun grafp.set-foreground (disp color)
  (when (x-display-color? disp)
    (let ( (color (if (grafp-color-p color)
                      (alloc-color 
                        (screen-default-colormap
                          (display-default-screen (x-display-display disp)))
                        (grafp-color-internal color))
                      color)) )
       (setf (gcontext-foreground (x-display-gcontext disp)) color))))

(defun grafp.get-foreground (disp)
   (gcontext-foreground (x-display-gcontext disp)))

(defun grafp.set-erasure (disp erasure?)
  (let ( (color (if erasure?
                    (x-display-background disp)
                    (x-display-foreground disp))) )
    (setf (x-display-color disp) color)
    (setf (gcontext-foreground  (x-display-gcontext disp)) color)))

(defun grafp.get-erasure (disp)
  (eq (x-display-color disp) (x-display-background disp)))

(defun grafp.set-invert (disp invert?)
  (if invert?
      (setf (gcontext-function (x-display-gcontext disp)) boole-xor)
      (setf (gcontext-function (x-display-gcontext disp)) boole-1)))

(defun grafp.get-invert (disp)
  (eq boole-xor (gcontext-function (x-display-gcontext disp))))

;**********************************************************************;
;**********************************************************************;
;***       Bitmap Stuff
;**********************************************************************;
;**********************************************************************;


(defstruct x-bitmap
  w h picture)

(defun grafp.load-bitmap (disp pathname)
  (declare (ignore disp))
  (read-bitmap-file pathname))

(defun grafp.draw-bitmap (disp xbm x1 y1 x2 y2)
  (multiple-value-bind (x1 y1)
                       (->display-coordinates disp x1 y1)
    (multiple-value-bind (x2 y2)
                         (->display-coordinates disp x2 y2)
      (let ( (xoffset (truncate (- (- x2 x1) (image-width xbm)) 2))
             (yoffset (truncate (- (- y2 y1) (image-height xbm)) 2)) )
        (put-image (x-display-window disp) (x-display-gcontext disp)
                   xbm :x (+ x1 xoffset) :y (+ y1 yoffset) :bitmap-p T))))
  (values))

(defun grafp.bitmap-size (xbm)
  (values (image-width xbm) (image-height xbm)))

;**********************************************************************;
;**********************************************************************;
;***       Initialize and terminate                                   *;
;**********************************************************************;
;**********************************************************************;

(defmethod draw-object (obj) (declare (ignore obj)) (values))

(defmacro grafp.with-active-display (disp &rest body)
  `(unwind-protect
     (progn ,@body)
     (grafp.force-output ,disp)))

(defun grafp.force-output (disp)
  (display-force-output (x-display-display disp)))

(defun parse-x-defaults ()
  (cond ((probe-file "~/.Xdefaults")
         (let ( (dbase (make-resource-database)) )
           (read-resources dbase "~/.Xdefaults")
           (values 
             (get-resource dbase "foreground" "*" 
                                 '("TW" "*") '("application" "*"))
             (get-resource dbase "background" "*" 
                                 '("TW" "*") '("application" "*"))
             (get-resource dbase "topshadow" "*" 
                                 '("TW" "*") '("application" "*"))
             (get-resource dbase "bottomshadow" "*" 
                                 '("TW" "*") '("application" "*")))))
        (T (values nil nil nil nil))))

(defun calculate-default-colors (screen fore back top bot)
  (let ( (colormap (screen-default-colormap screen))
         (color?   (> (screen-root-depth screen) 1)) )
    (cond (color?
            (multiple-value-bind (backcolor backxcolor)
               (alloc-color colormap (or back *default-truckworld-background*))
             (let ( (forecolor
                      (alloc-color colormap 
                                   (or fore *default-truckworld-foreground*))) )
               (multiple-value-bind (topcolor botcolor)
                  (calculate-default-shading colormap backxcolor 
                                                      top bot forecolor)
                 (values forecolor backcolor topcolor botcolor)))))
          (T
           (let ( (forecolor (alloc-color colormap (or fore "black")))
                  (backcolor (alloc-color colormap (or back "white"))) )
             (values forecolor backcolor forecolor forecolor))))))

(defun calculate-default-shading (colormap backxcolor top bot forecolor)
  (multiple-value-bind (red green blue)
                       (color-rgb backxcolor)
    (cond ((and top bot)
           (values (alloc-color colormap top)
                   (alloc-color colormap bot)))
          ((or (and (<= red 0.1) (<= green 0.1) (<= blue 0.1))
               (or  (>= red 0.9) (>= green 0.9) (>= blue 0.9)))
           (values forecolor forecolor))
          (T
           (values (alloc-color colormap 
                      (make-color :red   (min 1.0 (* 1.25 red))
                                  :green (min 1.0 (* 1.25 green))
                                  :blue  (min 1.0 (* 1.25 blue))))
                   (alloc-color colormap
                      (make-color :red   (* 0.75 red)
                                  :green (* 0.75 green)
                                  :blue  (* 0.75 blue))))))))

(defun grafp.open-window (displayer x y w h name
                                    &key (foreground nil)
                                         (background nil)
                                         (top-shadow nil)
                                         (bottom-shadow nil))
  (declare (ignore displayer))
  (let* ((display (parse-and-open-display name))
         (screen (display-default-screen display))
         (window (create-window :parent (screen-root screen)
                                :x x :y y
                                :width w :height h
                                :backing-store :always))
         (color?   (> (screen-root-depth screen) 1)))
    (change-property window 
                     :wm_name "Truckworld" 
                     :string 8 
                     :transform #'char-code)
    (multiple-value-bind (fore back top bot) (parse-x-defaults)
      (multiple-value-bind (forecolor backcolor topcolor botcolor)
          (calculate-default-colors screen (or foreground fore)
                                           (or background back)
                                           (or top-shadow top)
                                           (or bottom-shadow bot))
        (let ((gcontext (create-gcontext :drawable window)))
          (setf (gcontext-foreground gcontext) forecolor)
          (setf (gcontext-background gcontext) backcolor)
          (let ((disp (make-x-display :x0 0 :y0 0 :w w :h h :scale 1
                                      :window window
                                      :gcontext gcontext
                                      :display display
                                      :foreground forecolor
                                      :background backcolor
                                      :color? color?
                                      :top-shadow topcolor
                                      :bottom-shadow botcolor)))
            (map-window window)
            (grafp.clear-display disp)
            disp))))))

(defun parse-and-open-display (name)
  (multiple-value-bind (host display) (parse-display-name name)
    (open-display host :display display)))

;;;**************************************
;;;  Name can be one of the following:
;;;     NIL or "" => host = "", display = 0
;;;     "name"    => host = "name", display = 0
;;;     "name:num" => host = "name", display = num
;;;
;;;  Note that we don't allow 
;;;       name:num.num 
;;;  because is seems that the CLX version of open-display 
;;;  can't handle a display + screen, though the C version does (?)
;;;

(defun parse-display-name (name)
  (cond
   ((or (null name) 
        (and (stringp name) (= 0 (length name))))
    (values "" 0))
   ((not (stringp name)) 
    (error "Bad display name argument ~a" name))
   (t (let ((colon (position #\: name :test #'char=)))
        (cond 
         ((null colon) (values name 0))
         (t (let* ((display (subseq name 0 colon))
                   (host-string (subseq name (+ colon 1) (length name)))
                   (host-num (read-from-string host-string)))
              (when (not (integerp host-num))
                (error "Bad display name argument ~a" name))
              (values display host-num))))))))
              

(defun grafp.terminate (disp)
  (close-display (x-display-display disp)))
