;;;***************************************************************
;;; MAP.LISP
;*********

(defstruct (map-displayer (:print-function print-map-displayer)
                          (:constructor really-make-map-displayer))
  displayed-object
  frame 
  subdisplayers)  ; subdisplayers one for each node and road

(defun print-map-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer FOR ~a}" (map-displayer-displayed-object self)))

(defvar *map-border-width* 2
  "Horizontal separation between map's frame and the first object")
(defvar *map-border-height* 14
  "Vertical separation between map's frame and the first object")

(defun make-map-displayer (truck map-data origin width)
  (let* ((real-origin (add-point origin 
                                 :x *map-border-width* 
                                 :y *map-border-height*))
         (node-displays 
          (mapcar #'(lambda (node) 
                      (make-node-displayer node truck real-origin))
                  (nodes map-data)))
         (road-displays 
          (mapcar #'(lambda (road) 
                      (make-road-displayer road truck node-displays real-origin))
                  (links map-data)))
         (subdisplayers (append node-displays road-displays))
         (bottom (find-bottom-from-subdisplayers subdisplayers)))
    (really-make-map-displayer
     :displayed-object (list truck map-data)
     :frame (make-rectangle :origin origin 
                            :width width 
                            :height (+ (- bottom (point-y origin)) 
                                       *map-border-height*))
     :subdisplayers subdisplayers)))

(defun find-bottom-from-subdisplayers (subdisplayers)
  (apply #'max (mapcar #'rectangle-max-y (mapcar #'display-region subdisplayers))))

;;;**************************************

(defmethod displayed-object ((self map-displayer))
  (map-displayer-displayed-object self))

(defmethod draw ((self map-displayer))
  (disp.draw-3d-rectangle :down (map-displayer-frame self))
  (mapc #'draw (map-displayer-subdisplayers self)))

(defmethod display-region ((self map-displayer))
  (map-displayer-frame self))

(defmethod update-display ((self map-displayer) object)
  (mapc #'(lambda (subd) (update-display subd object))
        (map-displayer-subdisplayers self)))

;;;***************************************************************
;;; Nodes   
;;;

(defstruct (node-displayer (:print-function print-node-displayer)
                           (:constructor really-make-node-displayer))
  displayed-object
  highlighted
  name
  truck 
  frame)

(defun print-node-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer FOR ~a}" (node-displayer-displayed-object self)))

(defun make-node-displayer (node truck offset)
  (let ((display-rect (display-info node)))
    (really-make-node-displayer 
     :displayed-object node
     :truck truck
     :highlighted NIL
     :name (symbol-name (id node))
     :frame (make-rectangle :origin (add-points (rectangle-origin display-rect)
                                                offset)
                            :width (rectangle-width display-rect)
                            :height (rectangle-height display-rect)))))

(defmethod displayed-object ((self node-displayer))
  (node-displayer-displayed-object self))

(defmethod display-region ((self node-displayer))
  (node-displayer-frame self))

(defmethod draw ((self node-displayer))
  (let* ((name (node-displayer-name self))
         (rect (node-displayer-frame self))
         (width (rectangle-width rect))
         (font (multiple-value-bind (p1x p1y) 
                   (disp.with-font (disp.small-font) (disp.text-extent name))
                 (declare (ignore p1y))
                 (multiple-value-bind (p2x p2y)
                     (disp.with-font (disp.tiny-font)  (disp.text-extent name))
                   (declare (ignore p2y))
                   (cond  ((< p1x width) (disp.small-font))
                          ((< p2x width) (disp.tiny-font))
                          (T nil))))))
    (disp.draw-3d-rectangle :up rect)
    (when font
      (disp.with-font font
        (disp.draw-text name (rectangle-center rect) :center)))
    (node-displayer-update self)))


(defmethod update-display ((self node-displayer) object)
  (when (eq object (node-displayer-displayed-object self))
    (node-displayer-update self)))

(defun node-displayer-update (self)
  (when (and (not (node-displayer-highlighted self))
             (member (node-displayer-truck self)
                     (holdings (node-displayer-displayed-object self))))
    (node-displayer-highlight self))
  (when (and (node-displayer-highlighted self)
             (not (member (node-displayer-truck self)
                          (holdings (node-displayer-displayed-object self)))))
    (node-displayer-unhighlight self)))


(defun node-displayer-highlight (self)
  (let ((rect (node-displayer-frame self)))
    (disp.draw-rectangle (offset-rectangle rect :x 1 :y 1 :width -2 :height -2))
    (setf (node-displayer-highlighted self) T)))

(defun node-displayer-unhighlight (self)
  (disp.with-erasure (node-displayer-highlight self))
  (setf (node-displayer-highlighted self) NIL))

;;;******************************************************************

(defstruct (road-displayer (:print-function print-road-displayer)
                           (:constructor really-make-road-displayer))
  displayed-object 
  truck
  highlighted
  path-points 
  offset-path-points)

(defun print-road-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer FOR ~a}" (road-displayer-displayed-object self)))

                                              
(defvar *road-highlight-offset* 2
  "Number of pixels to offset the road's lines to make it 
   look doubled.")

(defun make-road-displayer (road truck node-displays offset)
  (let* ((start-node (start-node road))
         (start-dir  (start-node-direction road))
         (end-node   (finish-node road))
         (end-dir    (finish-node-direction road))
         (start-points (extract-road-connect-points start-node 
                                                    start-dir 
                                                    node-displays))
         (end-points    (extract-road-connect-points end-node 
                                                     end-dir 
                                                     node-displays))
         (mid-point  (if (display-info road) 
                         (list (add-points (display-info road) offset))
                         '()))
         (path (append start-points mid-point (reverse end-points))))
    (really-make-road-displayer 
     :displayed-object road
     :truck truck
     :highlighted NIL
     :path-points path
     :offset-path-points (rd-make-offset-path-points path))))

;;; This should return a rectangle big 
;;;  enough to cover the entire region spanned by this display.
;;; Should be used only to compute how big the map displayer is.
;;; If it's ever used for erasure, things will get ugly!

(defmethod display-region ((self road-displayer))
  (let* ((path-points (road-displayer-offset-path-points self))
         (x-list (mapcar #'point-x path-points))
         (y-list (mapcar #'point-y path-points))
         (min-x  (apply #'min x-list))
         (max-x  (apply #'max x-list))
         (min-y  (apply #'min y-list))
         (max-y  (apply #'max y-list)))
    (make-rectangle :origin (make-point :x min-x :y min-y)
                    :other-point (make-point :x max-x :y max-y))))

(defmethod displayed-object ((self road-displayer))
  (road-displayer-displayed-object self))

(defmethod draw ((self road-displayer))
  (do ((points (road-displayer-path-points self) (cdr points)))
      ((null (cdr points)) (values))
    (disp.draw-line (first points) (second points))))

(defmethod update-display ((self road-displayer) object)
  (when (eq object (road-displayer-displayed-object self))
    (road-displayer-update self)))

(defun road-displayer-update (self)
  (when (and (not (road-displayer-highlighted self))
             (truck-in-road? (road-displayer-truck self) 
			     (road-displayer-displayed-object self)))
    (road-displayer-highlight self))
  (when (and (road-displayer-highlighted self)
             (not (truck-in-road? (road-displayer-truck self) 
				  (road-displayer-displayed-object self))))
    (road-displayer-unhighlight self)))

(defun truck-in-road? (truck road)
  (or (member truck (holdings road))
      (some #'(lambda (node) (member truck (holdings node))) (holdings road))))
  
(defun road-displayer-highlight (self)
  (do ((points (road-displayer-offset-path-points self) (cdr points)))
      ((null (cdr points)) (values))
    (disp.draw-line (first points) (second points)))
  (setf (road-displayer-highlighted self) t))

(defun road-displayer-unhighlight (self)
  (disp.with-erasure (road-displayer-highlight self))
  (setf (road-displayer-highlighted self) nil))

;;;************************************

(defvar *road-connect-border* 3)

(defun extract-road-connect-points (node dir node-displays)
  (let* ((node-disp (find-if #'(lambda (nd) 
                                 (eq node (node-displayer-displayed-object nd))) 
                             node-displays))
         (frame (node-displayer-frame node-disp))
         (origin (rectangle-origin frame))
         (width  (rectangle-width frame))
         (height (rectangle-height frame))
         (surface-point 
          (add-point origin 
                     :x (case dir
                          ((nw w sw) 0)
                          ((n s) (truncate width 2))
                          ((ne e se) width))
                     :y (case dir
                          ((nw n ne) 0)
                          ((w e)     (truncate height 2))
                          ((sw s se) height))))
         (other-point
          (add-point surface-point
                     :x (case dir
                          ((n s)     0)
                          ((e ne se)  *road-connect-border*)
                          ((w nw sw) (- *road-connect-border*)))
                     :y (case dir
                          ((e w)     0)
                          ((n ne nw) (- *road-connect-border*))
                          ((s se sw) *road-connect-border*)))))
    (list surface-point other-point)))

(defvar *road-offset-border* 2)

(defun rd-make-offset-path-points (point-list)
  (if (null (cdr point-list)) 
      '()
      (let* ((p1 (first point-list))
             (p2 (second point-list))
             (sideways? (> (abs (- (point-y p2) (point-y p1)))
                           (abs (- (point-x p2) (point-x p1))))))
        (if sideways?
            (cons (add-point p1 :x *road-offset-border*)
                  (cons (add-point p2 :x *road-offset-border*)
                        (rd-make-offset-path-points (cdr point-list))))
            (cons (add-point p1 :y (- *road-offset-border*))
                  (cons (add-point  p2 :y (- *road-offset-border*))
                        (rd-make-offset-path-points (cdr point-list))))))))




