
(defvar *truckworld-display* nil)

(defmacro with-active-display (display &rest body)
  `(let ( (*truckworld-display*  ,display) )
     (grafp.with-active-display *truckworld-display*
        ,@body)))

;******************************************************************

(defun disp.tiny-font ()   (grafp.font *truckworld-display* :tiny))
(defun disp.small-font ()  (grafp.font *truckworld-display* :small))
(defun disp.medium-font () (grafp.font *truckworld-display* :medium))
(defun disp.large-font ()  (grafp.font *truckworld-display* :large))
(defun disp.huge-font ()   (grafp.font *truckworld-display* :huge))
(defun disp.standard-font () (disp.medium-font))

(defun disp.color (red green blue) 
  (grafp.color *truckworld-display* red green blue))

(defparameter disp.black-color   (disp.color 0.0 0.0 0.0))
(defparameter disp.white-color   (disp.color 1.0 1.0 1.0))
(defparameter disp.red-color     (disp.color 1.0 0.0 0.0))
(defparameter disp.green-color   (disp.color 0.0 1.0 0.0))
(defparameter disp.blue-color    (disp.color 0.0 0.0 1.0))
(defparameter disp.cyan-color    (disp.color 1.0 1.0 0.0))
(defparameter disp.yellow-color  (disp.color 0.0 1.0 1.0))
(defparameter disp.magenta-color (disp.color 1.0 0.0 1.0))

(defun disp.init (displayer rec display-string
                            &key background foreground
                                 top-shadow bottom-shadow)
  (grafp.open-window displayer
                     (point-x (rectangle-origin rec))
                     (point-y (rectangle-origin rec))
                     (rectangle-width rec)
                     (rectangle-height rec)
                     display-string 
                     :foreground foreground 
                     :background background
                     :top-shadow top-shadow 
                     :bottom-shadow bottom-shadow))

(defun disp.terminate (disp)
  (grafp.terminate disp))

; ----------------------------------------------------------------------
; Drawing functions
; -----------------

(defun disp.draw-line (point1 point2)
  (grafp.draw-line *truckworld-display* 
                   (point-x point1)
                   (point-y point1)
                   (point-x point2)
                   (point-y point2)))

(defun disp.draw-circle (point r)
  (grafp.draw-circle *truckworld-display* (point-x point) (point-y point) r nil))

(defun disp.fill-circle (point r)
  (grafp.draw-circle *truckworld-display* (point-x point) (point-y point) r T))

(defvar triangle-list (list 0 0 0 0 0 0))

(defun disp.draw-triangle (p1 p2 p3)
  (setf (first  triangle-list) (point-x p1))
  (setf (second triangle-list) (point-y p1))
  (setf (third  triangle-list) (point-x p2))
  (setf (fourth triangle-list) (point-y p2))
  (setf (fifth  triangle-list) (point-x p3))
  (setf (sixth  triangle-list) (point-y p3))
  (grafp.draw-polygon *truckworld-display* triangle-list nil))

(defun disp.fill-triangle (p1 p2 p3)
  (setf (first  triangle-list) (point-x p1))
  (setf (second triangle-list) (point-y p1))
  (setf (third  triangle-list) (point-x p2))
  (setf (fourth triangle-list) (point-y p2))
  (setf (fifth  triangle-list) (point-x p3))
  (setf (sixth  triangle-list) (point-y p3))
  (grafp.draw-polygon *truckworld-display* triangle-list T))

(defun disp.draw-polygon (points)
  (grafp.draw-polygon *truckworld-display* points nil))

(defun disp.fill-polygon (points)
  (grafp.draw-polygon *truckworld-display* points T))

; -----------------------------------------------------------------------
; Rectangles
; ----------

(defun disp.draw-rectangle (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.draw-rectangle *truckworld-display* 
                          (point-x p1) (point-y p1)
                          (point-x p2) (point-y p2) nil)))

(defun disp.fill-rectangle (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.draw-rectangle *truckworld-display* 
                          (point-x p1) (point-y p1)
                          (point-x p1) (point-y p2) T)))

(defun disp.clear-rectangle (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.clear-rectangle *truckworld-display* 
                           (point-x p1) (point-y p1) 
                           (point-x p2) (point-y p2)))) 

(defun disp.draw-rectangle-interior (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.draw-rectangle-interior *truckworld-display*
                                   (point-x p1) (point-y p1)
                                   (point-x p2) (point-y p2) NIL)))

(defun disp.fill-rectangle-interior (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.draw-rectangle-interior *truckworld-display* 
                                   (point-x p1) (point-y p1)
                                   (point-x p2) (point-y p2) T)))

(defun disp.clear-rectangle-interior (rec)
  (let ((p1 (rectangle-origin rec))
        (p2 (rectangle-other-point rec)))
    (grafp.clear-rectangle-interior *truckworld-display* 
                                    (point-x p1) (point-y p1)
                                    (point-x p2) (point-y p2))))

; -----------------------------------------------------------------------
; Text
; ----

(defun disp.text-extent (str)
  (grafp.text-extent *truckworld-display* str))

(defun disp.draw-text (str point &optional pos1 pos2)
  (let ((x (point-x point))
        (y (point-y point)))
    (multiple-value-bind (w h d) (grafp.text-extent *truckworld-display* str)
      (let ( (x (cond ((or (eq pos1 :left) (eq pos2 :left))
                     x)
                    ((or (eq pos1 :right) (eq pos2 :right))
                     (- x w))
                    (T
                     (- x (truncate w 2)))))
           (y (cond ((or (eq pos1 :top) (eq pos2 :top))
                     (+ y (- h d)))
                    ((or (eq pos1 :bottom) (eq pos2 :bottom))
                     (- y d))
                    (T
                     (+ y (- (truncate h 2) d))))) )
      (grafp.draw-text *truckworld-display* str x y)))))

;; ------------------------------------------
;; 3-D stuff
;; ---------

(defun disp.draw-3d-circle (up/down point r)
  (let ((x (point-x point))
        (y (point-y point)))
    (grafp.draw-3d-circle *truckworld-display* up/down x y r)))

(defun disp.draw-3d-rectangle (up/down rec)
  (let* ((p1 (rectangle-origin rec))
         (p2 (rectangle-other-point rec))
         (x1 (point-x p1))
         (y1 (point-y p1))
         (x2 (point-x p2))
         (y2 (point-y p2)))
    (grafp.draw-3d-rectangle *truckworld-display* up/down x1 y1 x2 y2)))

;; ------------------------------------------
;; X Bitmaps
;; ---------

(defvar *truckworld-default-bitmap-directory*
  (cond ((find-package "COMMON-LISP-USER")
         (if (boundp 'common-lisp-user::*truckworld-default-bitmap-directory*)
             common-lisp-user::*truckworld-default-bitmap-directory*
             nil))
        ((find-package "USER")
         (if (boundp 'user::*truckworld-default-bitmap-directory*)
             user::*truckworld-default-bitmap-directory*
             nil))
        (T nil)))

(defvar *disp.bitmap-list* '())

(defun disp.draw-bitmap-with-name (bitmap rec)
  (grafp.draw-bitmap *truckworld-display* 
                     bitmap
                     (+ (rectangle-min-x rec) 5)
                     (+ (rectangle-min-y rec) 5)
                     (- (rectangle-max-x rec) 5)
                     (- (rectangle-max-y rec) 5)))

(defun disp.fetch-bitmap (name)
  (let ((pair (assoc name *disp.bitmap-list* :test #'equal)))
    (cond
     (pair (cdr pair))
     ((probe-file name)
      (let ((bitmap (grafp.load-bitmap *truckworld-display* name)))
        (push (cons name bitmap) *disp.bitmap-list*)
        bitmap))
     (t NIL))))


;; ------------------------------------------
;; Macros
;; ------

(defmacro disp.with-font (font &rest body)
  (let ( (ft (gentemp)) )
    `(let ( (,ft (grafp.get-font *truckworld-display*)) )
       (unwind-protect
         (progn
           (grafp.set-font *truckworld-display* ,font)
           ,@body)
         (grafp.set-font *truckworld-display* ,ft)))))

(defmacro disp.with-clip-window (rec &rest body)
  (let ((x0 (gentemp)) (y0 (gentemp)) 
        (w  (gentemp)) (h  (gentemp)) )
    `(multiple-value-bind (,x0 ,y0 ,w ,h)
                          (grafp.get-drawing-region *truckworld-display*)
       (unwind-protect
           (progn
             (grafp.set-drawing-region *truckworld-display* 
                                       (rectangle-min-x ,rec)
                                       (rectangle-min-y ,rec)
                                       (rectangle-width ,rec)
                                       (rectangle-height ,rec))
             ,@body)
         (grafp.set-drawing-region *truckworld-display* ,x0 ,y0 ,w ,h)))))


(defmacro disp.with-erasure (&rest body)
  (let ( (erasure (gentemp)) )
    `(let ( (,erasure (grafp.get-erasure *truckworld-display*)) )
       (unwind-protect
         (progn
           (grafp.set-erasure *truckworld-display* T)
           ,@body)
         (grafp.set-erasure *truckworld-display* ,erasure)))))

(defmacro disp.with-invert (&rest body)
  (let ( (invert (gentemp)) )
    `(let ( (,invert (grafp.get-invert *truckworld-display*)) )
       (unwind-protect
         (progn
           (grafp.set-invert *truckworld-display* T)
           ,@body)
         (grafp.set-invert *truckworld-display* ,invert)))))

(defmacro disp.with-color (color &rest body)
  (let ( (old-color (gentemp)) )
    `(let ( (,old-color (grafp.get-foreground *truckworld-display*)) )
       (unwind-protect
         (progn
           (grafp.set-foreground *truckworld-display* ,color)
           ,@body)
         (grafp.set-foreground *truckworld-display* ,old-color)))))
