;;;*******************************************************************
;;;  Fuel Gauge Displayer
;;;************

(defvar *fuel-displayer-width* 40)
(defvar *fuel-displayer-height* 73)

(defstruct (fuel-displayer  (:print-function print-fuel-displayer)
                            (:constructor really-make-fuel-displayer))
  displayed-object
  displayed-level
  capacity 
  outer-frame
  inner-frame
  gripper-data)

(defun print-fuel-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer for ~a}" (fuel-displayer-displayed-object self)))

(defun make-fuel-displayer (the-tank point)
  (really-make-fuel-displayer
   :displayed-object the-tank
   :capacity (query the-tank 'capacity)
   :displayed-level 0
   :outer-frame (make-rectangle :origin point 
                                :width *fuel-displayer-width*
                                :height *fuel-displayer-height*)
   :inner-frame (make-rectangle :origin (add-point point
                                                   :x 20
                                                   :y 15)
                                :width 10
                                :height (- *fuel-displayer-height* 5))
   :gripper-data '()))

(defmethod displayed-object ((self fuel-displayer))
  (fuel-displayer-displayed-object self))

(defmethod draw ((self fuel-displayer))
  (fuel-displayer-draw self))

(defun fuel-displayer-draw (self)
  (fuel-displayer-draw-frame self)
  (fuel-displayer-update self))

(defun fuel-displayer-draw-frame (self)
  (let* ((frame (fuel-displayer-inner-frame self))
         (origin (rectangle-origin frame))
         (width  (rectangle-width frame))
         (tick-sep (/ (rectangle-height frame) 4)))
    (disp.draw-3d-rectangle :down  frame)
    (disp.with-font (disp.small-font)
      (disp.draw-text "F"   
                      (add-point origin :x -5) :center :right)
      (disp.draw-text "3/4" 
                      (add-point origin :x -5 :y tick-sep) :center :right)
      (disp.draw-text "1/2" 
                      (add-point origin :x -5 :y (* 2 tick-sep)) :center :right)
      (disp.draw-text "1/4" 
                      (add-point origin :x -5 :y (* 3 tick-sep)) :center :right)
      (disp.draw-text "E"   
                      (add-point origin :x -5 :y (* 4 tick-sep)) :center :right))
    (dotimes (i 5)
      (let ((y-offset (* i tick-sep)))
        (disp.draw-line (add-point origin :y y-offset)
                        (add-point origin :x -3 :y y-offset))
        (disp.draw-line (add-point origin :x width :y y-offset)
                        (add-point origin :x width :y y-offset))))))

(defmethod update-display ((self fuel-displayer) object)
  (when (eq object (fuel-displayer-displayed-object self))
    (fuel-displayer-update self)))

(defun fuel-displayer-update (self)
  (let ((new-level (query (fuel-displayer-displayed-object self) 
                          'amount-held)))
    (when (not (= new-level (fuel-displayer-displayed-level self)))
      (fuel-displayer-erase-bar self)
      (setf (fuel-displayer-displayed-level self) new-level)
      (fuel-displayer-draw-bar self))))

(defun fuel-displayer-erase-bar (self)
  (disp.with-erasure 
   (fuel-displayer-draw-bar self)))

(defun fuel-displayer-draw-bar (self)
  (let* ((frame (fuel-displayer-inner-frame self))
         (fill-height (* (rectangle-height frame)
                         (- 1 (/ (fuel-displayer-displayed-level self)
                                 (fuel-displayer-capacity self))))))
         (disp.fill-rectangle-interior 
          (offset-rectangle frame :y fill-height :height (- fill-height)))))

;;;*****************************************************************
;;; For displaying the gripper.  Just like the bays do it, 
;;; except we don't let it go inside or anything...
;;;  Gripper data is of the form (ARM-DISPLAYER ORIENTATION)

(defmethod display-gripper ((self fuel-displayer) 
                            arm-displayer 
                            container 
                            position)
  (declare (ignore position));;  Better be 0!
  (when (eq container (fuel-displayer-displayed-object self))
    (let ((current-gripper-displays (fuel-displayer-gripper-data self))
          (orientation NIL))
      (cond
       ((= 0 (length current-gripper-displays))
        (setf orientation :LEFT))
       ((= 1 (length current-gripper-displays))
        (let ((current-orientation (third (car current-gripper-displays))))
          (setf orientation 
                (if (eq current-orientation :BOTTOM)
                    :LEFT
                  :BOTTOM))))
       (t (displayer-warning "Too many grippers displayed at ~a" self)))
      (let* ((coord (fuel-displayer-gripper-coords self orientation)))
        (arm-displayer-draw-gripper arm-displayer coord orientation :OUTSIDE)
        (push (list arm-displayer orientation)
              (fuel-displayer-gripper-data self))
        t))))

(defmethod undisplay-gripper ((self fuel-displayer) arm-displayer)
  (let ((gripper-display (find-if #'(lambda (gd) (eq arm-displayer (first gd)))
                                  (fuel-displayer-gripper-data self))))
    (when gripper-display
      (let* ((gripper-coord 
              (fuel-displayer-gripper-coords self (second gripper-display))))
        (arm-displayer-erase-gripper arm-displayer 
                                     gripper-coord
                                     (second gripper-display)
                                     :OUTSIDE)
        (setf (fuel-displayer-gripper-data self)
          (remove gripper-display (fuel-displayer-gripper-data self)))))))

;;;  Upper left corner where the gripper should display itself
;;;  Possible orientations are :LEFT and :BOTTOM

(defun fuel-displayer-gripper-coords (self orientation)
  (let ((frame (fuel-displayer-outer-frame self)))
    (case orientation
      ((:LEFT) (add-point (rectangle-origin frame) :x -10 :y 25))
      ((:BOTTOM) (add-point (rectangle-origin frame) 
                            :y (+ (rectangle-height frame) 10)))
      (OTHERWISE (displayer-warning 
                  "Weird arguments to fuel-displayer-gripper-coords ~a ~a"
                  self orientation)))))

;;;*******************************************************************
;;;  Heading Gauge Displayer
;;;***************

(defvar *heading-displayer-width* 50)
(defvar *heading-displayer-height* 50)
(defvar *heading-displayer-y-offset* 20)

(defstruct (heading-displayer (:print-function print-heading-displayer)
                              (:constructor really-make-heading-displayer))
  displayed-object
  displayed-heading
  frame
  center-point
  radius
  angle)

(defun print-heading-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer for ~a}" (heading-displayer-displayed-object self)))

(defun make-heading-displayer (d point)
  (let* ((frame (make-rectangle :origin (add-point point 
                                                   :y *heading-displayer-y-offset*)
                                :width *heading-displayer-width*
                                :height *heading-displayer-height*))
         (gauge (really-make-heading-displayer
                 :displayed-object d
                 :displayed-heading nil
                 :frame frame
                 :center-point (rectangle-center frame)
                 :radius (- (truncate (min (rectangle-width frame)
                                           (rectangle-height frame)) 
                                      2) 
                            10)
                 :angle nil)))
    gauge))

(defmethod displayed-object ((self heading-displayer))
  (heading-displayer-displayed-object self))

(defmethod draw ((self heading-displayer))
  (heading-displayer-draw-frame self)
  (heading-displayer-update self))

(defun heading-displayer-draw-frame (self)
  (let* ((origin (heading-displayer-center-point self))
         (radius (heading-displayer-radius self))
         (offset (+ radius 6)))
    (disp.with-font (disp.small-font)
      (disp.draw-text "N" (add-point origin :y (- offset)) :center)
      (disp.draw-text "S" (add-point origin :y offset) :center)
      (disp.draw-text "W" (add-point origin :x (- offset)) :center)
      (disp.draw-text "E" (add-point origin :x offset) :center))
    (disp.draw-3d-circle :down origin (+ radius 1))))

(defmethod update-display ((self heading-displayer) object)
  (when (eq object (heading-displayer-displayed-object self))
    (heading-displayer-update self)))

(defun heading-displayer-update (self)
  (let ((new-heading (query (heading-displayer-displayed-object self) 'value)))
    (when (not (eq new-heading (heading-displayer-displayed-heading self)))
      (heading-displayer-redraw-arrow self))))

(defun heading-displayer-redraw-arrow (self)
  (let* ((new-heading (query (heading-displayer-displayed-object self) 'value))
         (angle (case new-heading
                   ((N)  (* -0.5 pi))  ((S)  (* 0.5 pi))
                   ((E)  0)            ((W)  pi)
                   ((NE) (* -0.25 pi)) ((NW) (* -0.75 pi))
                   ((SE)  (* 0.25 pi)) ((SW) (* 0.75 pi))
                   (otherwise nil))))
    (when (not angle)
      (displayer-warning "Can't find angle for heading displayer ~a" self))
    (when (heading-displayer-angle self)
      (disp.with-erasure 
       (draw-compass-arrow (heading-displayer-center-point self)
                           (heading-displayer-radius self)
                           (heading-displayer-angle self))))
    (setf (heading-displayer-angle self) angle)
    (setf (heading-displayer-displayed-heading self) new-heading)
    (draw-compass-arrow (heading-displayer-center-point self)
                        (heading-displayer-radius self)
                        angle)))

(defun draw-compass-arrow (origin radius angle)
  (draw-arrow origin radius angle)
  (disp.fill-circle origin 2))

(defun draw-arrow (origin r angle)
  (let* ((x (point-x origin))
         (y (point-y origin))
         (tip-x (+ x (truncate (* r (cos angle)))))
         (tip-y (+ y (truncate (* r (sin angle)))))
         (x1    (+ x (truncate (* 0.8 r (cos (- angle 0.5))))))
         (y1    (+ y (truncate (* 0.8 r (sin (- angle 0.5))))))
         (x2    (+ x (truncate (* 0.8 r (cos (+ angle 0.5))))))
         (y2    (+ y (truncate (* 0.8 r (sin (+ angle 0.5)))))))
    (disp.draw-line origin (make-point :x tip-x :y tip-y))
    (disp.fill-triangle (make-point :x x1 :y y1)
                        (make-point :x tip-x :y tip-y)
                        (make-point :x x2 :y y2))
    (values)))


;;;*******************************************************************
;;;  Info Gauges (Status and Time)
;;;*******************************

(defvar *info-displayer-width* 63)
(defvar *info-displayer-height* 36)

(defstruct (info-displayer  (:print-function print-info-displayer)
                            (:constructor really-make-info-displayer))
  displayed-object
  displayed-value
  inner-frame
  outer-frame
  center-point
  name-string 
  value-string)

(defun print-info-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer for ~a}" (info-displayer-displayed-object self)))

(defun make-info-displayer (obj name point)
  (let* ((outer-frame (make-rectangle :origin point 
                                      :width *info-displayer-width*
                                      :height *info-displayer-height*))
         (inner-frame (offset-rectangle outer-frame :y 16 :height -16))
         (gauge (really-make-info-displayer
                :displayed-object obj
                :outer-frame outer-frame
                :inner-frame inner-frame
                :center-point (rectangle-center inner-frame)
                :name-string (->string name)
                :displayed-value nil
                :value-string "")))
    gauge))

(defmethod display-region ((self info-displayer))
  (info-displayer-outer-frame self))

(defmethod displayed-object ((self info-displayer))
  (info-displayer-displayed-object self))

(defmethod draw ((self info-displayer))
  (info-displayer-draw-frame self)
  (info-displayer-update self))

(defun info-displayer-draw-frame (self)
  (let ((frame (info-displayer-inner-frame self)))
    (disp.draw-3d-rectangle :down frame)
    (disp.with-font (disp.medium-font)
      (disp.draw-text (info-displayer-name-string self) 
                      (add-point (rectangle-origin frame) :y -2)
                      :bottom 
                      :left))))


(defmethod update-display ((self info-displayer) object)
  (when (eq object (info-displayer-displayed-object self))
    (info-displayer-update self)))

(defun info-displayer-update (self)
  (let ((new-value (query (info-displayer-displayed-object self) 'value)))
    (when (not (equal new-value (info-displayer-displayed-value self)))
      (let* ((frame (info-displayer-inner-frame self))
             (width (rectangle-width frame))
             (height (rectangle-height frame)))
        (setf (info-displayer-displayed-value self) new-value)
        (setf (info-displayer-value-string self) (->string new-value))
        (disp.clear-rectangle-interior frame)
        (disp.with-clip-window (info-displayer-inner-frame self)
          (disp.with-font (disp.small-font)
            (disp.draw-text (info-displayer-value-string self) 
                            (make-point :x (truncate width 2) :y (truncate height 2))
                            :center)))))))
(defun ->string (thing)
  (cond
   ((symbolp thing) (symbol-name thing))
   ((stringp thing) thing)
   (t (format nil "~a" thing))))


;;;*******************************************************************
;;;  Speed Gauge Displayer
;;;*************

(defvar *speed-displayer-width* 50)
(defvar *speed-displayer-height* 81)

(defstruct (speed-displayer (:print-function print-speed-displayer)
                            (:constructor really-make-speed-displayer))
  displayed-object
  displayed-value
  outer-frame 
  inner-frame
  tick-sep 
  offset
  actual-height)

(defun print-speed-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer for ~a}" (speed-displayer-displayed-object self)))

(defun make-speed-displayer (obj point)
  (let* ((frame (make-rectangle :origin point 
                                :width *speed-displayer-width*
                                :height *speed-displayer-height*))
         (gauge (really-make-speed-displayer 
                 :displayed-object obj
                 :displayed-value nil
                 :outer-frame  frame
                 :inner-frame (offset-rectangle frame 
                                                :x 35 
                                                :y 10 
                                                :width -35 
                                                :height -10)
                :offset 5)))
    (setf (speed-displayer-tick-sep gauge)
          (truncate (rectangle-height frame) 4))
    (setf (speed-displayer-actual-height gauge)
          (truncate (- (point-y (rectangle-other-point frame))
                       (* 0.5 (speed-displayer-tick-sep gauge)))))
    gauge))

(defmethod displayed-object ((self speed-displayer))
  (speed-displayer-displayed-object self))

(defmethod draw ((self speed-displayer))
  (speed-displayer-draw-frame self)
  (speed-displayer-update self))

(defun speed-displayer-draw-frame (self)
  (let* ((tick-sep (speed-displayer-tick-sep self))
         (offset   (speed-displayer-offset   self))
         (x1  (point-x (rectangle-origin (speed-displayer-inner-frame self))))
         (y2  (point-y (rectangle-other-point (speed-displayer-inner-frame self))))
         (tick-3 (truncate (- y2 (* 3.5 tick-sep))))
         (tick-2 (truncate (- y2 (* 2.5 tick-sep))))
         (tick-1 (truncate (- y2 (* 1.5 tick-sep))))
         (tick-0 (truncate (- y2 (* 0.5 tick-sep)))))
    (disp.draw-3d-rectangle :down (speed-displayer-inner-frame self))
    (disp.with-font (disp.small-font)
      (disp.draw-text "Fast" 
                      (make-point :x (- x1 offset) :y tick-3) 
                      :center :right)
      (disp.draw-text "Med"  
                      (make-point :x (- x1 offset) :y tick-2) 
                      :center :right)
      (disp.draw-text "Slow" 
                      (make-point :x (- x1 offset) :y tick-1)
                      :center :right)
      (disp.draw-text "Stop" 
                      (make-point :x (- x1 offset) :y tick-0)
                      :center :right))
    (disp.draw-line (make-point :x x1 :y tick-3) 
                    (make-point :x (- x1 3) :y tick-3))
    (disp.draw-line (make-point :x x1 :y tick-2)
                    (make-point :x (- x1 3) :y tick-2))
    (disp.draw-line (make-point :x x1 :y tick-1)
                    (make-point :x (- x1 3) :y tick-1))
    (disp.draw-line (make-point :x x1 :y tick-0)
                    (make-point :x (- x1 3) :y tick-0))))

(defmethod update-display ((self speed-displayer) object)
  (when (eq object (speed-displayer-displayed-object self))
    (speed-displayer-update self)))

(defun speed-displayer-update (self)
  (let ((new-value (query (speed-displayer-displayed-object self) 'value)))
    (when (not (eq new-value (speed-displayer-displayed-value self)))
      (let* ((frame (speed-displayer-inner-frame self))
             (new-level (case new-value
                          ((FAST) 3.5) ((MEDIUM MED) 2.5)
                          ((SLOW) 1.5) ((STOP STOPPED ZERO) 0.5)
                          (otherwise nil)))
             (right-x   (point-x (rectangle-other-point frame)))
             (bottom-y  (point-y (rectangle-other-point frame))))
        (cond 
         ((not new-level) (displayer-warning "Weird speed displayer update ~a"
                                             self))
         (T (let ((height (truncate (- bottom-y 
                                       (* new-level 
                                          (speed-displayer-tick-sep self))))))
              (setf (speed-displayer-actual-height self) height)
              (setf (speed-displayer-displayed-value self) new-value)
              (disp.clear-rectangle-interior (speed-displayer-inner-frame self))
              (draw-arrow (make-point :x (- right-x 1) :y height) 
                          (rectangle-width frame)
                          pi))))))))

