;;;************************************************************
;;;  Bay displayers.  For the cargo bays, the arm bays, 
;;;  tire and weapon bays, and the current location.
;;;  These things know how to draw and update themselves, 
;;;  and to return display information for an object (suitable 
;;;  for use by an ARM-MOVE command.

(defvar *small-icon-width*  35)
(defvar *small-icon-height* 35)

(defvar *gripper-display-depth* 5)

(defvar *bay-object-width* *small-icon-width*
  "Width of a single object to be displayed in the bay.")
(defvar *bay-object-height* (+ *small-icon-height* 15)
  "Height of a single object to be displayed in the bay.")

;;; These numbers 
(defvar *bay-object-horizontal-separation*  (+ 5 *gripper-display-depth*)
  "Amount of width between objects and between the borders and the adjoining 
  objects.")
(defvar *bay-object-vertical-separation*    (+ 5 *gripper-display-depth*)
  "Amount of height between objects and between the borders and the adjoining 
   objects on the top and bottom")

(defvar *bay-header-width* *bay-object-horizontal-separation*
  "The amount of horizontal space between the left of the display
   region and the topmost displayed objects.")
(defvar *bay-header-height* 15
  "The amount of vertical space between the top of the display 
   region and the leftmost displayed objects.")

(defvar *basic-horizontal-separation* 
    (+ *bay-object-width* *bay-object-horizontal-separation*)
  "The amount of horizontal space between the sides of adjoining objects.")
(defvar *basic-vertical-separation* 
        (+ *bay-object-height* *bay-object-vertical-separation*)
  "The amount of horizontal space between the top/bottom of adjoining objects.")

;;;********************************************************************
;;; Fields in a bay-displayer
;;;
;;;  Displayed-object -- the simulator object I store
;;;  Displayed-name   -- the string I print at the top
;;;  Display-region   -- a rectangle representing my total extent
;;;  Inner-frame      -- rectangle that surrounds the objects.  I draw a box.
;;;  Num-rows/Num-columns -- number of rows and columns of icons I display.
;;;  Num-positions -- the number of positions devoted to displaying
;;;    simulator objects.
;;;  Displayed-objects -- 
;;;     a list of length num-positions at one-to-one correspondence 
;;;     with the display positions.  These are the objects I am 
;;;     currently displaying.   NIL means there's nothing there.
;;;  Display-positions -- these are the (x y) positions at which 
;;;     I should display my objects.  They are in fill order (i.e.
;;;     I should display at position 0, then 1, then....  
;;;  Gripper-data -- a list of 
;;;      (arm position orientation inside/outside)
;;;     where arm is an arm object, position is the position in this bay, 
;;;     orientation is either :LEFT or :BELOW, and inside/outside is 
;;;     either :INSIDE or :OUTSIDE.  Indicates that 
;;;     the arm's gripper is currently being displayed at this position, 
;;;     to the :LEFT of (or below) the current icon, and :INSIDE (or :OUTSIDE)
;;;     of the object at this position. 

(defstruct (bay-displayer (:print-function print-bay-displayer)
                          (:constructor really-make-bay-displayer))
  displayed-object
  displayed-name
  display-border-p
  outer-frame
  inner-frame
  num-rows
  num-columns
  num-positions
  displayed-objects
  display-positions
  gripper-data)

(defun print-bay-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Bay displayer for ~a}" (bay-displayer-displayed-object self)))

(defun get-bay-displayer-display-position (displayer position)
  (let ((coords (bay-displayer-display-positions displayer)))
    (cond
     ((> 0 position)
      (first coords))
     ((>= position (length coords))
      (first (last coords)))
     (t
      (nth position coords)))))
     
;;;*******************************************************************
;;;  Instance creation.   Basic task is to create my display positions.
;;;
;;;  Displayed-object    -- the simulator object to be displayed
;;;  Displayed-name      -- a string to display at the top of the bay.
;;;    If this is NIL, we'll make one up by asking the displayed-object.
;;;  Display-border-p    --  should we display a border and the bay's 
;;;    name?  (T for all bays except current location)
;;;  X, Y               -- upper-left corner of the display rectangle
;;;  Num-rows, Num-cols -- number of rows and columns in the matrix of 
;;;                        displayed objects.  This determines the size 
;;;                        of the display area too.
;;;  First-corner       -- a list of two symbols, one of 
;;;    '(TOP LEFT) '(TOP RIGHT) '(BOTTOM LEFT) '(BOTTOM RIGHT) which 
;;;    says where the first object should be displayed.
;;;  First-direction    -- either HORIZONTAL or VERTICAL saying whether
;;;    the next object should be displayed left/right or above/below 
;;;    the previous.
;;;

(defun make-bay-displayer (displayed-object  
                           displayed-name                           
                           origin
                           num-rows num-cols
                           first-corner       
                           first-direction
                           &key (display-border T))
  (let* ((displayer (really-make-bay-displayer))
         (outer-frame (make-rectangle 
                       :origin origin
                       :width (+ *bay-header-width*  
                                 (* num-cols *basic-horizontal-separation*))
                       :height (+ *bay-header-height* 
                                  (* num-rows *basic-vertical-separation*)))))
    (setf (bay-displayer-displayed-object displayer) displayed-object)
    (setf (bay-displayer-displayed-name displayer)
      (cond (displayed-name displayed-name)
            (t (string (id displayed-object)))))
    (setf (bay-displayer-display-border-p displayer) display-border)
    (setf (bay-displayer-outer-frame displayer) outer-frame)
    (cond
     (display-border 
      (setf (bay-displayer-inner-frame displayer)
        (make-rectangle :origin (add-point origin 
                                           :x *bay-header-width* 
                                           :y *bay-header-height*)
                        :other-point (rectangle-other-point outer-frame))))
      (t (setf (bay-displayer-inner-frame displayer) outer-frame)))
    (setf (bay-displayer-num-rows displayer) num-rows)
    (setf (bay-displayer-num-columns displayer) num-cols)
    (let ((num-display-positions (* num-rows num-cols))
          (num-bay-positions (length (query displayed-object 'contents))))
      (when (> num-bay-positions num-display-positions)
        (error "Not enough space to display bay ~a" displayed-object))
      (setf (bay-displayer-display-positions displayer)
        (compute-display-positions 
         (add-point (rectangle-origin outer-frame) 
                    :x *bay-header-width*
                    :y *bay-header-height*)
         num-rows 
         num-cols 
         first-corner 
         first-direction))
      (setf (bay-displayer-displayed-objects displayer)
        (make-list num-display-positions :initial-element NIL))
      (setf (bay-displayer-num-positions displayer) num-display-positions)
      (setf (bay-displayer-gripper-data displayer) '()))
    displayer))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; To compute display positions:
;;  Loop through all rows and columns in the order determined by 
;;; first-corner and first-direction.  Each time compute the correct
;;; (x,y) offset for this row/column pair and store them in successive
;;; elements of the returned vector.

(defun compute-display-positions (origin-point
                                  num-rows num-cols 
                                  first-corner first-direction)
  (let* ((x (point-x origin-point))
         (y (point-y origin-point))
         (num-positions (* num-rows num-cols))
         (output-list (make-list num-positions))
         (horizontal-fill (if (eq (second first-corner) 'right) 'left 'right))
         (vertical-fill   (if (eq (first first-corner) 'bottom) 'up 'down)))
    (do ((i 0 (+ i 1))
         (row (if (eq (first first-corner) 'top) 0 (- num-rows 1)))
         (col (if (eq (second first-corner) 'left) 0 ( - num-cols 1))))
        ((>= i num-positions) (values))
      ;;; Easy to compute the correct (x,y) pair given a row and column
      (setf (nth i output-list)
            (make-point :x (+ x (* col *basic-horizontal-separation*) 5)
                        :y (+ y (* row *basic-vertical-separation*) 5)))
      ;;; Now figure out what the next row,column should be based 
      ;;; on the fill directions.
      (cond
       ((and (eq first-direction 'vertical) (eq vertical-fill 'down))
        (setf row (+ row 1))
        (when (>= row num-rows)
          (setf row 0)
          (setf col (if (eq horizontal-fill 'left) (- col 1) (+ col 1)))))
       ((and (eq first-direction 'vertical) (eq vertical-fill 'up))
        (setf row (- row 1))
        (when (< row 0)
          (setf row (- num-rows 1))
          (setf col (if (eq horizontal-fill 'left) (- col 1) (+ col 1)))))
       ((and (eq first-direction 'horizontal) (eq horizontal-fill 'right))
        (setf col (+ col 1))
        (when (>= col num-cols)
          (setf col 0)
          (setf row (if (eq vertical-fill 'down) (+ row 1) (- row 1)))))
       ((and (eq first-direction 'horizontal) (eq horizontal-fill 'left))
        (setf col (- col 1))
        (when (< col 0)
          (setf col (- num-cols 1))
          (setf row (if (eq vertical-fill 'down) (+ row 1) (- row 1)))))))
    output-list))

;;;*************************************************************************
;;;  What the displayer knows how to do:
;;;     return displayed object
;;;     draw/erase/redraw itself
;;;     update itself
;;;     update gripper information about an arm

;;; Drawing-like operations.  This assumes that there's nothing 
;;; currently on the display, so draw the frame and all the objects.

(defmethod displayed-object ((self bay-displayer))
  (bay-displayer-displayed-object self))

(defmethod draw ((self bay-displayer))
  (bay-displayer-draw self))

(defmethod display-region ((self bay-displayer))
  (bay-displayer-outer-frame self))
  
(defun bay-displayer-draw (self)
  (bay-displayer-draw-border self)
  (dotimes (i (bay-displayer-num-positions self))
    (setf (nth i (bay-displayer-displayed-objects self)) nil))
  (bay-displayer-update-self self))
  
(defun bay-displayer-draw-border (self)
  (when (bay-displayer-display-border-p self)
    (disp.with-font (disp.medium-font)
      (disp.draw-text (bay-displayer-displayed-name self)
                      (add-point (rectangle-origin (bay-displayer-outer-frame self))
                                 :x 1 :y 10)
                      :bottom :left))
    (disp.draw-3d-rectangle :down (bay-displayer-inner-frame self))))

(defmethod redraw ((self bay-displayer))
  (bay-displayer-redraw self))

;;;****************************************************************
;;;  Erase

(defmethod erase ((self bay-displayer))
  (disp.clear-rectangle (bay-displayer-outer-frame self))
  (let ((displayed (bay-displayer-displayed-objects self)))
    (dotimes (i (length displayed))
      (setf (nth i displayed) NIL))
    (setf (bay-displayer-gripper-data self) '())))

;;;****************************************************************
;;;  Redraw.

(defun bay-displayer-redraw (self)
  (let ((displayed (bay-displayer-displayed-objects self))
        (positions (bay-displayer-display-positions self)))
    (dotimes (i (length positions))
      (let* ((point (nth i positions))
             (rect (make-rectangle :origin point 
                                   :width *small-icon-width*
                                   :height *small-icon-height*)))
        (when (nth i displayed)
          (disp.clear-rectangle rect)
          (draw-object-as-icon (nth i displayed) rect)
          (bay-displayer-redraw-grippers self))))))

;;;*****************************************************************
;;;  Update.  I have no subdisplayers, so I respond only when 
;;;  I'm the object to be updated.   My display is OK as long as 
;;;  every object I display is still in the container I'm displaying, 
;;;  and every object i'm displaying corresponds to the object 
;;;  in the container at the corresponding position.
;;;

(defmethod update-display ((self bay-displayer) object)
  (cond
   ((eq object (bay-displayer-displayed-object self))
    (bay-displayer-update-self self))
   ((member object (bay-displayer-displayed-objects self))
    (bay-displayer-update-member self object))
   (T NIL)))

(defun bay-displayer-update-self (self)
  (let* ((displayed (bay-displayer-displayed-objects self))
         (actual (query (bay-displayer-displayed-object self) 'contents))
         (n         (min (length displayed) (length actual))))
    (dotimes (i n)
      (when (not (eq (nth i displayed) (nth i actual)))
        (let* ((point (get-bay-displayer-display-position self i))
               (rect (make-rectangle :origin point
                                     :width *bay-object-width*
                                     :height *bay-object-height*)))
          (disp.clear-rectangle rect)
          (when (nth i actual)
            (draw-object-as-icon (nth i actual) rect))
          (setf (nth i displayed) (nth i actual)))))))

;;;(defun bay-displayer-update-self (self)
;;;  (format t "Updating ~a~%" self)
;;;  (let* ((displayed (bay-displayer-displayed-objects self))
;;;         (actual (query (bay-displayer-displayed-object self) 'contents))
;;;         (n         (min (length displayed) (length actual))))
;;;    (format t "My 0 is ~a displayed, ~a actual" 
;;;            (nth 0 displayed) (nth 0 actual))
;;;    (dotimes (i n)
;;;      (when (not (eq (nth i displayed) (nth i actual)))
;;;        (format t "Found mismatch at position ~a, displayed ~a, actual ~a~%" 
;;;                i (nth i displayed) (nth i actual))
;;;        (let* ((point (nth i (bay-displayer-display-positions self)))
;;;               (rect (make-rectangle :origin point
;;;                                     :width *bay-object-width*
;;;                                     :height *bay-object-height*)))
;;;          (disp.clear-rectangle rect)
;;;          (when (nth i actual)
;;;            (format t "Drawing the object~%")
;;;            (draw-object-as-icon (nth i actual) rect))
;;;          (setf (nth i displayed) (nth i actual))))))
;;;  (format t "Done.~%"))

(defun bay-displayer-update-member (self object)
  (let ((p (position object (bay-displayer-displayed-objects self))))
    (cond 
     ((not p) 
      (displayer-warning "Weird update failure on contained object ~a ~a~%"
                         self object))
     (t  (let* ((point (get-bay-displayer-display-position self p))
                (rect (make-rectangle :origin point
                                      :width *bay-object-width*
                                      :height *bay-object-height*)))
           (disp.clear-rectangle rect)
           (draw-object-as-icon (nth p (bay-displayer-displayed-objects self))
                                rect))))))

;;;************************************

(defmethod display-gripper ((self bay-displayer) arm-displayer container position)
  (let* ((inside-outside NIL)
         (orientation NIL)
         (real-position
          (cond
           ((eq container (bay-displayer-displayed-object self))
            (setf inside-outside :OUTSIDE)
            (if (> position (bay-displayer-num-positions self)) 0 position))
           (t (setf inside-outside :INSIDE)
              (position-if #'(lambda (c) 
                               (or (eq c container)
                                   (contains-object c container)))
                           (bay-displayer-displayed-objects self))))))
    (cond 
     ((null real-position) NIL)
     (t (let ((current-gripper-displays 
               (remove-if-not #'(lambda (gd) (= real-position (second gd)))
                              (bay-displayer-gripper-data self))))
          (cond
           ((= 0 (length current-gripper-displays))
            (setf orientation :LEFT))
           ((= 1 (length current-gripper-displays))
            (let ((current-orientation (third (car current-gripper-displays))))
              (setf orientation 
                (if (eq current-orientation :BOTTOM)
                    :LEFT
                    :BOTTOM))))
           (t (displayer-warning "Too many grippers displayed at ~a" self)))
          (let ((point (bay-displayer-gripper-coords self 
                                                      real-position 
                                                      orientation)))
            (arm-displayer-draw-gripper arm-displayer 
                                        point 
                                        orientation 
                                        inside-outside)
            (push (list arm-displayer real-position orientation inside-outside)
                  (bay-displayer-gripper-data self))
            t))))))

(defmethod undisplay-gripper ((self bay-displayer) arm-displayer)
  (let ((gripper-display (find-if #'(lambda (gd) (eq arm-displayer (first gd)))
                                  (bay-displayer-gripper-data self))))
    (when gripper-display
      (let ((point (bay-displayer-gripper-coords self 
                                                 (second gripper-display)
                                                 (third gripper-display))))
        (arm-displayer-erase-gripper arm-displayer 
                                     point
                                     (third gripper-display) 
                                     (fourth gripper-display))
        (setf (bay-displayer-gripper-data self)
          (remove gripper-display (bay-displayer-gripper-data self)))))))

;;;  Upper left corner where the gripper should display itself
;;;  Possible orientations are :LEFT and :BOTTOM

(defun bay-displayer-gripper-coords (self position orientation)
  (let ((icon-coords (get-bay-displayer-display-position self position)))
    (case orientation
      ((:LEFT) (add-point icon-coords :x -7 :y -5))
      ((:BOTTOM) (add-point icon-coords :x -5 :y (+ *small-icon-height* 20)))
      (OTHERWISE (displayer-warning "Can't compute gripper coords for ~a ~a ~a~%"
                                    self position orientation)))))
  