;***************************************************************************
; ARM.LISP
;*********
;;;
;;;**************************************

(defstruct (arm-displayer (:constructor really-make-arm-displayer)
                          (:print-function print-arm-displayer))
  displayed-object
  displayed-container
  displayed-position
  display-string
  dock-point
  parent-displayer)

(defun print-arm-displayer (self stream depth)
  (declare (ignore depth))
  (format stream "{Displayer for ~a}" (arm-displayer-displayed-object self)))

(defun make-arm-displayer (arm 
                           arm-bay-displayer 
                           parent
                           display-string)
  (let*  ((abdr   (display-region arm-bay-displayer))
          (y1 (point-y (rectangle-origin abdr)))
          (x2 (point-x (rectangle-other-point abdr)))
          (y2 (point-y (rectangle-other-point abdr)))
          (y (truncate (+ y1 y2) 2))
          (ad (really-make-arm-displayer 
               :displayed-object arm
               :displayed-container nil
               :displayed-position nil
               :display-string display-string
               :dock-point (make-point :x (+ 3 x2) :y (- y 15))
               :parent-displayer parent)))
    ad))

;;;****************************************************************************
;;;  Method definitions.   We actually don't respond to most because 
;;;  we're called only by the truck displayer.

(defmethod displayed-object ((self arm-displayer))
  (arm-displayer-displayed-object self))

(defmethod draw ((self arm-displayer))
  (arm-displayer-update self))

(defun arm-displayer-erase (self)
  (let ((position  (arm-displayer-displayed-position self)))
    (cond
     ((eq position 'folded)
      (disp.with-erasure (arm-displayer-really-draw self)))
     (t (undisplay-gripper (arm-displayer-parent-displayer self) self)))))

(defmethod redraw ((self arm-displayer))
  (arm-displayer-really-draw self))

;;;*****************************************************************

(defun arm-displayer-really-draw (self)
  (let ((container (arm-displayer-displayed-container self))
        (position  (arm-displayer-displayed-position self)))
    (when (or container position)
      (cond
       ((eq position 'folded)
        (arm-displayer-draw-folded self))
       (t (display-gripper (arm-displayer-parent-displayer self) 
                           self 
                           container 
                           position))))))

;;;*****************************************************************

(defmethod update-display ((self arm-displayer) object)
  (when (eq object (arm-displayer-displayed-object self))
    (arm-displayer-update self)))

(defun arm-displayer-update (self)
  (let* ((the-arm (arm-displayer-displayed-object self))
         (container (query the-arm 'claw-container))
         (position  (query the-arm 'claw-position)))
    (when (not (and (eq container (arm-displayer-displayed-container self))
                    (eq position (arm-displayer-displayed-position self))))
      (arm-displayer-erase self)
      (setf (arm-displayer-displayed-container self) container)
      (setf (arm-displayer-displayed-position self) position)
      (arm-displayer-really-draw self))))
  
;;;*****************************************************************

(defvar *gripper-length* 55)
(defvar *gripper-depth* 6)

(defun arm-displayer-draw-gripper (self p-input orientation inside-outside)
  (case orientation
    ((:LEFT)
     (let* ((p1 p-input)
            (p2 (add-point p1 :x *gripper-depth*))
            (p3 (add-point p1 :y (/ *gripper-length* 2)))
            (p4 (add-point p3 :x -3))
            (p5 (add-point p1 :y *gripper-length*))
            (p6 (add-point p5 :x *gripper-depth*))
            (p7 (add-point p3 :x *gripper-depth* )))
       (case inside-outside
         ((:OUTSIDE)
          (disp.draw-line p2 p1)
          (disp.draw-line p1 p5)
          (disp.draw-line p5 p6)
          (disp.draw-line p4 p3)
          (disp.with-font (disp.tiny-font)
            (disp.draw-text (arm-displayer-display-string self) 
                            (add-point p3 :x 3))))
       ((:INSIDE)
        (disp.draw-line p1 p5)
        (disp.draw-line p1 p7)
        (disp.draw-line p7 p5)
        (disp.draw-line p4 p3)
        (disp.with-font (disp.tiny-font)
          (disp.draw-text (arm-displayer-display-string self) 
                          (add-point p3 :x 3))))
       (OTHERWISE 
        (displayer-warning "Draw gripper got weird args: ~a ~a ~a ~a~%"
                           self p-input orientation inside-outside)))))
                                     
    ((:BOTTOM)
     (let* ((p1 p-input)
            (p2 (add-point p1 :y *gripper-depth*))
            (p3 (add-point p1 :x (/ *gripper-length* 2)))
            (p4 (add-point p2 :x (/ *gripper-length* 2)))
            (p5 (add-point p1 :x *gripper-length*))
            (p6 (add-point p2 :x *gripper-length*))
            (p7 (add-point p4 :y *gripper-depth*)))
       (case inside-outside
         ((:OUTSIDE)
          (disp.draw-line p1 p2)
          (disp.draw-line p2 p6)
          (disp.draw-line p6 p5)
          (disp.draw-line p4 p7)
          (disp.with-font (disp.tiny-font)
            (disp.draw-text (arm-displayer-display-string self) 
                            (add-point p4 :y -3))))
       ((:INSIDE)
        (disp.draw-line p2 p3)
        (disp.draw-line p3 p6)
        (disp.draw-line p6 p2)
        (disp.draw-line p4 p7)
        (disp.with-font (disp.tiny-font)
          (disp.draw-text (arm-displayer-display-string self) 
                          (add-point p3 :y -3))))
       (OTHERWISE 
        (displayer-warning "Draw gripper got weird args: ~a ~a ~a ~a~%"
                           self p-input orientation inside-outside)))))
    (OTHERWISE 
     (displayer-warning "Draw gripper got weird args: ~a ~a ~a ~a~%"
                        self p-input orientation inside-outside))))


(defun arm-displayer-draw-folded (self)
  (arm-displayer-draw-gripper self 
                              (arm-displayer-dock-point self) 
                              :left 
                              :outside))

(defun arm-displayer-erase-gripper (self point orientation inside-outside)
  (disp.with-erasure
   (arm-displayer-draw-gripper self point orientation inside-outside)))

