(let ((*stop-time* nil))

  ;; STOP-SIMULATOR
  ;;
  ;; Forces the simulator to stop execution at the current time by setting
  ;; the "go no further than..." variable, *stop-time*,to now.
  ;;

  (defun stop-simulator ()
    (setf *stop-time* (actual-time *the-world*)))


  ;; ADVANCE-SIMULATOR
  ;;
  ;; Given a time (possibly nil),
  ;; the simulator will be advanced until the time is reached (if nil,
  ;;  time is ignored), or someone halts the loop with a call to
  ;; (stop-simulator).
  ;; Note that *stop-time* is used to control when the simulator halts this
  ;;  loop, so setting *stop-time* can be used to halt the simulator
  ;;  early, (as stop-simulator does).

  (defun advance-simulator (to-time)
    (setf *stop-time* to-time)
    
    (execute-all-handlers *the-handler-queue*)
    (let ((boring-world
	   (do ((first-event (first-event *the-event-queue*)
			     (first-event *the-event-queue*)))
	       
	       ;; Stop the loop if there is the possibility that it may run
	       ;; forever (no stopping time, and no events on the queue).
	       
	       ((and (null *stop-time*) (null first-event))  t)
	     
	     ;; If the next event occurrs in the future, then try to generate
	     ;; exogenous events between now and then, or the stop-time,
	     ;; whichever is earlier.
	     
	     (let ((next-time
		    (cond ((null first-event)       
			   *stop-time*)
			  ((null *stop-time*) 
			   (time-when first-event))
			  ((compare-times (time-when first-event) '> 
					  *stop-time*)
			   *stop-time*)
			  (t  (time-when first-event)))))
	       
	       (when (compare-times (actual-time *the-world*) '< next-time)
		 (generate-exogenous-events (actual-time *the-world*))))
	     
	     ;; Generating exogenous  events may have changed the events in the
	     ;;  queue.  Make sure we are working with the FIRST event.
	     
	     (setf first-event (first-event *the-event-queue*))
	     
	     (cond
	      
	      ;; Now, the first event may be further in the future than we want
	      ;; to advance, or there may be no events in queue.  End the loop.
	      ((or (null first-event)
		   (and *stop-time*
			(compare-times (time-when first-event) '> 
				       *stop-time*)))
	       (return nil))
	      
	      ;; Or, The next event is earlier than the stop-time.  Advance
	      ;; the simulator state.
	      (t
	       
	       (progn
		 (advance-world-time *the-world* (time-when first-event))
		 
		 ;; Before every event, let processes update themselves,
		 ;; as opposed to only updating processes each time the clock
		 ;; is advanced.  This is necessary because at time T, new
		 ;; processes may be spawned that require updating at time T
		 ;; (such as processes for commands which take no time to
		 ;; complete- Within time T, a process is created, then dies.
		 ;; But before terminating, the update function must be called
		 ;; to effect the results of the command (e.g. post sensory
		 ;; info))
		 
		 ;; NOte: the above discussion is somewhat moot, as
		 ;; START-PROCESS automatically updates the process
		 ;; as soon as it is created.
		 
		 (update-all *the-process-queue* (actual-time *the-world*))
		 ;; (execute-all...) is not really necessary, since this is
		 ;; called in (update-all) after each process is updated
		 (execute-all-handlers *the-handler-queue*)
		
		 ;; updating processes and executing handlers may have
		 ;; changed the time of the first event.  Don't execute
		 ;; it if this is the case.
		 
		 (when (and (eq first-event (first-event *the-event-queue*))
			    (compare-times (time-when first-event) '=
					   (actual-time *the-world*)))
		   (dequeue-event *the-event-queue*)
		   (execute-event first-event)
		   (execute-all-handlers *the-handler-queue*))))))))
	   
      ;; After finishing the loop, if the desired time is still not
      ;; reached, advance to that time.
      
      (when (and *stop-time*
		 (compare-times (actual-time *the-world*) '< 
				*stop-time*))
	(advance-world-time *the-world* *stop-time*)
	(update-all *the-process-queue* *stop-time*)
	(execute-all-handlers *the-handler-queue*))
    
      ;; If execution stopped because all trucks are advancing arbitrarily
      ;; (*stop-time* = nil) and there are no events in the queue to
      ;; advance to, return the fact that we are currently in a boring world.
      ;; To do otherwise will cause the simulator to go into an infinite
      ;; loop, because all the trucks will be waiting for something
      ;; interesting to happen, when it clearly will not.
      
      (values boring-world)))

  )

