;;; PROPERTIES.LISP
;;;
;;; Implements a Class of objects where object state is not kept
;;; in slots (necessarily), but PROPERTIES.
;;;

(defvar *update-often* T)

;;; CLASS SIMULATOR-OBJECT
;;;
;;;  Is the superclass of all classes defined with DEFOBJECT.
;;;  The slot PROPERTIES is used to implement the property list of
;;;  SIMULATOR-OBJECT objects.

(defclass simulator-object 
    ()
    ((properties  :accessor properties
		  :initform (make-hash-table))
     (property-order :accessor property-order
		     :initform nil)))


;;;
;;; *CLASS-TABLE*
;;;
;;;  Keeps track of the representative object of each class defined
;;; with DEFOBJECT.  (gee, I wish the CL comittee would
;;;  agree on the meta-class system).

(defvar *class-table* (make-hash-table))


;;;
;;; CLASS OBJECT-PROPERTY
;;;
;;; These objects contain information about a given property of an object
;;;

(defclass object-property
    ()
    ((value       :accessor value	; the property value
		  :initarg :value)
     (side-effect :accessor side-effect	; Side effect of the property
		  :initform nil
		  :initarg :side-effect)
     (marker      :accessor marker	; Place to mark this property
		  :initform nil)	;  when traversing the side effect
					;  graph
     (input       :accessor dependents	; Conditions that depend on property's
		  :initform nil)	; value (input conditions of process)
     (output      :accessor updaters	; Conditions upon which the property's
		  :initform nil)))	; value depends (output conds of proc)

(defun make-prop (val side-effect)
  (make-instance 'object-property :value val :side-effect side-effect))

;;;
;;; DEFOBJECT
;;;
;;; Like DEFCLASS, lets the programmer define a subclass of SIMULATOR-OBJECT,
;;;
;;; The form of DEFOBJECT is:
;;;
;;;  (defobject <class-name> (<superclass-name>*)
;;;       (<property> <initform> [ (<effected-fn> ((<eprop> <pefn>)*)) ] )* )
;;;
;;; A new class called <class-name> is defined, with the given superclasses.
;;; The superclasses must all be subclasses of SIMULATOR-OBJECT.
;;; If no superclasses are given, the class is a direct subclass of
;;; SIMULATOR-OBJECT.
;;; As far as defining superclasses goes, this is equivalent to:
;;;
;;;   (defclass <class-name> (<superclass-name>* | SIMULATOR-OBJECT) ....)
;;; 
;;; The new class will inherit all properties (including initforms
;;; and side-effect functions) of the superclasses.  When
;;; more than one superclass defines the same property,
;;; the property definition of the superclass listed first shadows
;;; property definitions of later superclasses.
;;; This shadowing is determined separately for initforms and side-effects,
;;; i.e. If A and B are superclasses, listed in that order, and they both
;;; define property P, then A's initform for property P is the one used.
;;; But if A defines no side-effect function, and B does, then
;;; B's side-effect function is taken to be P's side effect function
;;; in the new class.
;;;
;;; <property> is an unquoted symbol that defines the property name.
;;; <initform> is a form that defines the default initialization value
;;;   for the property when a new object of this class is created.
;;;   This initform is evaluated when the DEFOBJECT is evaluated.
;;;
;;; If the (<effected-fn> ....) part is specified, then it will define
;;; a side effect that takes place whenever that property is SETP'ed.
;;;
;;;   <effected-fn> evaluates to a function that returns a list of objects
;;;     when given the object being SETPed.  For each object X in the returned
;;;     list, the following procedure is applied:
;;;   For each (<eprop> <pefn>) pair, the function <pefn> is called with 2
;;;     arguments: the object being SETPed, and object X.
;;;     Object X's property <eprop> is then SETPed to the value that <pefn>
;;;     returns.
;;;
;;; A more general mechanism may be used to specify SETP side effects by
;;; placing the keyword :EFFECT in place of the <effected-fn>.
;;; In this case, the :EFFECT is immediately followed by a function
;;; that takes 3 arguments: The object being SETP'ed, the old value of the
;;; property that was SETP'ed, and the new value of the property.  This
;;; function is invoked during a SETP just after the property's value has
;;; been changed to the new value.
;;;

;;; It may seem that forms are being wrapped in lambda functions
;;; unnecessarily.  Indeed, in #'(lambda () ,(second prop-def)),
;;; the lexical scope of the evaluation of DEFOBJECT would have
;;; been captured without wrapping the lambda around.
;;; But by wrapping the lambda around, we create a thunk that
;;; can be evaluated as late as possible (namely, at object creation time).
;;; This way, if #'foo is mentioned as an effected-object function
;;; before foo has been defined, that's OK, as long as it is defined
;;; before the first creation of an instance that uses foo.

(defmacro defobject (class-name (&rest superclasses) &rest properties)
  (let ((props (remove-if #'(lambda (p) (eq (car p) :slot)) properties))
	(slots (mapcar #'cdr 
		       (remove-if-not #'(lambda (p) (eq (car p) :slot)) 
				      properties))))
    `(prog1
	 (defclass ,class-name ,(if `,superclasses `,superclasses 
				  (list 'simulator-object)) 
		   ((property-order :allocation :class
				    :reader property-order
				    :initform nil)
		    ,@slots))
       (define-representative-object ',class-name ',superclasses
	 ;; For each property definition, pass a list of info about each
	 ;; property:
	 ,@(mapcar #'create-property-form props)))))

(defun create-property-form (prop-def)
  `(list
    ;; The name of the property
    ',(first prop-def)
    ;; The Initvalue of the property (to be evaluated
    ;; at object initialization time)
    #'(lambda () ,(second prop-def))
    
    ,@(mapcar #'create-side-effect-form (cddr prop-def))))

(defun create-side-effect-form (side-effect-spec)
  `(list
    ;; The Side-effect designator symbol
    ,(first side-effect-spec)
    ;; The locator/designator function (to be evaluated
    ;; at object initialization time)
    #'(lambda () ,(second side-effect-spec))
    ;; The rest of the side-effect definition
    ',(cddr side-effect-spec)))

(defun define-representative-object (class-name superclasses &rest prop-evals)
  ;; Set up the representative object of the class
  (let ((rep-obj (make-instance class-name)))

    (setf (slot-value rep-obj 'property-order)
      (combine-property-orders (mapcar #'car prop-evals)
			       (mapcar 
				#'(lambda (cn)
				    (property-order (gethash cn *class-table*)))
				superclasses)))
    
       
    ;; Set the properties unique to this class
    (dolist (prop-info prop-evals)
      (setf (gethash (first prop-info) (properties rep-obj)) 
	;; Load property with initform
	(make-prop (second prop-info) 
		   ;; Load property with side effect function
		   (make-side-effect-function (cddr prop-info)))))
    
    ;; Inherit properties from superclasses
    (dolist (sclass superclasses)
      (copy-property-table (properties rep-obj)
			   (properties (gethash sclass *class-table*))))
    
    ;; Put representative object in class table
    (setf (gethash class-name *class-table*) rep-obj)))

;;;
;;; Here's the order for initializing properties:
;;;
;;; First, the rightmost superclasses properties are initialized in the
;;;   order they were declared, minus properties mentioned in a left
;;;   superclass or in the subclass.
;;; Then, the next-left superclass's properties are initialized, minus
;;;   properties in a left superclass or in the subclass.
;;; And so on, until the leftmost superclass's properties are initialized.
;;; Then, the subclass's properties are initialized in the order
;;;   in which they were declared.
;;;
;;; This order is an attempt to provide some kind of useful scheme
;;; for allowing the "right" thing to happen at initialization time.
;;; Since initializing a property also invokes the side-effect function
;;; of that property, it is important that the effected properties
;;; already be created in the object.
;;;
;;; The philosophy behind this ordering is that the subclass is the
;;; "most specific" class, and the leftmost superclass is the next most
;;; "specific" class.  The assumption is that the most specific class
;;; will have properties that affect properties defined in less specific
;;; classes. So by initializing the less specific class properties first,
;;; they will exist to be affected when the more specific class properties
;;; are initialized.
;;;
;;; Note that this scheme breaks down when a more specific class re-defines
;;; a property declared in one of it's superclasses.  If this property
;;; is affected by a side-effect from one of these less specific classes,
;;; it (the property) won't be affected by that side effect at initialization
;;; time, because it won't have been created in the new object until
;;; _after_ the property causing the side-effect has been initialized.
;;;

(defun combine-property-orders (prop-order superclass-orders)
  (remove-duplicates (append (apply #'append (reverse superclass-orders))
			     prop-order)))

;;; The side effect function of a property is the execution of each
;;; side effect specified in DEFOBJECT, in the order they were specified.

(defun make-side-effect-function (side-effect-list)
  (if side-effect-list
      #'(lambda ()
	  (let ((side-effect-fns (mapcar #'make-single-side-effect-function
					 side-effect-list)))
	    #'(lambda (obj old new)
		(mapc #'(lambda (sef) 
			  (funcall sef obj old new))
		      side-effect-fns))))
    nil))

(defun make-single-side-effect-function (side-effect-spec)
	  
  ;; This function returns a function to be funcalled at object creation time.
  ;;   The result will be the actual function to use as the side-effect
  ;;   each time the property is setp'ed.
  ;; Recall that LOCALITY itself is a lambda form that encapsulates
  ;; the user's locality function within the environment of the DEFOBJECT
  (let ((kind (first side-effect-spec))
	(locality (second side-effect-spec))
	(rest (third side-effect-spec)))
    (cond
     ;; Keyword :effect as the effect-fn specifies that evaluating
     ;; LOCALITY will supply the side-effect
     ((eq kind :effect)

      (funcall locality))
     
     ((eq kind :create-objects)
      (let ((loc-fn (funcall locality)))
	#'(lambda (obj old new)
	    (let ((loc (funcall loc-fn obj)))
	      (mapc
	       #'(lambda (objinfo)
		   (dotimes (x (second objinfo))
		     (put-in loc
			     (apply #'make-sim-object
				    (first objinfo) ; The class's name
				    ;; Initial property values
				    (create-prop-val-list obj (cddr objinfo))))))
	       rest)))))
     
     ((eq kind :destroy-objects)
      (let ((victim-fn (funcall locality))) ; Evaluated at obj creation
	#'(lambda (obj old new)
	    (mapc #'destroy-object (insure-list (funcall victim-fn obj))))))
     
     
     ((eq kind :set-properties)
      (let ((gatherer (funcall locality)))
	#'(lambda (obj old new)
	    (mapc
	     #'(lambda (affected-obj)
		 (mapc
		  #'(lambda (pe)
		      (setp affected-obj (first pe)
			    (process-probability (second pe) obj
						 affected-obj)))
		  rest))
	     (insure-list (funcall gatherer obj))))))
     
     ((null kind)
      nil)
     
     (t   (error "Unrecognized side-effect: ~S" kind)))))


(defun create-prop-val-list (obj pval-pairs)
  (let ((list nil))
    (dolist (pair pval-pairs)
      (push (process-probability (second pair) obj) list)
      (push (first pair) list))
    list))

(defun copy-property-table (dest source &key evaluate-forms)
  (maphash #'(lambda (prop info)
	       (let* ((dest-entry (gethash prop dest))
		      (value      (if evaluate-forms
				      (funcall (value info))
				    (value info)))
		      (side-effect (side-effect info)))
		 (cond
		  ;; If property didn't exist in destination table,
		  ;;  add it
		  ((null dest-entry) 
		   (setf (gethash prop dest)
		     (make-prop value side-effect)))
		  
		  ;; If property in destination table didn't have SIDE-EFFECT
		  ;;  function, inherit SIDE-EFFECT function.
		  ((null (side-effect dest-entry))
		   (setf (gethash prop dest)
		     (make-prop (value dest-entry) side-effect))))))
	   source))
		

;;;
;;; (QUERY <object> <property>)
;;;
;;;  Returns the value of an object's property.
;;;  <object> is an instance of some SIMULATOR-OBJECT (or subclass)
;;;  <property> is a symbol that names some property of
;;;    the object.
;;;
;;; If the object has the property, the values <property-value> T are returned
;;; If the object does not have the property,  NIL NIL is returned.
;;;
;;; The call to update-updaters insures that all process which have
;;; declared the property as an output condition, have been updated.
;;;

(defun query (object property)
  (if (typep object 'simulator-object)
      (let ((entry (gethash property (properties object))))
	(cond
	 (entry
	  (update-updaters entry) 
	  (values (value entry) t))
	 (t (values nil nil))))
    (values nil nil)))

;;;
;;; (SETP <object> <property> <new-value>)
;;;
;;; Sets an object's property
;;;
;;; <object> is an instance of some SIMULATOR-OBJECT (or subclass)
;;; <property> is a symbol that names some property of the object.
;;; <new-value> is a form that evaluates to the new value
;;;   the property will take on.
;;;
;;; After the property value has been set, the side-effect function
;;; (defined in DEFOBJECT) for that property is invoked.  The 
;;;
;;; If the object has the property, the values <new-value> T is returned.
;;; If the object does'nt have the property,   NIL NIL is returned.
;;;


(defvar *side-effect-flag* nil)		; Gets set to the current simulator
					; time when a side effect
					; is being propigated
(setf *side-effect-flag* nil)

(defvar *side-effect-marker* 0)
(setf *side-effect-marker* 0)

(defun setp (object property new-value)
  (if (typep object 'simulator-object)
      (let ((entry (gethash property (properties object))))
	(cond
	 ((null entry) (values nil nil))
	 (t
	  (setf old-value (value entry))
	  (setf (value entry) new-value)
	  (notify-dependents entry)
	  (when (side-effect entry)
	    (cond
	     
	     ;; If we are currently propigating a side effect through
	     ;; the system, and this property has already been marked
	     ;; by the current traversal, flag it as a possible circularity,
	     ;; and prune this path of the traversal.
	     
	     ((and (eql *side-effect-flag* (actual-time))
		   (eql *side-effect-marker* (marker entry)))
	      (format *terminal-io* 
		      "WARNING: Cycle detected while propagating side effect~% in Object: ~S  Property: ~S~%  Side effect not propagated any further~%"
		      object property))

	     ;; We're continuing the side effect's propagation.
	     
	     ((eql *side-effect-flag* (actual-time))
	      (setf (marker entry) *side-effect-marker*)
	      (funcall (side-effect entry) object old-value (value entry))
	      (setf (marker entry) nil))
	     
	     ;; We're starting the propagation of a side effect
	     (t
	      (setf *side-effect-flag* (actual-time))
	      (incf *side-effect-marker*)
	      (setf (marker entry) *side-effect-marker*)
	      (funcall (side-effect entry) object old-value (value entry))
	      (setf (marker entry) nil)
	      (setf *side-effect-flag* nil))))
	      
	  ;; Experiment: update displayers every time an object changes state
	  (when *update-often*
	    (update-displayers object))	;**UPDATE**
	  (values (value entry) t))))
    (values nil nil)))

(defun notify-dependents (property)
  (dolist (depend (dependents property))
    (when (enable depend)
      (cond
       
       ;; If this input condition has been marked with the current
       ;;   simulator time, then that means that we're somewhere
       ;;   in the middle of executing a handler invoked because
       ;;   of this very same condition.  Let's be conservative,
       ;;   flag this as a possible circularity, and not process the
       ;;   condition.  (see execute-handler)
       
       ((eql (actual-time) (marker depend))
	(format *terminal-io* 
		"WARNING: Cycle detected while processing input condition handler~% for process: ~S  Name: ~S  Object: ~S  Property: ~S~%  Second handler not invoked~%"
		(proc depend) (name depend) (object depend) (property depend)))

       (t
	(enqueue-handler *the-handler-queue*
			 (make-instance 'handler-info 'who depend)))))))

;;;
;;; UPDATE-UPDATERS
;;;
;;; This invokes each process that has declared itself an updater
;;; (the property is an output condition of the process).
;;; *query-update-marker* and *query-update-flag* are used
;;; to mark each maintainance condition, so that if, during the course
;;; of this update, any of these processes would be invoked again because of
;;; the same output condition, It will be flagged as a warning, and
;;; the cycle will be broken.
;;;
;;; Actually, there isn't any danger of a cycle, since UPDATE-PROCESS
;;; doesn't invoke the process's update code if it has already
;;; been invoked at this simulator time.  But this makes everything
;;; official, and secure.


(defvar *query-update-marker* 0)
(defvar *query-update-flag*  nil)	; If in the middle of processing
					; a query, will be set to the
					; current simulator time.

(defun update-updaters (property)
  (dolist (updater (updaters property))
    (when (enable updater)
      (cond
       
       ;; This maintainance condition has already been visited
       ;; by the current QUERY traversal.  Flag a warning, and
       ;; don't process the condition.
       
       ((eql *query-update-marker* (marker updater))
	(format *terminal-io* 
		"WARNING: Cycle detected while processing output condition for process: ~S  Name: ~S  Object: ~S  Property: ~S~%  Output condition not processed~%"
		(proc updater) (name updater) (object updater)
		(property updater)))
	
       ((not (eql *query-update-flag*
		  (actual-time)))	; We are not currently processing
					;  as a result of another QUERY
	(setf *query-update-flag* (actual-time))
	(incf *query-update-marker*)	; make a new "visited" marker
	(setf (marker updater)		; Mark this output condition as visited
	  *query-update-marker*)
	
	(update-process (proc updater)	; Execute the process which
			(token (proc updater)) ; will update the property
			(actual-time))
	(setf (marker update) nil)
	(setf *query-update-flag* nil))	; End the QUERY graph traversal
       
       
       (t				; We are currently in the middle of
					;  a QUERY graph traversal
	(setf (marker updater)
	  *query-update-marker*)	; Mark this output condition as visited
	(update-process (proc updater)	; Execute the process which
			(token (proc updater)) ; will update the property
			(actual-time))
	(setf (marker updater) nil))))))
	

;;;
;;;
;;;

(defun make-prop-value-lists (list)
  (do ((props nil)
       (values nil))
      ((null list) (values props values))
    (push (pop list) props)
    (push (pop list) values)))


;;;
;;; (make-object <classname> {<property> <value>}*)
;;;
;;; Creates an instance of the class named by <classname>.
;;; Initialization values can be given for specific properties/slots,
;;; which will override the default values defined by DEFOBJECT.
;;; <property> is a symbol that names a property or a slot.
;;; <value> is a form that evaluates to the initialization value of
;;;   the property.
;;;

(defun make-object (class &rest property-pairs)
  (multiple-value-bind (props initvalues)
      (make-prop-value-lists property-pairs)
    (let ((meta-obj (gethash class *class-table*))
	  (new-obj (make-instance class)))
      
      ;; Initialize each property in order it was declared
      
      (mapcar
       #'(lambda (prop-name)
	   (let* ((entry (gethash prop-name (properties meta-obj)))
		  (pos   (position prop-name props))
		  ;; If a initvalue is specified as a parameter, use it,
		  ;;  else, evaluate the default initvalue
		  (init  (if pos
			     (nth pos initvalues)
			   (funcall (value entry)))))
	     
	     ;; Install the property in the new object, and evaluate
	     ;; the side effect function.
	     (setf (gethash prop-name (properties new-obj))
	       (make-prop init (if (side-effect entry)
				   (funcall (side-effect entry))
				 nil)))))
	     
       (property-order meta-obj))
      
      ;; This is kludgy: After properties have been assigned their initial
      ;; values, SETP all of them, invoking their side effects
      
      (mapcar
       #'(lambda (prop-name)
	   (setp new-obj prop-name (query new-obj prop-name)))
       (property-order meta-obj))

      ;; Initialize slots
      (mapcar
       #'(lambda (prop value)
	   (let ((entry (gethash prop (properties new-obj))))
	     (cond
	      ;; If prop names a property, do nothing: it has already been done
	      ((gethash prop (properties new-obj)) t)
	      ;; If it is a slot, initialize it
	      ((slot-exists-p new-obj prop)
	       (setf (slot-value new-obj prop) value))
	      (t
	       (cerror "Create object, ignoring property"
		       "~S not a property or slot of ~S"
		       prop
		       class)))))
       props
       initvalues)
      
      new-obj)))
    
	     
      
      
      
;;;      ;; Put default values in properties
;;;      (copy-property-table (properties new-obj)
;;;			   (properties (gethash class *class-table*))
;;;			   :evaluate-forms t)
;;;      
;;;      ;; Assign given values to properties
;;;      (mapcar
;;;       #'(lambda (prop value)
;;;	   (let ((entry (gethash prop (properties new-obj))))
;;;	     (cond
;;;	      (entry (setf (value entry) value))
;;;	      ((slot-exists-p new-obj prop)
;;;	       (setf (slot-value new-obj prop) value))
;;;	      (t
;;;	       (cerror "Create object, ignoring property"
;;;		       "~S not a property or slot of ~S"
;;;		       prop
;;;		       class)))))
;;;       props
;;;       initvalues)
;;;      new-obj)))

      
    

;;;
;;; INSTALL-CONDITION
;;;
;;;  Places a maintainance condition onto the dependents list of a property,
;;;  and the condition list of a process.
;;;

(defun install-condition (maint-cond)
  (let ((entry (gethash (property maint-cond) 
			(properties (object maint-cond)))))
    (if entry
	(push maint-cond (slot-value entry (inout maint-cond)))
      (cerror "Ignore" "Object ~A has no property ~A" object property))
    (setf (prop-entry maint-cond) entry)
    (push maint-cond (slot-value (proc maint-cond) (inout maint-cond)))))


;;; UNINSTALL-CONDITION
;;;
;;;   Removes a maintainance condition process from the process'
;;; List of maintainance conditions, and removes maintainance condition
;;; from the property it is bound to.
;;;

(defun uninstall-condition (maint-cond)
  (let ((entry (gethash (property maint-cond) 
			(properties (object maint-cond)))))
    (setf (slot-value entry (inout maint-cond))
      (delete maint-cond (slot-value entry (inout maint-cond))))
    (setf (slot-value (proc maint-cond) (inout maint-cond))
      (delete maint-cond (slot-value (proc maint-cond) (inout maint-cond))))))



;;;;; Debugging functions

(defun show-properties (obj)
  (maphash #'(lambda (key info) (format t "~S = ~S~%" key (value info)))
	   (properties obj)))

(defun show-classes ()
  (maphash #'(lambda (key info) (format t "~S = ~S~%" key value))
	   *class-table*))

(defun representative (class-name)
  (gethash class-name *class-table*))

