;; PROCESS.LISP
;;
;; DEFINES:
;;   class      process
;;
;; BUGS:
;;   None


;; CLASS PROCESS DEFINED IN FILE "PROCESS-DEF.LISP"

;;;*****************************************************
;;;
;;;  CONDITION MANAGEMENT : public
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; handling INPUT conditions
;;;

;;;
;;; CONDITION-LIST
;;;
;;; Takes arguments which are grouped into quadruples of
;;; <object> <property-name> <maint-cond-name>
;;; Where <object> is a simulator object, <property-name> is the name
;;; of a property of <object>, <maint-cond-name> is the name the conditional
;;; will be known by.
;;; Returns a list of these triples, in no particular order.
;;;
;;; This is a support function to the process-manager. See
;;; START-PROCESS and process definition functions
;;;

(defun condition-list (&rest triples)
  (do ((temp nil)
       (head triples (cdddr head)))
      ((null head) temp)
    (push (list (car head) (cadr head) (caddr head)) temp)))

;;;
;;; ENABLE-CONDITION
;;;
;;; Enables a maintainance condition for the process named by TOK,
;;; so that changes to the condition's state will invoke the process.
;;;

(defun enable-condition (tok name)
  (diddle-io-condition 'input tok name t))

;;
;; DISABLE-CONDITION
;;
;; Disables a maintainance condition for the process named by TOK,
;; so that changes to the condition's state do not invoke the process.
;;

(defun disable-condition (tok name)
  (diddle-io-condition 'input tok name nil))

;;;
;;; ADD-CONDITION
;;;
;;; Adds a maintainance condition to the process with token TOK
;;;

(defun add-condition (tok object property name)
  (add-io-condition 'input tok object property name))

;;; DELETE-CONDITION
;;;
;;; Deletes a maintainance condition with name NAME
;;; from the process with token TOK.
;;;

(defun delete-condition (tok name)
  (delete-io-condition 'input tok name))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; handling OUTPUT conditions
;;;

;;;
;;; ENABLE-OUTPUT-CONDITION
;;;
;;; Enables a maintainance condition for the process named by TOK,
;;; so that changes to the condition's state will invoke the process.
;;;

(defun enable-OUTPUT-condition (tok name)
  (diddle-io-condition 'output tok name t))

;;
;; DISABLE-OUTPUT-CONDITION
;;
;; Disables a maintainance condition for the process named by TOK,
;; so that changes to the condition's state do not invoke the process.
;;

(defun disable-OUTPUT-condition (tok name)
  (diddle-io-condition 'output tok name nil))

;;;
;;; ADD-OUTPUT-CONDITION
;;;
;;; Adds a maintainance condition to the process with token TOK
;;;

(defun add-output-condition (tok object property name)
  (add-io-condition 'output tok object property name))

;;; DELETE-OUTPUT-CONDITION
;;;
;;; Deletes a maintainance condition with name NAME
;;; from the process with token TOK.
;;;

(defun delete-OUTPUT-condition (tok name)
  (delete-io-condition 'output tok name))


;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; General functions to deal with both input and output conditions (private)
;;;


(defun diddle-IO-condition (io tok name value)
  (let ((proc (fetch-process tok)))
    (if (null proc)
	(cerror "Ignore and continue" "Process ~S does not exist" tok)
      (let ((mc (fetch-io-cond io proc name)))
	(if (null mc)
	    (cerror "Ignore and continue" 
		    "Process ~S does not have maintainance condition ~S" 
		    tok name)
	  (setf (enable mc) value))))))

;;;
;;; ADD-IO-CONDITION
;;;
;;; Adds a maintainance condition to the process with token TOK
;;;

(defun add-IO-condition (io tok object property name)
  (let ((proc (fetch-process tok)))
    (if proc
	(install-condition 
	 (make-instance 'maintainance-condition 
	   :proc     proc
	   :object   object
	   :property property
	   :name     name
	   :inout    io
	   :enable   t)))))

;;;
;;; DELETE-IO-CONDITION
;;;
;;; Deletes a maintainance condition with name NAME
;;; from the process with token TOK.
;;;

(defun delete-IO-condition (io tok name)
  (let ((proc (fetch-process tok)))
    (if proc
	(let ((cond (find-if #'(lambda (x) (eq name (name x)))
			     (slot-value proc io))))
	  (if cond
	      (uninstall-condition cond))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; PROCESS MANAGEMENT : Visible functions
;;
;;


;; START-PROCESS
;;
;; Creates a process, with the given token, 3-tuples, representing
;; the maintainance conditions of the process (see CONDITION-LIST, above),
;; and update function.
;; Conditionals are linked to, and the process is put on the process queue
;;

;; Note: start-process should be modified so that it either doesn't take
;; input (maintainance) conditions, or else takes both input _and_ output
;; conditions, to preserve symmetry.

(defun start-process (maint-triples update &key (token (new-token "PROC")))

  ;; Create the process instance
  (let ((new-process (make-instance 'process 
				    'token token 
				    'update update)))
    ;; Link to all it's conditionals.
    ;; conditional
    (mapcar #'(lambda (triple)
		(install-condition 
		 (make-instance 'maintainance-condition 
		   :proc new-process
		   :object (first triple)
		   :property (second triple)
		   :name    (third triple)
		   :inout   'input
		   :enable  t)))
	    maint-triples)
    
    ;; Turn process 'ON'
    (setf (active new-process) t)

    ;; Put process in process queue
    (setf *the-process-queue* (nconc *the-process-queue* (list new-process)))

    ;; Give the process a chance to run, perhaps initializing itself
    (update-process new-process (token new-process) (actual-time) 'advance)
		    
    (token new-process)))

;;;
;;; KICK-PROCESS
;;;
;;; Places an null event on the queue at the specified time, insuring
;;; that processes will be updated.
;;;

(defvar *null-event-action* #'(lambda (tok time) nil))

(defun kick-process (tok time)
  (insert-action time *null-event-action*))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Internal functions

;;
;; FETCH-IO-COND
;;
;; Fetches a process's condition (input or output) by name
;;

(defun fetch-io-cond (io proc mc-name)
  (find-if #'(lambda (mc) (eq (name mc) mc-name)) (slot-value proc io)))




;; KILL-PROCESS
;;
;; unlinks any conditionals,
;; and removes itself from the process queue.
;;

(defmethod kill-process  ((self process))

  ;; remove any stop event in the queue

  (remove-event *the-event-queue* (token self))

  ;; Unlink all maintainance conditions
  (loop while (conditions self) do
	(uninstall-condition (car (conditions self))))
  
  (loop while (output self) do
	(uninstall-condition (car (output self))))
  
  ;; Turn process 'OFF'
  (setf (active self) nil)

  ;; Remove from the process queue
  (setf *the-process-queue* (delete self *the-process-queue*)))


;; Kills any processes (there should be only one) with the given token.

(defmethod kill-process (tok)
  (let ((proc (fetch-process tok)))
    (if proc
	(kill-process proc))))

;;
;; FETCH-PROCESS
;;
;; Given a token, returns the process in the process-queue that corresponds
;; to the token, and nil if it wasn't found.
;;

(defun fetch-process (token)
  (find-if #'(lambda (p) (eq (token p) token)) *the-process-queue*))



;;
;; UPDATE-PROCESS
;;
;;  Calls the process'es UPDATE function, unless:
;;     - The update is a normal update (not conditional) AND
;;     - the process has already been updated for the given time
;;  The most immediate result is that a process is only updated
;;    ONCE for a given time, unless one of it's maintainance conditions
;;    also invokes the process.
;;  This also prevents redundant (and possibly expensive) updates.
;;  

(defun update-process (proc tok time &optional (why 'advance))
  (when (update-function proc)
    
    ;; Execute the update function
    (case why
      (advance
       (when (or (null (last-advanced proc))
		 (compare-times (last-advanced proc) '/= time))
	 ;;	 (format t "Updating ~S at ~S : ~S~%" tok time why)
	 (execution-context `(:UPDATE ,tok)
  	    (funcall (update-function proc) tok time why)))
       (setf (last-advanced proc) time))

      (otherwise (funcall (update-function proc) tok time why)))))



;;
;; UPDATE-ALL
;;
;; Updates all the processes in a queue
;;

(defun update-all (pq time &optional (why 'advance))
;;  (format t "Updating all processes at ~S~%" time) ; *DEBUG*
  (mapc #'(lambda (p) 
	    (update-process p (token p) time why)
	    (execute-all-handlers *the-handler-queue*))
	pq))

;;
;; PRINT-OBJECT
;;

(defmethod print-object ((self process) out)
  (format out "<PROC:~S>" (token self)))

