;; HANDLER-INFO.LISP
;;
;; DEFINES:
;;   class    handler-info
;;   methods  execute-handler

(defclass handler-info ()          ; Contains info to invoke a handler
  ((who    :accessor who           ;   Which process' handler to invoke?
	   :initarg  who))
  (:default-initargs who nil))


;;**********************************************************************
;;
;; METHODS
;;

;;;
;;; EXECUTE-HANDLER
;;;
;;;   Given a handler-info, invokes the handler for the specified process,
;;; passing the condition that caused the invocation to the update function
;;; of the process.
;;; If the process has already stopped (by the invocation of a handler
;;; earlier in the queue), then the handler will not be invoked.
;;; While a handler is executing, calling (name <conditional>) will
;;; return the name assigned to the conditional by the handling process.
;;; Note that NAME only lets you ask about the name of the conditional
;;; currently being handled
;;;
;;; The MARKER of the input condition that caused the invocation
;;;  is set to the current time while the process is executing.
;;; This will allow the system to determine if a circularity exists
;;;  whenever the property of the condition is diddled in the course
;;;  of executing the process.
;;;

(let ((current-conditional-name nil)
      (current-conditional      nil))

  (defun execute-handler (self)
    (when (active (proc (who self)))
      (setf current-conditional-name (name (who self)))
      (setf (marker (who self)) (actual-time))
      (execution-context `(:HANDLER ,self)
         (update-process (proc (who self))
			 (token (proc (who self))) 
			 (actual-time *the-world*)
			 'condition))
      (setf (marker (who self)) nil)
      (setf current-conditional-name nil)))
  
  (defmethod name (self)
    current-conditional-name)
  
)
  

;;
;; PRINT-OBJECT
;;

(defmethod print-object ((self handler-info) out)
  (format out "<HI:~S>" (who self)))
