
;;;
;;; CLASS EVENT-GENERATOR
;;;
;;; An event-generator is responsible for generating a series of exogenous
;;; events.  

(defclass event-generator     
    ()
    ((prev-event       :accessor prev-event
		       :initform nil)
     (gen-function     :accessor gen-function
		       :initarg  :gen-function)))  

;;; The GEN-FUNCTION of an event-generator has the following semantics:
;;;
;;; The function must take no arguments.
;;; The function must return 1 value: either nil or an event.
;;;    If it returns an event, then that event will be enqueued
;;;     on the event queue.
;;;    If it returns nil, nothing happens.
;;;
;;; Each invocation of gen-function should return the next event in the
;;; series.
;;; The sequence of events returned by
;;; the invocations of this function must have non-decreasing times.
;;;
;;; The function is guaranteed to be called at least once at the simulator's
;;; initial time.  For each event which the function returns whose execution
;;; time is X, then the function is guarnteed to be called at least once
;;; at time X.  I.E. If the function returns a time 10 event, then the function
;;; will be called again at time 10.  If the function then returns another
;;; time 10 event, it will be called again at time 10.
;;; For efficiency purposes, if the event which the function
;;; last returned has execution time Y, then the function will not be called
;;; again until time Y.
;;;

;;;
;;; GENERATE-EVENT
;;;
;;; Calling GENERATE-EVENT gives an event generator the chance
;;; to generate its next event.  This function decides when
;;; to invoke the GEN-FUNCTION, as described above.
;;;

(defun generate-event (gen now)
  (if (or (null (prev-event gen))
	  (compare-times now '>= (time-when (prev-event gen))))
      (let ((next-event (funcall (gen-function gen))))
	(when next-event
	  (cond
	   ((compare-times now '<= (time-when next-event))
	    (insert-event next-event)
	    (setf (prev-event gen) next-event))

	   (t (cerror "Continue, ignoring event"
		      "At time ~A, exogenous event generated for time ~A"
		      now (time-when next-event))))))))


;;;
;;; MAKE-EVENT-GENERATOR
;;;
;;; Creates an event generator out of the given function which must
;;; conform to the above semantics.
;;;

(defun make-event-generator (gen-func)
  (make-instance 'event-generator :gen-function gen-func))
    

