

;;;
;;; GENERATE-EXOGENOUS-EVENTS
;;;
;;; Called with the current simulator time, invokes all of the event generators
;;; in *event-generators*
;;; This function must always be called before simulator time is advanced.
;;;

(defun generate-exogenous-events (now-time)
  (mapc #'(lambda (gen) (generate-event gen now-time))  (exogenous-events *the-world*)))

;;; INSTALL-EXOGENOUS-EVENT
;;;
;;; Installs an event generator on the generator list of a world
;;;

(defun install-exogenous-event (event &optional (world *the-world*))
  (push event (exogenous-events world)))

;;; MAKE-EXOGENOUS-EVENT
;;;
;;; Creates an exogenous event
;;;

(defun make-exogenous-event (&rest keywords
			     &key time-list
				  time-period
				  (time-offset 0)
			     &allow-other-keys)
  
  ;; TIMER is the function that generates a sequence of times
  (let ((timer (cond (time-list   (make-time-list-timer time-list))
		     (time-period (make-periodic-timer  time-period time-offset)))))
    (make-event-generator
     (make-timed-event-generator
      timer
      ;; Compute the effect function of the event
      (make-event-generator-function keywords)))))

  
  
;;; MAKE-TIMED-EVENT-GENERATOR
;;;
;;;  Returns a function  () -> event, for installing in an event-generator
;;;
;;; TIMER : A function that returns successive times for successive calls
;;; GEN   : An event function: (token time) -> void 
;;;

(defun make-timed-event-generator (timer gen)
  #'(lambda ()
      (let ((next-time (funcall timer)))
	(if next-time
	    (make-event next-time gen)
	  nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Exogenous event actions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; MAKE-EVENT-GENERATOR-FUNCTION
;;;
;;; Given: the keyword arguments to MAKE-EXOGENOUS-EVENT
;;; Returns : An event function (token time) -> void   that
;;;   executes the create/destroy/set effects of the event, as specified
;;;   by the (possibly multiple) keywords :SET-PROPERTIES, CREATE-OBJECTS
;;;   and DESTROY-OBJECTS.
;;;

(defun make-event-generator-function (keyword-args)
  (let ((effects nil))
    (do ((head keyword-args (if (keywordp (first head))
				(cddr head)
			      (cdr head))))
	((null head) 
	 (setf effects (nreverse effects))
	 (if effects
	     #'(lambda (tok time)
		 (mapc #'(lambda (e) (funcall e tok time)) effects))
	   (error "Exogenous event must have at least one effect")))
      
      (case (first head)
	(:CREATE-OBJECTS
	 (push (apply #'make-object-creation-event (second head)) effects))
	(:DESTROY-OBJECTS
	 (push (apply #'make-object-destruction-event (second head)) effects))
	(:SET-PROPERTIES
	 (push (apply #'make-property-setting-event (second head)) effects))))))

;;;
;;; MAKE-PROPERTY-SETTING-EVENT
;;;
;;; OBJECTS : list of objects to affect
;;; OBJECT-FUNCTION : locality function (optional) to apply to OBJECTS list
;;;                   at event execution time.
;;; PROPERTES : list of properties to set
;;; VALUES   : PROB-COND that returns a list of values to set the properties
;;;
;;; Returns : An event function (token time) -> void   that starts a process
;;;           which sets the properties of the objects to the values
;;;

(defun  MAKE-PROPERTY-SETTING-EVENT (objects object-function &rest prop-vals)
  #'(lambda (tok start-time)
      (start-process
       nil
       
       #'(lambda (tok time why)
	   (mapcar
	    #'(lambda (obj)
		(mapcar
		 #'(lambda (prop-info)
		     (setp obj
			   (first prop-info)
			   (process-probability (second prop-info) obj)))
		 prop-vals))
	    (if object-function
		(mapcan object-function objects)
	      objects))
	   (stop-process tok)))))

      
;;;
;;; MAKE-OBJECT-CREATION-EVENT
;;;
;;; LOCATION: A container object in which to place the created objects
;;; OBJ-SPECS: A list of (<class-name> <number> (<property> <prob-cond>)*)
;;;            object specifications.
;;;
;;; Returns : An event function (token time) -> void   that starts a process
;;;           which creates the specified number of objects, with initial
;;;           property values specified by the (<property> <prob-cond>) pairs.
;;;

(defun  MAKE-OBJECT-CREATION-EVENT (location &rest obj-specs)
  #'(lambda (tok start-time)
      (start-process
       nil
       
       #'(lambda (tok time why)
	   (mapc
	    #'(lambda (objinfo)
		(dotimes (x (process-probability (second objinfo)))
		  (put-in location
			  (apply #'make-sim-object
				 (first objinfo) ; The class's name
				 ;; Initial property values
				 (create-prop-val-list nil (cddr objinfo))))))
	    obj-specs)
      
	   (stop-process tok)))))

      
;;;
;;; MAKE-OBJECT-DESTRUCTION-EVENT
;;;
;;; OBJECTS : list of objects to destroy or use in computing destroyed objects
;;; OBJECT-FUNCTION : locality function (optional) to apply to OBJECTS list
;;;                   which computes the objects to be destroyed.
;;;
;;; Returns : An event function (token time) -> void   that starts a process
;;;           which destroys the specified objects.
;;;

(defun  MAKE-OBJECT-DESTRUCTION-EVENT (objects object-function)
  #'(lambda (tok start-time)
      (start-process
       nil
       
       #'(lambda (tok time why)
	   (mapc #'destroy-object
		 (if object-function
		     (mapcan object-function (insure-list objects))
		   (insure-list objects)))
	   (stop-process tok)))))

      
;;;
;;; MAKE-SET-OBJECT-EVENT
;;;
;;; OBJECTS : list of objects to affect
;;; OBJECT-FUNCTION : locality function (optional) to apply to OBJECTS list
;;;                   at event execution time.
;;; VALUES   : PROB-COND that returns a list of values to use as arguments
;;;            to SET-OBJECT
;;;
;;; Returns : An event function (token time) -> void   that starts a process
;;;           which invokes SET-OBJECT on each object in the OJBECTS list
;;;

;;;  (defun make-set-object-event (objects object-function values)
;;;    #'(lambda (tok time)
;;;	(start-process
;;;	 nil
;;;	 #'(lambda (tok time why)
;;;	     (mapcar
;;;	      #'(lambda (obj)
;;;		  (apply #'set-object obj nil (process-probability values obj)))
;;;	      (if object-function
;;;		  (mapcan object-function objects)
;;;		objects))
;;;	     (stop-process tok)))))
;;;      
;;;
;;;;;;
;;;;;; MAKE-PROCESS-STARTING-EVENT
;;;;;;
;;;;;; OBJECTS : list of objects to affect
;;;;;; OBJECT-FUNCTION : locality function (optional) to apply to OBJECTS list
;;;;;;                   at event execution time.
;;;;;; PROCESS : A function that takes as input the OBJECTS list, and
;;;;;;           the current time of the event, and returns
;;;;;;           2 values: the condition list, and the update function for
;;;;;;           the process, used in the call to START-PROCESS
;;;;;;
;;;;;; Returns : An event function (token time) -> void   that starts the
;;;;;;           process specified by the PROCESS argument.
;;;
;;;  (defun make-process-starting-event (objects object-function process)
;;;    #'(lambda (tok time)
;;;	(multiple-value-bind (conds update)
;;;	    (funcall process 
;;;		     (if object-function
;;;			 (mapcan object-function objects)
;;;		       objects)
;;;		     time)
;;;	  (start-process conds update))))

;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Exogenous event timers
;;;
;;;;;;;;;;;;;;;;;;;;;;

(defun make-time-list-timer (tv-list)
  (let ((tv-head tv-list))
    #'(lambda ()
	(prog1
	    (car tv-head)
	  (setf tv-head (cdr tv-head))))))

(defun make-periodic-timer (periods offset)
  (let ((previous-time offset)
	(period-list periods))
    #'(lambda ()
	(setf previous-time
	  (max (round (add-times previous-time (process-probability (pop period-list))))
	       previous-time))
	(when (null period-list)
	  (setf period-list periods))
	previous-time)))









