;; EVENT-QUEUE.LISP
;;
;; DEFINES:
;;
;;   class  event-queue
;;   var    *the-event-queue*
;;   functions:
;;          insert-event
;;          remove-event
;;          replace-event
;;          execute-events-until
;;          execute-events-until-visible
;;
;; BUGS:
;;   None yet
;;

;;******************************************************
;;
;; EVENT-QUEUE class
;;


(defclass event-queue ()
  ((events :accessor events    ; Time-ordered list (earliest to latest)
	   :initform '() )))   ; of events to happen

;;**********************************************************************
;;
;; *THE-EVENT-QUEUE*  :  The global event queue that controls the simulator
;;

(defvar *the-event-queue* (make-instance 'event-queue)
  "Global event queue that drives the entire simulator")

;;**********************************************************************
;; METHODS
;;

;;
;; INSERT-EVENT
;;
;;   destructively inserts new-event into queue,
;;  preserving the time-ordering of the queue
;; The new event is placed so that it is dequeued after
;; any pre-existing events which have the same time-when's.
;;
;; Returns the token of the event inserted.

(defun insert-event (new-event &optional (self *the-event-queue*))

  ;; Find spot to insert new-event in the queue

  (let ((insert-at nil))
    (do ((search (events self) (cdr search)))
	((or (null search)
	     (compare-times (time-when new-event) '< (time-when (car search))))
	 nil)
      (setf insert-at search))
    
    ;; At end of loop, insert-at is the list whose first element
    ;;  should go just before the new-event: i.e. the event
    ;;  that is the latest event before new-event.  Insert
    ;;  new-event just after this event.  If insert-at is nil,
    ;;  Then the new-event goes at the head of the queue.
    
    (if (null insert-at)
	(setf (events self) (cons new-event (events self)))
      (setf (cdr insert-at) (cons new-event (cdr insert-at)))))
  
  (setf (in-queue? new-event) t)

  ;; Now, enter the event in the token table
  (insert-token *the-token-table* (token new-event) new-event)
  (token new-event))

;;
;; INSERT-ACTION
;;
;; A convenience: Takes an action (a function of 2 arguments, a token and time)
;; and places it in the event-queue for execution at the given time.
;; All the keywords arguments acceptable to MAKE-EVENT are acceptable
;; to INSERT-ACTION, and have the same semantics.
;; Returns the token of the event enqueued.

(defun insert-action (time action &rest keys &key &allow-other-keys)
  (insert-event (apply #'make-event
		       time
		       action
		       keys)))
;;
;; REMOVE-EVENT
;;
;;  destructively removes the specified event from the queue
;; Nil is OK to pass as the event to remove.
;;

;; Removes the event itself

(defmethod remove-event ((dead event) &optional (self *the-event-queue*))
  (when dead
    (remove-token *the-token-table* (token dead))
    (setf (in-queue? dead) nil)
    (setf (events self) (delete dead (events self)))))

;; Removes the event with given token

(defmethod remove-event (dead-token &optional (self *the-event-queue*))
  (let ((dead (lookup-token *the-token-table* dead-token)))
    (when dead
      (remove-token *the-token-table* dead-token)
      (setf (in-queue? dead) nil)
      (setf (events self) (delete dead (events self))))))


;; REPLACE-EVENT
;;
;;  Repositions an event in the queue so that the time-ordering
;; is preserved
;;

(defun replace-event (new-and-improved-token new-time 
		      &optional (self *the-event-queue*))
  (let ((new (lookup-token *the-token-table* new-and-improved-token)))
    (when (and new (not (compare-times new-time '= (time-when new))))
      (remove-event new self)
      (setf (time-when new) new-time)
      (insert-event new self))))



;;
;; DEQUEUE-EVENT
;;
;; Removes and returns the first event on the queue, Nil if none exists
;;

(defmethod dequeue-event (self)
  (let ((e (first (events self))))
    (if e (remove-event e self))
    e))

;;
;; FIRST-EVENT
;;
;; Return the first event without dequeueing it. Nil if no event exists
;;

(defmethod first-event (self)
  (first (events self)))

;;
;; PRINT-OBJECT, so that the queue can be printed legibly
;;

(defmethod print-object ((self event-queue) out)
  (format out "<QUEUE:~S>" (events self)))

			    