
;;;;
;;;
;;; The mechanisms in this file attempt to let the user recover from errors.
;;;


(defvar *execution-context* nil)

;;; The forms of the body are evaluated within the supplied execution
;;; context.

(defmacro execution-context (context &body forms)
  `(progn 
     (push ,context *execution-context*)
     (unwind-protect
	 ,(if (> (length forms) 1)
	      `(progn ,@forms)
	    (first forms))
       (pop *execution-context*))))

;;;
;;; RECOVER-FROM-ERROR attempts to do the right thing to clear
;;; the error, and re-enter the simulator.  It guesses what to do
;;; by looking at the execution context.
;;; It understands the following contexts:
;;;
;;;   (:UPDATE <token>)  ->  remove offending process, and restart.
;;;                          If we are within the execution contexts of
;;;                          multiple processes, stop them all.
;;;   (:HANDLER <handler>)  -> restarts (by the time the error
;;;                            ocurred, the handler has already been
;;;                            removed from *handler-queue*)
;;;   (:EVENT <event>)   -> makes sure the offending event has been removed,
;;;                          and restarts.
;;;
;;; Note that even if the recovery is successful, there may still
;;; be problems in the state of the simulator: transactions may have been
;;; cut off in the middle.
;;;


(defun recover-from-error ()
  (cond
   ((not (consp (first *execution-context*)))
    (warn "Don't know how to recover from execution context ~S~%Doing nothing"
	  (first *execution-context*))
    (server-loop))
   
   ((eq (first (first *execution-context*)) :UPDATE)
    (format *terminal-io* "Removing process ~S from system~%"
	    (second (first *execution-context*)))
    (really-stop-process (second (first *execution-context*)))
    ;; It is quite common that arm commands are the ones that blow up.
    ;; So as a precaution, make all arms idle (otherwise they would
    ;; be busy forever, and no other arm commands would work)
    (mapc #'(lambda (truck) (mapc #'arm-is-idle (arms truck)))
	  *the-truck-table*)
    (pop *execution-context*)
    ;; Recover from the context that caused us to go into this
    ;; process.
    (recover-from-error))
   
   ((eq (first (first *execution-context*)) :HANDLER)
    (format *terminal-io* "Continuing past faulty condition handler~%")
    (setf *execution-context* nil)
    (server-loop))

   ((eq (first (first *execution-context*)) :EVENT)
    (format *terminal-io* "Continuing past faulty event~%")
    ;; In the current implementation, the event is removed from
    ;; the queue before it is executed, so no worries.
    (setf *execution-context* nil)
    (server-loop))))
   

