;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; This interactive client allows the user to interact directly with the
;;; server, asynchronously. At any time, the user can type a command command
;;; or control command, and it will be sent to the server.  Whenever
;;; a command or control message comes back from the server, it is immediately
;;; displayed.
;;;
;;; Commands are of the form:
;;;  <c-or-c> <command>, or
;;;  quit/bye/stop/halt/exit/whoa/whoah/desist (disconnects the client
;;;                                             from the simulation)
;;;
;;; Where <c-or-c> specifies whether the command is a CONTROL or COMMAND
;;;   command (got that?).
;;;
;;;   For COMMAND messages, <c-or-c> := command | cmd | com | m
;;;   For CONTROL messages, <c-or-c> := control | cntl | con | ctl | cnt | n
;;;


(defun start-raw-interactive-client (&rest keys)
  (apply #'start-client :client #'raw-interactive-client
	 :allow-other-keys t keys)
  (close-channel *client-truck-channel*))


(defun raw-interactive-client (channel)
  (loop 
      while (print-input channel)
      do
	(if (listen *terminal-io*)
	    (multiple-value-bind (c-or-c cmd)
		(read-command-message *terminal-io*)
	      (cond
	       ((eq c-or-c 'command)
		(send-command channel cmd))
	       ((eq c-or-c 'control)
		(send-control channel cmd))
	       ((eq c-or-c 'quit)
		(send-control channel '(BYE)))
	       (t (cerror "Ignore this command"
			  "Not command or control message")))))))

(defun print-input (channel)
  (let ((msg nil))
    (loop while (control-message-available? channel) do
	  (format t "~%CON: ~S~%raw-truckworld> " 
		  (setf msg (read-control channel))))

    (loop while (command-message-available? channel) do
	  (format t "~%CMD: ~S~%raw-truckworld> " 
		  (read-command channel)))
    (not (or (equal msg '(ERROR HANDSHAKE-ERROR))
	     (equal msg '(BYE))))))


(defun read-command-message (stream)
  (let* ((dir (read stream))
	 (cmd (read stream)))
    (case dir
      ((command cmd com m)          (values 'command cmd))
      ((control cntl con ctl cnt n) (values 'control cmd))
      ((quit bye stop halt exit whoa whoah desist)  (values 'quit 'who-cares))
      (otherwise                  (values 'huh? 'who-cares)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Here's a friendlier interactive interface: 
;;;
;;; Here is a really friendly interactive command processor
;;; It blocks for user input and passes all input through
;;; the command channel to the server.  The command processor then
;;; issues GO commands until the end of the command is reached.
;;;
;;; This means that commands are sent synchronously, to the truck,
;;; and are executed serially.  So you can't be doing two things at once,
;;; like moving both arms.  But the user doesn't have to worry about
;;; synchronizing the client and server.
;;;
;;; At the "truckworld>" prompt, the following commands may be typed:
;;;
;;;   - A truck command (truck-move, arm-move, arm-set, etc)
;;;   - (BYE)   , which will terminate the demo
;;;   - (WAIT nn) , where nn is an amount of time to wait.
;;;                 The command processor will keep issuing (GO nn) commands
;;;                 to the server until the desired time is reached.

;;; Note that these last 2 "commands" are not commands issued to the truck,
;;; but commands that are intercepted by the command processor.  The command
;;; processor then issues the correct control commands to the server to
;;; carry out the user's command.

(defun start-friendly-client (&rest keys)
  (apply #'start-client :client #'friendly-interactive-client
	 :allow-other-keys t keys)
  (close-channel *client-truck-channel*))

(defvar *current-simulator-time* 0)

(defun friendly-interactive-client (channel)
  (setf *current-simulator-time* 0)
  (loop
      with input = nil
      with time = nil
      while (not (member (read-all-messages-until-command-done channel time)
			 '(nil HANDSHAKE-ERROR)))
      do
	(format *terminal-io* "~%truckworld> ")
	(setf input (read *terminal-io*))
	(setf time (process-input channel input))))

(defun process-input (channel input)
  (cond
   ((consp input)
    (case (car input)
      (WAIT   (+ *current-simulator-time* (cadr input)))
      (BYE    (send-control channel input) nil)
      (t      (send-command channel input) nil)))
   (t
    (send-command channel input)
    nil)))

(defvar *interactive-output* t)
  
(defun read-all-messages-until-command-done (channel time &optional 
							  (output *interactive-output*))
  (loop named main
      with commands = nil
      with waiting = nil
      with lasttime = 0
      with token = nil
      with stop = nil
      while (or (not stop) token)
      do
	(loop 
	    while (command-message-available? channel)
	    for msg = (read-command channel) then (read-command channel)
	    do
	      (setf commands (nconc commands (list msg)))
	      (when output
		(format *terminal-io* "COMMAND FROM SERVER: ~S~%" msg)))
	      

	(loop
	    while (control-message-available? channel)
	    for msg = (read-control channel) then (read-control channel)
	    do
	      (when output
		(format *terminal-io* "CONTROL FROM SERVER: ~S~%" msg))
	      (case (car msg)
		(ERROR      (setf lasttime (second msg)) (setf stop t))
		(PROCESS    (setf token (second msg)))
		(TERMINATED (setf token nil))
		(TIME       (setf lasttime (second msg))
			    (setf *current-simulator-time* (second msg))
			    (setf waiting nil)
			    (if (or (eql (cadr msg) time) ; arrived at time
				    (and (null time) ; or command completed
					 (null token)))
				(setf stop t)))
		(BYE	    (setf stop 'bye))))
	
	;; This loop re-checks the command channel, to pick up any
	;; messages that were missed due to communication lag,
	;; so if we are exiting the main loop
	;; because of a command completion, or arrival at a time,
	;; the messages will be detected before exiting.

	(loop 
	    while (command-message-available? channel)
	    for msg = (read-command channel) then (read-command channel)
	    do
	      (setf commands (nconc commands (list msg)))
	      (when output
		(format *terminal-io* "COMMAND FROM SERVER: ~S~%" msg)))

;;;	(format *terminal-io* "T:~S W:~S LT:~S TOK:~S S:~S~%"
;;;		time waiting lasttime token stop)
	(cond
	 ((and token (not waiting))
	  (when output
	    (format *terminal-io* "CONTROL TO SERVER: (GO)~%"))
	  (send-control channel '(GO))
	  (setf waiting t))

	 ((and time (not stop) (not waiting))
	  (when output
	    (format *terminal-io* "CONTROL TO SERVER: (GO ~S)~%" time))
	  (send-control channel `(GO ,time))
	  (setf waiting t)))
	
      finally (return (values (if (eq stop 'BYE) nil lasttime)
			      commands ))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Functional interface for driving an integrated simulation
;;;
;;; This is much like the friendly interactive interface,
;;; except that a command is passed as an argument to EXECUTE-TRUCK-COMMAND
;;; rather than typed in at a prompt.
;;;
;;; To start:
;;;   (start-functional-client ...normal keyword args to initialize-client...)
;;;
;;; To use:
;;;   (execute-truck-command <truck-command>) -> status command-info
;;;
;;;    truck-command: any truck command, or
;;;                  (wait n) where n is a time for the truck to wait until, or
;;;                  (bye)    disconnects the functional client
;;;                           from the simulation
;;;
;;;   status : either a number, representing the time upon completion,
;;;            or a symbol describing an error.
;;;
;;;   command-info : a list of all things passed back over the command channel.
;;;
;;; There is a variable *INTEGRATED-OUTPUT* that controls whether or not
;;; the interface prints out the communication between client and server.
;;; It defaults to T.
;;;


(defvar *functional-channel* nil)


(defun start-functional-client (&rest keys)
  (apply #'start-client :client #'functional-interface
	 :allow-other-keys t keys))


;;;
;;; Supply FUNCTIONAL-INTERFACE as the client function to START-SIMULATOR
;;; or START-CLIENT
;;;

(defun functional-interface (channel)
  (setf *functional-channel* channel))

;;;
;;; EXECUTE-TRUCK-COMMAND command -> status command-info
;;;
;;; After starting the simulator, using FUNCTIONAL-INTERFACE as
;;; a client function, use this to send commands to the simulator
;;; The command will be executed to completion, and 2 values returned:
;;;
;;;   STATUS : If there were no errors, the current simulator time
;;;            if errors, the symbol describing the error
;;;   COMMAND-INFO : a list of all messages returned on the command channel
;;;

(defun execute-truck-command (command)
  (read-all-messages-until-command-done *functional-channel*
					(process-input *functional-channel*
						       command)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; The tour-guide interface (runs only in the integrated simulator)
;;;

(defvar *tour-file*)

(defun start-tour (&rest keys &key pause world &allow-other-keys)
  (apply #'start-simulator :client #'functional-interface :allow-other-keys
	 keys)
  (setf *tour-file* (open (make-pathname :type "tour" :defaults *current-world-file*)))
  (run-tour :pause pause)
  (close *tour-file*))

(defun run-tour (&key pause)
  (do* ((input (read *tour-file* nil) (read *tour-file* nil)))
      ((null input)  nil)
    (cond
     ((listp input)
      (format *terminal-io* "~%truckworld> ~S~%~%" input)
      (execute-truck-command input))
     ((stringp input)
      (format *terminal-io* input))
     ((or pause (eq input 'pause))
      (format *terminal-io* "--- Pausing: press a key ---")
      (read-char *terminal-io*))
     (t
      (format *terminal-io* "WARNING: strangeness in tour file: ~S~%" input)))))

	
