;;;
;;; INTEGRATED.LISP
;;;
;;; For the scope of this file, "server" refers to the code that would
;;; normally execute in it's own process as the truckworld simulator.
;;; "Client" is the code that sits across the IPC connection, feeding
;;; truck commands to the server.
;;;
;;; This file is basically a patch file that takes the true multiprocess
;;; client/server simulator, and hacks it so that the client and server can run
;;; in the same lisp process.
;;;
;;; This involves 3 things: 
;;;   - initiaizing the "client" and its truck, along with the server
;;;   - bypassing the IPC implementation of the CHANNEL object
;;;   - Allowing control to pass back and forth between client and server
;;;     at the proper times.
;;;     


;;; With the "client" and "server" both running in the same process,
;;; there will be 2 channel objects in the system.  One installed in the
;;; truck in the world, and one for the client to communicate with the truck.
;;; The channels have to know who they are so that they can take actions
;;; specific to the side they are on.  This will be done by comparing
;;; the channel in question to *client-truck-channel*, which is set to
;;; the client's channel object.  If equal, we're dealing with the
;;; client channel, if not, assume the server channel, and it is connected
;;; to the client's channel via the io connection

;;; These variables simulate io streams that go from client to server,
;;; and vica versa.  They replace the UN*X socket connections

(defvar *comm-to-client* nil)
(defvar *comm-to-server* nil)
(defvar *give-control-to-server* nil)

;;; INstead of writing to the stream of the channel, write to the appropriate
;;; communication queue.  Whenever the client talks to the server,
;;; it should wait for the server's response.  When the server has someting
;;; to say, it will return to the client.  Note that this mechanism
;;; will not be able to mimic the real-time mode of the simulator.

(defmethod send-command-or-control :around ((self channel) message c-or-c)
  (cond
   ((eq self *client-truck-channel*)
    (setf *comm-to-server* (nconc *comm-to-server*
				  (list (cons c-or-c message))))
    (if *give-control-to-server* (server-loop)))
   (t
    (setf *comm-to-client* (nconc *comm-to-client*
				  (list (cons c-or-c message)))))))

;;;
;;; Likewise, read from the proper io line
;;;

(defmethod read-messages :around ((self channel))
  (let ((io-stream (if (eq self *client-truck-channel*)
		       '*comm-to-client*
		     '*comm-to-server*)))
    (loop
      (when (null (symbol-value io-stream))
	(return))
      (let ((message (pop (symbol-value io-stream))))
	(cond ((eql (first message) 'COMMAND)
	       (setf (command-messages self) (append (command-messages self)
						     (list (rest message)))))
	      ((eql (first message) 'CONTROL)
	       (setf (control-messages self) (append (control-messages self)
						     (list (rest message)))))
	      (t
	       (error "malformed message received: ~d" message)))))))


;;;
;;; This is the normal server loop, modified to pass control back to
;;; the client when the server communicates with it.
;;;

(defun server-loop ()
  (loop 
      with earliest
      for now-time      = (real-time)
      for all-advancing = *the-truck-table*
      until *stop-server* do
	
	(setf earliest now-time)
	
	(service-new-client-requests)
	
	(loop for the-truck in *the-truck-table* do
	      
	      (when (and (null (advance-clock the-truck))
			 (command-message-available? the-truck))
		(execute-command the-truck (read-command the-truck)))
	      
	      (when (and (null (advance-clock the-truck))
			 (control-message-available? the-truck))
		(execute-control the-truck (read-control the-truck)))
      
	      ;; If the truck's controller wants to advance the simulator,
	      ;;  check that time against the other times for the earliest
	      ;;  time to advance to.
	      ;; If not, flag the fact that we are waiting for somebody
	      ;;  to advance.
	      
	      (cond
	       ((null (advance-clock the-truck))
		(setf all-advancing nil))
	       ((typep (advance-clock the-truck) 'sim-time)
		(if (or (null earliest)
			(compare-times (advance-clock the-truck) '< earliest))
		    (setf earliest (advance-clock the-truck))))))
	
	(when (or (and all-advancing *the-truck-table*)
		  (and now-time (compare-times now-time '> (actual-time))))
	  (format *terminal-io* "Advancing simulator to ~S~%" earliest) ; *DEBUG*
	  (let ((boring-world? (advance-simulator earliest)))
	    
	    ;; After advancing the simulator, if any truck has had it's
	    ;; advance-counter reset, or a truck has advanced up to, or past
	    ;; it's advance-counter time, signal that truck with a (TIME)
	    ;; message on its control channel
	    ;; Alternatively, if advancing the simulator would result
	    ;; in nothing interesting, (all trucks say (GO), but there are no
	    ;; events in the queue to go to), then all trucks will be signaled
	    
	    (dolist (truck *the-truck-table*)
	      (let ((ac (advance-clock truck)))
		(when (or boring-world?
			  (null ac)
			  (and (typep ac 'sim-time)
			       (compare-times ac '<= (actual-time *the-world*))))
		  (setf (advance-clock truck) nil)
		  (send-control truck `(TIME ,(actual-time *the-world*))))))))

	;;; After the simulator has done it's thing, if there is anything
	;;; to report to the client, return to the client so it can hear
	;;; the report
	
	(when *comm-to-client*
	  (return))))



;;;
;;; When the single-process simulator is started, it needs to initially
;;; accept one client connection over the simulated io channel.
;;; Here's what we do: If there aren't any trucks in the world yet,
;;; assume the client will try to connect, so automatically create
;;; a channel for the server.  The client should have been initialized
;;; before now, so there will be data in the "io stream" already waiting
;;; to be read.
;;;

(defun accept-client-connection ()
  (if (null *the-truck-table*)
      (make-instance 'channel)))



;;;
;;; This version of initialize-simulator initializes the server
;;; and the client.  As such, it also accepts the key arguments that
;;; initialize-client does.  However, the :PORT and :HOST arguments
;;; should NOT be supplied.
;;;
;;; Note that INITIALIZE-CLIENT creates a channel to the server, and
;;; sends its truck's information over it.  The first call to
;;; ACCEPT-CLIENT-CONNECTION will receive this info, create a truck,
;;; and then never do it again (as long as the simulation lasts).
;;;

(defun initialize-simulator (&rest key-args 
			     &key world
			     &allow-other-keys)
  
  (format *terminal-io* "Opening connection to port~%")
  (initialize-ipc nil)
  (setf *comm-to-client* nil)
  (setf *comm-to-server* nil)
  (setf *give-control-to-server* nil)
  (format *terminal-io* "Initializing control structures~%")
  (initialize-control)
  (format *terminal-io* "Initializing realtime system~%")
  (initialize-realtime nil nil)
  (format *terminal-io* "Initializing world~%")
  (setf *stop-server* nil)
  (initialize-domain world)
  (format *terminal-io* "Initializing robot truck~%")
  (apply #'initialize-client key-args))

;;;
;;; This is just like the normal START-SIMULATOR, except that you specify
;;; a function that takes 1 argument: the client's channel object.
;;; After the simulator is initialized, control will be passed to this
;;; function with the channel object assigned to the client.  This function
;;; may then issue commands to the simulator across this channel.

(defvar *client-function* nil)

(defun start-simulator (&rest keys &key client &allow-other-keys)
  (setf *client-function* client)
  (apply #'initialize-simulator :allow-other-keys t keys))


;;;
;;; This provides a raw, asynchronous interface to the
;;; simulator
;;;

(defun start-raw-interactive-simulator (&rest keys)
  (apply #'start-simulator :client #'raw-interactive-client keys)
  (terminate-simulator))

;;;
;;; This starts an integrated simulation using the friendly interactive client
;;; interface.
;;;

(defun start-interactive-simulator (&rest keys)
  (apply #'start-simulator :client #'friendly-interactive-client keys)
  (terminate-simulator))

;;;
;;; This starts an integrated simulation using the functional client
;;; interface.
;;;

(defun start-functional-simulator (&rest keys)
  (apply #'start-simulator :client #'functional-interface keys))

;;;
;;; Recovering from an error in integrated mode first requires that
;;; the server be allowed to clean up in the server-loop, then
;;; when it returns, manually pass control back to the client.
;;;

(defun recover-from-error ()
  (cond
   ((not (consp (first *execution-context*)))
    (warn "Don't know how to recover from execution context ~S~%Doing nothing"
	  (first *execution-context*))
    (restart-client))
   
   ((eq (first (first *execution-context*)) :UPDATE)
    (format *terminal-io* "Removing process ~S from system~%"
	    (second (first *execution-context*)))
    (really-stop-process (second (first *execution-context*)))
    ;; It is quite common that arm commands are the ones that blow up.
    ;; So as a precaution, make all arms idle (otherwise they would
    ;; be busy forever, and no other arm commands would work)
    (mapc #'(lambda (truck) (mapc #'arm-is-idle (arms truck)))
	  *the-truck-table*)
    (pop *execution-context*)
    ;; Recover from the context that caused us to go into this
    ;; process.
    (recover-from-error))
   
   ((eq (first (first *execution-context*)) :HANDLER)
    (format *terminal-io* "Continuing past faulty condition handler~%")
    (setf *execution-context* nil)
    (restart-client))

   ((eq (first (first *execution-context*)) :EVENT)
    (format *terminal-io* "Continuing past faulty event~%")
    ;; In the current implementation, the event is removed from
    ;; the queue before it is executed, so no worries.
    (setf *execution-context* nil)
    (restart-client))))

(defun restart-client ()
  (setf *comm-to-client* (nconc *comm-to-client*
				`((CONTROL TIME ,(actual-time)))))
  (funcall *client-function* *client-truck-channel*))
