
; --------------------------------------------------------------------
; ** Objects of type ARM **
; -------------------------
; This file defines the following things:
;
; Object builders:
;  (SI:MAKE-ARM-GENERIC id kind capacity visibles)
;
; This file makes reference to code in - LOWLEVEL LISP
;                                        OPERATIONS LISP
; --------------------------------------------------------------------
; ** Simulator support for ROBOT ARMS
; -----------------------------------
; Robot arms are used to manipulate objects in the simulated world.
;  Arms can GRASP and UNGRASP objects of type THINGOID (note that
;  CONTAINERS and VESSELS are generally of type THINGOID).  To GRASP
;  something, the arm must have enough unused capacity to handle the
;  THINGOIDS bigness, and it must already be holding an object of
;  the class of every tool needed by the THINGOID.  That is, the
;  thing to be grasped cannot be too big and the arm must already be
;  holding any necessary tools.
;
; Arms can also move around.  At all time an arm has both an environment 
; that it is in and a thing that it is at.  For
;  example, an arm may be in environment box45 at thing vessel27.  With
;  this in mind consider the different forms of move commands:
; (ARM-MOVE self 'FOLDED) - If the arm is not holding anything, then
;                            its environment becomes undefined and it is at
;                            thing 'FOLDED.
; (ARM-MOVE self 'INSIDE) - If the arm is at a parcel (ie. container
;                            or vessel) its environment becomes the parcel
;                            and it is at some thing in the parcel
;                            it there are any.
; (ARM-MOVE self 'OUTSIDE) - If the environment of the current environment
;                             is a parcel then the arms environment becomes
;                             that environment and the arm is at the
;                             object that was the previous environment.
; (ARM-MOVE self thing-id) - If the current environment holds an object
;                             with id 'thing-id, the are is set to be
;                             at that object.  The environment is not
;                             changed.
; (ARM-MOVE self object) - If the object is a parcel, then the
;                           environment of the arm becomes the object and
;                           the arm is at an object in it if there
;                           is one.
;
; Arms can only GRASP the object that they are at.  Arms can UNGRASP
;  an object inside any parcel (except a VESSEL) and the arm ends
;  up at the object UNGRASPED.
; NOTE: An arm cannot grasp an object if another arm is also at the
;  same object.
;
; Arms can pour the liquid from one VESSEL into another if they have
;  GRASPed the VESSEL to be poured and they are at, or in, the VESSEL to be
;  poured into.  The opposite of pouring is ladling.  An arm can
;  ladle the liquid from one VESSEL into another if the VESSEL to be
;  filled has been GRASPed and the arm is at, or in, the VESSEL containing
;  the liquid.  (See the file "packages.t" for a description of
;  liquids.)
;
; Arms can EXAMINE an object if they are at it or have GRASPed it.
;
; Robot arms are also ENVIRONMENTs and certain things can be visible
;  outside of the arm when they are being held.  (ie. radioactivity
;  etc.)  This property of arms is just like that of parcels.
; OPERATIONS AND ERRORS SUPPORTED BY ARMS:
;  (ARM-GRASP   arm thing)  => ARM-NOT-AT ARM-TOO-FULL ARM-CANT-GRIP
;                              ARM-INTERFERENCE
;  (ARM-UNGRASP arm thing)  => ARM-NOT-AT ARM-NOT-HOLDING
;                              CONATINER-FULL
;  (ARM-POUR    arm vessel) => ARM-NOT-HOLDING
;  (ARM-LADLE   arm vessel) => ARM-NOT-HOLDING ARM-NOT-AT
;  (ARM-MOVE    arm place)  => ARM-CANT-MOVE ARM-CANT-FIND
;  (ARM-EXAMINE arm thing)  => (LIST OF SENSORY DATA)
;                              ARM-NOT-AT
;  (ARM-TOGGLE  arm thing)  => ARM-NOT-AT ARM-CANT-TOGGLE
; --------------------------------------------------------------------
; * Generic Arm Constructor
; -------------------------
; This function makes a single object of type ARM.  The arguments
;  determine the properties of the object in the simulation.  Generic
;  ARMS are quite active within the simulation and are also of type
;  ENVIRONMENT.  Arms are designed primarily for use in robot trucks.
;
; (SI:MAKE-ARM-GENERIC id kind capacity visibles)
;  -    id: This should be a globally unique symbol differentiating
;            the object from all others.
;  -  kind: A symbol or list of symbols describing the classes that
;            the arm belongs to.  The first symbol in this argument
;            is known as the objects KIND-ID and is used for display
;            purposes.
;  - capacity: This argument is an integer giving the pseudo-size
;            that the arm is capable of GRASPing.  All things held
;            must have bignesses that sum to less than this.
;  - visibles: A list of object classes that this arm is transparent
;            to.
; These functions enforce the fact that an arm cannot grasp an object
;  if another arm is at it.

(let ((arm-at-pairs nil))
 
  (defun arm-is-now-at (arm thing)
    (let ((pair (assoc arm arm-at-pairs :test #'eq)))
      (cond ((null pair)
             (setf arm-at-pairs (cons (cons arm thing) arm-at-pairs)))
            (t 
             (setf (cdr pair) thing)))))
 
  (defun arm-can-grasp? (arm thing)
    (not (some #'(lambda (x) 
                   (and (eq (cdr x) thing) (not (eq (car x) arm))))
               arm-at-pairs)))
  
  )

(defun arm-still-at? (arm thing)
  (or (null thing)
      (symbolp thing)
      (let ((env (environment arm)))
        (and (class? env 'container)
             (member thing (holdings env) :test #'eq)))))


(defun si.make-arm-generic (id kind capacity visibles clumsiness)
  (let ((new-thing (make-instance 'arm
                                  :id id
                                  :kind kind
                                  :capacity capacity
                                  :visibles visibles
                                  :clumsiness clumsiness)))
    (setf (space-empty new-thing) capacity)
    (setf (thing-at new-thing) 'folded)
    (setf (bigness new-thing) 0)   ; (for gross-bigness)
    (if (symbolp kind)
      (setf (kind new-thing) (list kind)))
    new-thing))
