
; --------------------------------------------------------------------
; ** Objects of type ARM **
; -------------------------
; This file defines the following things:
;
; Object builders:
;  (SI:MAKE-ARM-GENERIC id kind capacity visibles)
;
; This file makes reference to code in - LOWLEVEL LISP
;                                        OPERATIONS LISP
; --------------------------------------------------------------------
; ** Simulator support for ROBOT ARMS
; -----------------------------------
; Robot arms are used to manipulate objects in the simulated world.
;  Arms can GRASP and UNGRASP objects of type THINGOID (note that
;  CONTAINERS and VESSELS are generally of type THINGOID).  To GRASP
;  something, the arm must have enough unused capacity to handle the
;  THINGOIDS bigness, and it must already be holding an object of
;  the class of every tool needed by the THINGOID.  That is, the
;  thing to be grasped cannot be too big and the arm must already be
;  holding any necessary tools.
;
; Arms can also move around.  At all time an arm has both an environment 
; that it is in and a thing that it is at.  For
;  example, an arm may be in environment box45 at thing vessel27.  With
;  this in mind consider the different forms of move commands:
; (ARM-MOVE self 'FOLDED) - If the arm is not holding anything, then
;                            its environment becomes undefined and it is at
;                            thing 'FOLDED.
; (ARM-MOVE self 'INSIDE) - If the arm is at a parcel (ie. container
;                            or vessel) its environment becomes the parcel
;                            and it is at some thing in the parcel
;                            it there are any.
; (ARM-MOVE self 'OUTSIDE) - If the environment of the current environment
;                             is a parcel then the arms environment becomes
;                             that environment and the arm is at the
;                             object that was the previous environment.
; (ARM-MOVE self thing-id) - If the current environment holds an object
;                             with id 'thing-id, the are is set to be
;                             at that object.  The environment is not
;                             changed.
; (ARM-MOVE self object) - If the object is a parcel, then the
;                           environment of the arm becomes the object and
;                           the arm is at an object in it if there
;                           is one.
;
; Arms can only GRASP the object that they are at.  Arms can UNGRASP
;  an object inside any parcel (except a VESSEL) and the arm ends
;  up at the object UNGRASPED.
; NOTE: An arm cannot grasp an object if another arm is also at the
;  same object.
;
; Arms can pour the liquid from one VESSEL into another if they have
;  GRASPed the VESSEL to be poured and they are at, or in, the VESSEL to be
;  poured into.  The opposite of pouring is ladling.  An arm can
;  ladle the liquid from one VESSEL into another if the VESSEL to be
;  filled has been GRASPed and the arm is at, or in, the VESSEL containing
;  the liquid.  (See the file "packages.t" for a description of
;  liquids.)
;
; Arms can EXAMINE an object if they are at it or have GRASPed it.
;
; Robot arms are also ENVIRONMENTs and certain things can be visible
;  outside of the arm when they are being held.  (ie. radioactivity
;  etc.)  This property of arms is just like that of parcels.
; OPERATIONS AND ERRORS SUPPORTED BY ARMS:
;  (ARM-GRASP   arm thing)  => ARM-NOT-AT ARM-TOO-FULL ARM-CANT-GRIP
;                              ARM-INTERFERENCE
;  (ARM-UNGRASP arm thing)  => ARM-NOT-AT ARM-NOT-HOLDING
;                              CONATINER-FULL
;  (ARM-POUR    arm vessel) => ARM-NOT-HOLDING
;  (ARM-LADLE   arm vessel) => ARM-NOT-HOLDING ARM-NOT-AT
;  (ARM-MOVE    arm place)  => ARM-CANT-MOVE ARM-CANT-FIND
;  (ARM-EXAMINE arm thing)  => (LIST OF SENSORY DATA)
;                              ARM-NOT-AT
;  (ARM-TOGGLE  arm thing)  => ARM-NOT-AT ARM-CANT-TOGGLE
; --------------------------------------------------------------------
; * Generic Arm Constructor
; -------------------------
; This function makes a single object of type ARM.  The arguments
;  determine the properties of the object in the simulation.  Generic
;  ARMS are quite active within the simulation and are also of type
;  ENVIRONMENT.  Arms are designed primarily for use in robot trucks.
;
; (SI:MAKE-ARM-GENERIC id kind capacity visibles)
;  -    id: This should be a globally unique symbol differentiating
;            the object from all others.
;  -  kind: A symbol or list of symbols describing the classes that
;            the arm belongs to.  The first symbol in this argument
;            is known as the objects KIND-ID and is used for display
;            purposes.
;  - capacity: This argument is an integer giving the pseudo-size
;            that the arm is capable of GRASPing.  All things held
;            must have bignesses that sum to less than this.
;  - visibles: A list of object classes that this arm is transparent
;            to.
; These functions enforce the fact that an arm cannot grasp an object
;  if another arm is at it.


(defclass arm
  (container)
  ((clumsiness :accessor clumsiness
               :initarg :clumsiness)
   (thing-at  :accessor thing-at
              :initform 'folded)
   (arm-anchor :accessor arm-anchor
           :initform *undefined-object*
           :type environment)
  ))

; ** Special ENVIRONMENT type operations

(defmethod bumpiness ((self arm)) 
  (bumpiness (arm-anchor self)))

;;; ** Special ARM type operations


(defmethod arm-move ((self arm) (pos-whatsit symbol))
  (let ((valid-motion t)
        (dm-message '()))
    (cond ((and (eq pos-whatsit 'folded)                      ; fold the arm
                (null (holdings self)))
           (setf (environment self) *undefined-object*)
           (setf (thing-at self) 'folded)
           (setf dm-message 'folded))
          ((and (eq pos-whatsit 'inside)                      ; put it inside something
                (class? (thing-at self) 'parcel))
           (setf (environment self) (thing-at self))
           (setf (thing-at self)
                 (let ((outside-things (holdings (environment self))))
                   (if (neq (car outside-things) (arm-anchor self))
                     (car outside-things)
                     (cadr outside-things))))
           (setf dm-message 'inside))
          ((and (eq pos-whatsit 'outside)                     ; move it outside something
                (class? (environment (environment self))
                        'parcel))
           (setf (thing-at self) (environment self))
           (setf (environment self) (environment (thing-at self)))
           (setf dm-message (unique-id (thing-at self))))
          ((and (neq pos-whatsit 'folded)
                (class? (environment self) 'parcel))
           (let ((thing (get-if-holds-id (environment self) pos-whatsit)))
             (cond (thing (setf (thing-at self) thing)
                          (setf dm-message
                                (unique-id (thing-at self))))
                   (t (setf valid-motion '())
                      (post-hardware-error 'arm-cant-find)))))
          (t (setf valid-motion '())
             (post-hardware-error 'arm-cant-move)))
    (if (and (displayer self) valid-motion)
      (cond ((null dm-message)
             (dm-mnes (displayer self) (id self) (unique-id (environment self))))
            (t (dm-move (displayer self) (id self) dm-message))))
    (cond (valid-motion (arm-is-now-at self (thing-at self)) t)
          (t '()))))

(defmethod arm-move ((Self arm) (pos-whatsit parcel)) ;; don't we need to do some checking here
  (let ((valid-motion t)
        (dm-message '()))
    (setf (environment self) pos-whatsit)
    (setf (thing-at self)
          (let ((outside-things (holdings (environment self))))
            (if (neq (car outside-things) (arm-anchor self))
              (car outside-things)
              (cadr outside-things))))
    (if (thing-at self)
      (setf dm-message (unique-id (thing-at self))))
    (if (and (displayer self) valid-motion)
      (cond ((null dm-message)
             (dm-mnes (displayer self) (id self) (unique-id (environment self))))
            (t (dm-move (displayer self) (id self) dm-message))))
    (cond (valid-motion (arm-is-now-at self (thing-at self)) t)
          (t '()))))

(defmethod arm-move ((Self arm) pos-whatsit)
  (let ((valid-motion '())
        (dm-message '()))
    (post-hardware-error 'arm-cant-move)
    (if (and (displayer self) valid-motion)
      (cond ((null dm-message)
             (dm-mnes (displayer self) (id self) (unique-id (environment self))))
            (t (dm-move (displayer self) (id self) dm-message))))
    (cond (valid-motion (arm-is-now-at self (thing-at self)) t)
          (t '()))))


(defmethod arm-grasp ((self arm) thing-id)
  (cond ((not (arm-still-at? self (thing-at self)))
         (post-hardware-error 'arm-not-at))
        ((not (and (thing-at self)
                   (class? (environment self) 'container)
                   (eq thing-id (unique-id (thing-at self)))))
         (post-hardware-error 'arm-not-at))
        ((not (<= (bigness (thing-at self)) (space-empty self)))
         (post-hardware-error 'arm-too-full))
        ((not (obj-list-holds-classes? (holdings self)
                                       (tool-needs (thing-at self))))
         (post-hardware-error 'arm-cant-grasp))
        ((not (arm-can-grasp? self (thing-at self)))
         (post-hardware-error 'arm-interference))
        ((random-chance? (clumsiness self))
         (post-hardware-error 'arm-dropped))
        (t (take-out (environment self) (thing-at self))
           (setf (space-empty self)
                 (- (space-empty self) (bigness (thing-at self))))
           (setf (holdings self)
                 (obj-list-put (holdings self) (thing-at self)))
           (setf (environment (thing-at self)) self)
           (if (displayer self)
             (dm-add (displayer self) (id self) (thing-at self)))
           (setf (thing-at self) '())
           (arm-is-now-at self (thing-at self))
           t)))

(defmethod arm-ungrasp ((self arm) thing-id)
  (if (class? (environment self) 'container)
    (let ((thing (obj-list-get-id (holdings self) thing-id)))
      (cond ((null thing)
             (post-hardware-error 'arm-not-holding))
            ((not (will-hold? (environment self) thing))
             (post-hardware-error 'container-full))
            (t (setf (space-empty self)
                     (+ (space-empty self) (bigness thing)))
               (setf (holdings self)
                     (obj-list-del-id! (holdings self)
                                       thing-id))
               (cond ((displayer self) 
                      (if (thing-at self)
                        (dm-mnes (displayer self)
                                 (id self)
                                 (unique-id (environment self))))
                      (dm-del (displayer self)
                              (id self)
                              thing)))
               (put-in (environment self) thing)
               (setf (thing-at self) thing)
               (arm-is-now-at self (thing-at self))
               t)))
    '()))


(defmethod arm-pour ((self arm) thing-id)
  (let ((sink (if (and (arm-still-at? self (thing-at self))
                       (class? (thing-at self) 'vessel))
                (thing-at self)
                (if (and (null (thing-at self))
                         (class? (environment self) 'vessel))
                  (environment self)
                  '())))
        (source (obj-list-get-id (holdings self) thing-id)))
    (cond ((null source)
           (post-hardware-error 'arm-not-holding))
          ((and (null sink) 
                (class? source 'vessel))
           (pour-out source (space-full source))
           t)
          ((and source
                sink
                (class? source 'vessel)
                (or (empty? sink)
                    (eq (liquid-held source)
                        (liquid-held sink))))
           (let ((amount (min (space-full source)
                                     (space-empty sink))))
             (setf (liquid-held sink) (liquid-held source))
             (pour-in sink amount)
             (pour-out source amount)
             t)))))

(defmethod arm-ladle ((self arm) thing-id)
  (let ((source (if (and (arm-still-at? self (thing-at self))
                         (class? (thing-at self) 'vessel))
                  (thing-at self)
                  (if (and (null (thing-at self))
                           (class? (environment self) 'vessel))
                    (environment self)
                    '())))
        (sink (obj-list-get-id (holdings self) thing-id)))
    (cond ((null sink)
           (post-hardware-error 'arm-not-holding))
          ((null source)
           (post-hardware-error 'arm-not-at))
          ((and (class? sink 'vessel)
                (or (empty? sink)
                    (eq (liquid-held source)
                        (liquid-held sink))))
           (let ((amount (min (space-full source)
                                     (space-empty sink))))
             (setf (liquid-held sink) (liquid-held source))
             (pour-in sink amount)
             (pour-out source amount)
             t))
          (t t))))

(defmethod arm-examine ((self arm) thing-id)
  (cond ((and (thing-at self)
              (arm-still-at? self (thing-at self))
              (eq thing-id (unique-id (thing-at self))))
         (examine (thing-at self)))
        (t (let ((thing (obj-list-get-id (holdings self)
                                         thing-id)))
             (if thing
               (examine thing)
               (post-hardware-error 'arm-not-at))))))

(defmethod arm-toggle ((self arm) thing-id)
  (cond ((and (thing-at self)
              (arm-still-at? self (thing-at self))
              (eq thing-id (unique-id (thing-at self))))
         (if (toggle (thing-at self))
           t
           (post-hardware-error 'arm-cant-toggle)))
        (t (let ((thing (obj-list-get-id (holdings self)
                                         thing-id)))
             (cond ((null thing)
                    (post-hardware-error 'arm-not-holding))
                   ((toggle thing) t)
                   (t (post-hardware-error 'arm-cant-toggle)))))))

(Defmethod arm-folded? ((self arm)) 
  (eq (thing-at self) 'folded))

                     
;;; ** Operations needed for neatness

(defmethod (setf displayer) :after (new-displayer (self arm)) 
  (if (displayer self)
    (cond ((eq (thing-at self) 'folded)
           (dm-empty (displayer self) (id self))
           (mapc #'(lambda (x)
                     (dm-add (displayer self) (id self) x))
                 (holdings self))
           (dm-move (displayer self) (id self) 'folded))
          (t (cerror "To continue" "(#{~a}) Cannot set displayer unless FOLDED"
                    (id self))))))

(defmethod show ((self arm))
  (format t "Name:    ~a~&Classes:  ~a~&" (id self) my-classes)
  (format t
          "Environment:  ~a <~a> - Space: ~a - Holding: ~a~&"
          (environment self)
          (thing-at self)
          (space-empty self)
          (holdings self))
  (format t "Let-See: ~a~&" (visibles self))
  t)

(defmethod show-contents ((self arm))
  (format t "~a at ~a - ~a" 
          (environment self) (thing-at self) (holdings self)))

