;************************************************************************
;  HEADING.LISP

(defclass heading-disp  ()
  ((display-window :accessor display-window :initarg :display-window)
   (direction :accessor direction :initform 'n)
   (center :accessor center :initarg :center)
   (radius :accessor radius :initarg :radius)))

(defun make-heading-disp (disp-window)
  (let* ((h (height disp-window))
		 (w (width disp-window))
		 (the-center (make-position (truncate (/ h 2))
									(truncate (/ w 2))))
		 (the-radius (- (truncate (min h w) 2) 10))
		 (the-displayer (make-instance 'heading-disp
									   :display-window disp-window
									   :center the-center
									   :radius the-radius)))
	(dm-refresh the-displayer)
	the-displayer))

(defun heading-draw-compass (heading-disp)
  (let ((disp-window (display-window heading-disp))
		(c (center heading-disp))
		(r (radius heading-disp))
		(d (direction heading-disp)))
	(let ((center-x (truncate (x-coord c) 2))
		  (center-y (truncate (y-coord c) 2)))
	  (disp.with-clip-window disp-window
		   (disp.with-font (disp.small-font)
				(disp.center-text "N" (make-position center-x 
													 (+ center-y (- -6 r))))
				(disp.center-text "S" (make-position center-x 
													 (+ center-y (+ 6 r))))
				(disp.center-text "E" (make-position (+ center-x (+ 6 r)) 
													 center-y))
				(disp.center-text "W" (make-position (+ center-x (- -6 r))  
													 center-y)))
		   (disp.draw-circle c (+ r 1))
		   (disp.draw-circle c 2)
		   (comp.draw-arrow c r d)))))

(defmethod dm-set ((self heading-disp) x new-direction)
  (declare (ignore x))
  (cond 
	((si.map-direction? new-direction)
	 (disp.with-erasure 
		(heading-draw-arrow self))
	 (setf (direction self) new-direction)
	 (heading-draw-arrow self))
	(t nil)))

(defmethod dm-refresh ((self heading-disp))
  (disp.clear-rectangle (display-window self))
  (heading-draw-compass-shell self)
  (heading-draw-arrow self))

(defun heading-draw-compass-shell (heading-disp)
  (let ((disp-window (display-window heading-disp))
		(c (center heading-disp))
		(r (radius heading-disp))
		(d (direction heading-disp)))
	(let ((center-x (x-coord c))
		  (center-y (y-coord c)))
	  (disp.with-clip-window disp-window
		   (disp.with-font (disp.small-font)
				(disp.text-at "N" (make-position (- center-x 3)
												 (- center-y (+ r 3))))
				(disp.text-at "S" (make-position (- center-x 3)
												 (+ center-y r 8)))
				(disp.text-at "E" (make-position (+ center-x r 3)
												 (+ center-y 3)))
				(disp.text-at "W" (make-position (- center-x (+ r 8))
												 (+ center-y 3))))
		   (disp.draw-circle c (+ r 1))
		   (disp.draw-circle c 2)))))

(defun heading-draw-arrow (self)
  (disp.with-clip-window (display-window self)
	   (comp.draw-arrow (center self) (radius self) (direction self))))

;**************************************************************************

(defun si.map-direction? (direction)
  (member direction '(n s w e ne nw se sw) :test #'eq))

(defun si.map-direction-opposite (direction)
  (cond ((eq direction 'e) 'w)
        ((eq direction 'ne) 'sw)
        ((eq direction 'n) 's)
        ((eq direction 'nw) 'se)
        ((eq direction 'w) 'e)
        ((eq direction 'sw) 'ne)
        ((eq direction 's) 'n)
        ((eq direction 'se) 'nw)
        (t nil)))

(defun si.map-direction-left (direction)
  (cond ((eq direction 'e) 'ne)
        ((eq direction 'ne) 'n)
        ((eq direction 'n) 'nw)
        ((eq direction 'nw) 'w)
        ((eq direction 'w) 'sw)
        ((eq direction 'sw) 's)
        ((eq direction 's) 'se)
        ((eq direction 'se) 'e)
        (t nil)))

(defun si.map-direction-right (direction)
  (cond ((eq direction 'e) 'se)
        ((eq direction 'ne) 'e)
        ((eq direction 'n) 'ne)
        ((eq direction 'nw) 'n)
        ((eq direction 'w) 'nw)
        ((eq direction 'sw) 'w)
        ((eq direction 's) 'sw)
        ((eq direction 'se) 's)
        (t nil)))

(defun si.map-direction-index (direction)
  (cond ((eq direction 'e) 0)
        ((eq direction 'ne) 1)
        ((eq direction 'n) 2)
        ((eq direction 'nw) 3)
        ((eq direction 'w) 4)
        ((eq direction 'sw) 5)
        ((eq direction 's) 6)
        ((eq direction 'se) 7)
        (t nil)))

(let ((x-direction (list (cons 'n 0.0)
                         (cons 'w -1.0)
                         (cons 's 0.0)
                         (cons 'e 1.0)
                         (cons 'nw -0.707)
                         (cons 'ne 0.707)
                         (cons 'sw -0.707)
                         (cons 'se 0.707)))
      (y-direction (list (cons 'n -1.0)
                         (cons 'w 0.0)
                         (cons 's 1.0)
                         (cons 'e 0.0)
                         (cons 'nw -0.707)
                         (cons 'ne -0.707)
                         (cons 'sw 0.707)
                         (cons 'se 0.707))))
  
  (defun comp.draw-line-in-dir (origin length direction)
    (let* ((tip-loc-x (truncate (* (float length)
                                   (cdr (assoc direction
                                               x-direction
                                               :test
                                               #'eq)))))
           (tip-loc-y (truncate (* (float length)
                                   (cdr (assoc direction
                                               y-direction
                                               :test
                                               #'eq)))))
           (tip (make-position (+ tip-loc-x (x-coord origin))
                               (+ tip-loc-y (y-coord origin)))))
      (disp.draw-line origin tip)
      tip))

  )

(defun comp.draw-arrow (origin length direction)
  (let* ((tip (comp.draw-line-in-dir origin length direction))
         (left (comp.draw-line-in-dir tip
                                      (* (float length) -0.8)
                                      (si.map-direction-left direction)))
         (right (comp.draw-line-in-dir tip
                                       (* (float length) -0.8)
                                       (si.map-direction-right direction))))
    (disp.fill-triangle tip left right)
    (comp.draw-line-in-dir origin
                           (* (float length) 0.8)
                           (si.map-direction-opposite direction))))
