/**************************************************************************/
/* communication.c                                              /\/\      */
/* Version x.x.x --  August  1991                               \  /      */
/*                                                              /  \      */
/* Author: P. Patrick van der Smagt                          _  \/\/  _   */
/*         University of Amsterdam                          | |      | |  */
/*         Dept. of Computer Systems                        | | /\/\ | |  */
/*         Amsterdam                                        | | \  / | |  */
/*         THE NETHERLANDS                                  | | /  \ | |  */
/*         smagt@fwi.uva.nl                                 | | \/\/ | |  */
/*                                                          | \______/ |  */
/* This software has been written with financial             \________/   */
/* support of the Dutch Foundation for Neural Networks                    */
/* and is therefore owned by the mentioned foundation.          /\/\      */
/*                                                              \  /      */
/*                                                              /  \      */
/*                                                              \/\/      */
/**************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <signal.h>


#include <errno.h>

#include "global_communication.h"
#include "data_code.h"

/* for sockets */
#include <fcntl.h>
#include <sys/types.h>
#include <sys/errno.h>

#define COMMUNICATION_EXTERN
#include "communication.h"

#include "setup.h"


/*
 * Sends the new joint values to the controller.  Returns:
 * 0 if everything ok, otherwise the number of the
 * failing joint as returned by the other side of the socket.
 */
unsigned short send_joint_values(int nr, joint_values q, joint_values dq,
			joint_values ddq, int check_reach)
{
  char buffer[BUF_SIZE];
  unsigned short length;
  char *return_buffer, type;
  char *p_buffer;
  unsigned short i, chunk;

  /*
   * The 0th character indicates whether the robot should check
   * on validness of the target position.
   */
  buffer[0] = check_reach;
  /*
   * For user-friendly encoding, the sprintf-like function encode() 
   * is devised.  Call it as follows:
   */
  p_buffer = buffer+1;
  length = 1;
  for (i=1; i<=NR_LINKS; i++)
  {
        chunk = encode(p_buffer, "%f%f%f", (float) q[i], (float) dq[i],
                                                         (float) ddq[i]);
        length += chunk;
	if (length > BUF_SIZE)
	{
		fprintf(stderr, "Out of buffer space (increase BUF_SIZE)\n");
		close_down(PANIC, PANIC);
	}
        p_buffer += chunk;
  }

  send_data(d_robot_socket[nr], NEWPOS, length, buffer);
  length = receive_data(d_robot_socket[nr], &return_buffer, &type);

  if (type == STOP)
	close_down(STOP_OK, STOP);
  if (type == PANIC)
	close_down(PANIC_OK, PANIC);
  if (type == NEWPOS_BAD)
  {
	unsigned short joint;
	float j_min, j_val, j_max;

	/*
	 * First stop all other robots, too.
	 */
	stop_robot();

	decode(return_buffer, length, "%d%f%f%f",
				&joint, &j_val,&j_max,&j_min);
	if (joint == 0)
	{
		/*
		 * This is an illegal end-effector error.
		 */
		printf("illegal end-effector position, robot %d\n", nr);
		joint = NR_LINKS+1;
	}
	else
		printf("joint %d robot %d: not %f < %f < %f\n",
				joint, nr, j_min, j_val, j_max);

	free(return_buffer);
	return joint;
  }

  if (type != NEWPOS_OK)
  {
	fprintf(stderr, "send_joint_values() got bad message `%c'\n", type);
	close_down(PANIC, PANIC);
  }

  if (length != 0)
  {
	fprintf(stderr, "send_joint_values() got far too many bytes\n");
	close_down(PANIC, PANIC);
  }

  return 0;
}



int get_joint_values(int nr, int wait,
			joint_values q, joint_values dq, joint_values ddq)
/*
 * Read the current joint values.
 * If wait != 0, then it waits until the robot has
 * finished moving.  Also, it returns a flag indicating
 * whether the robot moves or not.
 *
 * Data is read from the connected robot as given by nr; usually, nr
 * is set to 0.
 *
 * Any one of the pointers q, dq, and ddq may be NULL pointers, in
 * which case they are not assigned.  This facilitates calls which
 * are only used to wait for the robot to stop: 
 *	get_joint_values(0, 1, NULL, NULL, NULL).
 */
{
  float j[NR_LINKS+1], dj[NR_LINKS+1], ddj[NR_LINKS+1];
  char *p_buffer;
  unsigned short i, chunk;
  int moving;

  short fields;
  unsigned short length;
  char *buffer, type;

  do
  {
	moving = 0;

	send_data(d_robot_socket[nr], SEND_JOINTS, 0, (char *) NULL);
	length = receive_data(d_robot_socket[nr], &buffer, &type);

	if (type == STOP)
		close_down(STOP_OK, STOP);
	if (type == PANIC)
		close_down(PANIC_OK, PANIC);
	if (type != SENT_JOINTS)
	{
		fprintf(stderr, "Expected SENT_JOINTS from robot (got `%c')\n",
								type);
		close_down(PANIC, PANIC);
	}

	chunk = length / NR_LINKS;
	p_buffer = buffer;
	for (i=1; i<=NR_LINKS; i++, p_buffer += chunk)
	{
        	if (decode(p_buffer,chunk,"%f%f%f",&j[i],&dj[i],&ddj[i]) != 3)
        	{
                	fprintf(stderr,
				"get_joint_values failed for joint %d\n", i);
                	close_down(PANIC, PANIC);
        	}

		if (q != NULL) q[i] = (REAL) j[i];
		if (dq != NULL) dq[i] = (REAL) dj[i];
		if (ddq != NULL) ddq[i] = (REAL) ddj[i];
		if (!ciszero(dj[i]) || !ciszero(ddj[i]))
			moving++;
	}

	if (buffer != NULL)
		free(buffer);
  } while (wait && moving);

  return moving;
}



void send_robot(char msg)
{
  int i;

  for (i=0; i<robots+simulators; i++)
  	send_data(d_robot_socket[i], msg, 0, (char *) NULL);
}




/*
 * Camera info is ONLY read from the first connected robot.
 */
short get_camera(int robot_nr, char cam_type, vector view_1, vector view_2)
{
  char *buffer, type;
  unsigned short length;
  int fields;
  float a1, a2, a3, a4, a5, a6;
  short error;

  send_data(d_camera_socket[robot_nr], cam_type, 0, (char *) NULL);
  length = receive_data(d_camera_socket[robot_nr], &buffer, &type);

  if (type == STOP)
	close_down(STOP, STOP_OK);
  if (type == PANIC)
	close_down(PANIC, PANIC_OK);
  if (type == NO_OBJECT)
	return -2;
  if (type != cam_type)
  {
	fprintf(stderr, "get_camera() received bad message\n");
	close_down(PANIC, PANIC);
  }

  fields = decode(buffer, length, "%d%f%f%f%f%f%f",
				&error, &a1, &a2, &a3, &a4, &a5, &a6);
  if (fields != 7)
  {
	fprintf(stderr, "get_camera() got %d fields, expected 7\n",
	fields);
	close_down(PANIC, PANIC);
  }

  /*
   * Check to see if some image processing error occurred.
   */
  if (error)
	return error;

  view_1[0] = (REAL) a1;
  view_1[1] = (REAL) a2;
  view_1[2] = (REAL) a3;
  view_1[3] = 1.0;
  view_2[0] = (REAL) a4;
  view_2[1] = (REAL) a5;
  view_2[2] = (REAL) a6;
  view_2[3] = 1.0;

  if (buffer != NULL)
	free(buffer);

  return 0;
}



/* 
 * Here is the difference between the real and the simulated
 * worlds: the target position is indicated by typing in
 * coordinates and sending them to the simulator.  In fact,
 * they should be generated by the bemmel's.  Not yet, not yet.
 */
void send_object_position(vector target)
{
  char buffer[100];		/* surely not too small. */
  char type;
  unsigned short length;
  int i;

  for (i=0; i<robots+simulators; i++)
  {
	/*
	 * ONLY send this to connected simulators.
	 */
	if (type_of_connected[i] == TYPE_ROBOT)
		continue;

 
	length = encode(buffer, "%f%f%f%f%f",
		(float) target[0], (float) target[1], (float) target[2],
		(float) E, (float) focus);

	send_data(d_camera_socket[i], OBJECT_POSITION, length, buffer);

	length = receive_data(d_camera_socket[i], (char **) NULL, &type);
	if (type == STOP)
		close_down(STOP, STOP_OK);
	if (type == PANIC)
		close_down(PANIC, PANIC_OK);
	if (type != OBJECT_RECEIVED)
	{
		fprintf(stderr,"send_object_position() got strange message\n");
		close_down(PANIC, PANIC);
	}
	if (length != 0)
	{
		fprintf(stderr, "send_object_position() in big trouble\n");
		close_down(PANIC, PANIC);
	}
  }
}



void close_down(char m_robot, char m_camera)
{
  int i;

  /*
   * Allow no interruptions whil closing the communication channels.
   */
  signal(SIGQUIT, SIG_IGN);
  signal(SIGINT, SIG_IGN);

  for (i=0; i<robots+simulators; i++)
  {
	if (robot_connected != NULL && robot_connected[i])
	{
		robot_connected[i] = 0;
		send_data(d_robot_socket[i], m_robot, 0, (char *) NULL);
		close(d_robot_socket[i]);
		close(c_robot_socket[i]);
	}

	if (camera_connected != NULL && camera_connected[i])
	{
		camera_connected[i] = 0;
		send_data(d_camera_socket[i], m_camera, 0, (char *) NULL);
		close(d_camera_socket[i]);
		close(c_camera_socket[i]);
	}
  }
  exit(0);
}


void stop_robot(void)
{
  char type;
  int i;

  for (i=0; i<robots+simulators; i++)
	send_data(d_robot_socket[i], HALT, 0, (char *) NULL);
  for (i=0; i<robots+simulators; i++)
  {
	receive_data(d_robot_socket[i], (char **) NULL, &type);
	if (type != HALT_OK)
	{
		fprintf(stderr, "robot %d won't stop moving!!\n", i+1);
		close_down(PANIC, PANIC);
	}
  }
}



void on_robot(void)
{
  int i;

  c_robot_socket = (int *) calloc(robots+simulators, sizeof(int));
  d_robot_socket = (int *) calloc(robots+simulators, sizeof(int));
  robot_connected = (int *) calloc(robots+simulators, sizeof(int));
  if (type_of_connected == NULL)
	type_of_connected = (char *) calloc(robots+simulators, sizeof(char));

  if (c_robot_socket==NULL || d_robot_socket==NULL ||
      robot_connected==NULL || type_of_connected==NULL)
  {
	perror("on_robot()");
	close_down(PANIC, PANIC);
  }

  for (i=0; i<robots+simulators; i++)
  {
	char socket_file_name[BUF_SIZE];

	sprintf(socket_file_name, "../socketnr/robot%d", i+1);
	setup_communication(socket_file_name, &c_robot_socket[i],
							&d_robot_socket[i]);
	robot_connected[i] = 1;
	type_of_connected[i] = (i < robots ? TYPE_ROBOT : TYPE_SIMULATOR);
  }
}


void on_camera(void)
{
  int i;

  c_camera_socket = (int *) calloc(robots+simulators, sizeof(int));
  d_camera_socket = (int *) calloc(robots+simulators, sizeof(int));
  camera_connected = (int *) calloc(robots+simulators, sizeof(int));
  if (type_of_connected == NULL)
	type_of_connected = (char *) calloc(robots+simulators, sizeof(char));

  if (c_camera_socket==NULL || d_camera_socket==NULL ||
      camera_connected==NULL || type_of_connected==NULL)
  {
	perror("on_camera()");
	close_down(PANIC, PANIC);
  }

  for (i=0; i<robots+simulators; i++)
  {
	char socket_file_name[BUF_SIZE];

	sprintf(socket_file_name, "../socketnr/camera%d", i+1);
	setup_communication(socket_file_name, &c_camera_socket[i],
							&d_camera_socket[i]);
	camera_connected[i] = 1;
	type_of_connected[i] = (i < robots ? TYPE_ROBOT : TYPE_SIMULATOR);
  }
}


/************************************************************************/
/*									*/
/************************************************************************/
void setup_communication(char *socketfile,
			int *control_socket, int *data_socket)
{
  char type;
  extern int errno;
  int socknr;
  FILE *sockfile;

  if ((sockfile = fopen(socketfile,"r")) == NULL)
  {
	fprintf(stderr, "can't open %s\n", socketfile);
	exit(1);
  }
  fscanf(sockfile, "%d", &socknr);
  fclose(sockfile);

  printf("waiting at socket %d\n", socknr);

  /* Setup socket.  The number of the connection is
     socknr (see common.h).  The socket line is
     in control_socket, whereas communication goes over the
     file descriptor data_socket.  The last parameter
     is the maximum number of clients that is allowed
     to connect.
   */
  if (setup_server_socket(socknr, control_socket, 1) == -1)
	exit(1);

  /* Now for each client accept the connection.  In this case,
     we only have one client and have to accept only once.
   */
  accept_client(*control_socket, data_socket);

  printf("client accepted\n");

  /* set up connection with client wait for alive char */
  send_data(*data_socket, START, 0, (char *) NULL);
  do
  {
	receive_data(*data_socket, (char **) NULL, &type);
	if (type == START_ERROR)
	{
		fprintf(stderr, "client says it can't start\n");
		close_down(PANIC, PANIC);
	}
	if (type != START_OK)
		fprintf(stderr, "received wrong message from client\n");
  } while (type != START_OK);

  printf("chris: client accepts start\n");
}




/************************************************************************/
/*			Socket routines					*/
/************************************************************************/

void send_data(int sock, char code, unsigned short length, char *data)
{
  char *buf;
  unsigned short i;
  int status;

  if ((buf = (char *) malloc(length+3)) == NULL)
  {
	fprintf(stderr, "send_data(): out of buffer space\n");
	close_down(PANIC, PANIC);
  }

  buf[0] = code;
  us_encode(length, buf+1);

  if (data != NULL)
	for (i=0; i<length; i++) 
		buf[i+3] = data[i];


  do
	status = send(sock, buf, (int) length+3, 0);
  while (status == -1 && errno == EWOULDBLOCK);

  if (status == -1)
  {
	perror("send_data()");
	close_down(PANIC, PANIC);
  }
  else if (status != length+3)
  {
	fprintf(stderr, "couldn't send all bytes\n");
	perror("send_data()");
	close_down(PANIC, PANIC);
  }

  free(buf);
}



unsigned short receive_data(int sock, char **buffer, char *type)
{
  extern int errno;
  unsigned short message_len;
  char buf[3];

  /* Read the header of the message. */
  while (1)
  {
	int status;

	status = recv(sock, buf, 3, 0);
	if (status == 3) break;
	if (status == -1 && errno == EWOULDBLOCK) continue;
	if (status != -1)
	{
		fprintf(stderr, "receive_data() received too few bytes");
		fprintf(stderr, "; received %d, expected 3\n", status);
	} else
		perror("receive_data()");
	close_down(PANIC, PANIC);
  }

  *type = buf[0];
  if (*type == PANIC)
  {
	fprintf(stderr, "Robot request an exit\n");
	close_down(PANIC, PANIC);
  }

  message_len = us_decode(buf+1);

  /*
   * Read the rest of the message.
   */
  if (buffer != (char **) NULL)
	*buffer = NULL;
  if (message_len > 0)
  {
	if (buffer == (char **) NULL)
	{
		fprintf(stderr, "receive_data(): message too long\n");
		close_down(PANIC, PANIC);
	}

	*buffer = (char *) malloc(message_len+1);
	if (*buffer == NULL)
	{
		fprintf(stderr,"receive_data(): out of buffer space\n");
		close_down(PANIC, PANIC);
	}

	while (1)
  	{
		int status;

		status = recv(sock, *buffer, message_len, 0);
		if (status == message_len) break;
		if (status == -1 && errno == EWOULDBLOCK) continue;
		if (status != -1)
		{
			fprintf(stderr,
				"receive_data() received too few bytes");
			fprintf(stderr, "; received %d, expected %d\n",
				status, message_len);
		} else
			perror("receive_data()");
		close_down(PANIC, PANIC);
  	}
  }

  return message_len;
}



/*
 * Check if the number x is near enough to 0.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int ciszero(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return iszero(rounded_x);

}


/*
 * Check if the number x is near enough to 0 or positive.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cispos(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return (iszero(rounded_x) || !signbit(rounded_x));
}



/*
 * Check if the number x is near enough to 0 or negative.  Incoming data (via
 * the socket) have rather low precision; therefore < 0.001 is
 * interpreted as near enough to 0.
 */
int cisneg(REAL x)
{
  double rounded_x;

  rounded_x = 0.001 * ((int) (0.5 + x*1000.0));

  return (iszero(rounded_x) || signbit(rounded_x));
}


