/*
 * Structure definitions for the controller
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>


typedef unsigned char	 byte;		/* define an 8 bit byte */

/* 
 * Use single-format floating point for most values
 * Homogenous coordinates (3-vectors for 2-D world)
 * Angles in radians, distances in mm
 */



/* Definition of the robot structure. */

typedef struct 
{
  float x,y, th;		/* pos in mm and rads ccw */
  float tv, rv;			/* trans vel (mm/sec) and ang vel (rads/sec) */
  float leftv, rightv;		/* left and right wheel velocities, mm/sec */
  byte status;			/* status byte */
	
  float acc, mtv, mrv;		/* current acceleration and velocity maximums */

  int motor_packet_count,	/* network info */
      sonar_packet_count,
      vision_packet_count;

  int battery;			/* level in tenths of a volt */
  int bumpers;			/* bumper mask */
  float control;		/* heading control, in rads */
  int say;			/* acknowledgment of SAY commands */
} robot;

extern robot flakey;	/* global variable for simulation robot */


/* Sonar circular buffers for storage */

#define CBUF_LEN 100

typedef struct			/* Circular buffers. */
{
  int start;			/* Start is the point at which entries start. */
  int end;			/* This points to the last entry + 1.  */
				/* Equals start when empty */
  int limit;			/* Extent of buffer */
  float xbuf[CBUF_LEN];
  float ybuf[CBUF_LEN];
} cbuf;


extern  cbuf *sraw_buf;		/* holds raw sonar readings */
extern  cbuf *sl_buf;		/* holds interpreted left/right readings */
extern  cbuf *sr_buf;
extern  cbuf st_buf;		/* stereo readings */



/* Sonar data collection buffer */

#define MAX_SONARS 12		/* number of sonars on robot */

typedef struct			/* sonar data collection buffer */
{
  float fx, fy, fth;		/* robot position when sonar read */
  float x, y;			/* sonar reading in RW coords */
  int range;			/* sonar range reading */
  int new;			/* whether it's a new reading */
} sdata;

extern sdata sbucket[];		/* holds one sdata per sonar, indexed by sonar number */


/* structures for fuzzy control rules and closures */

typedef union			/* a param can be either a fp number */
{				/* or a pointer */
  float f;
  void * p;
} param;

typedef param * beh_params;

extern param *
beh_alloc(int n);

/* these are for the action (consequent) of rules */

#define ACCEL 0
#define DECEL 1
#define TURN_LEFT 2
#define TURN_RIGHT 3

typedef struct
{
  char *name;			/* name of the rule */
  int antecedent,		/* activity of this rule */
      consequent,		/* action to take */
      parameter;		/* mean value of action */
} beh_rule;

typedef struct
{
  char *name;			/* name of behavior */
  beh_params (*setup_fn)(beh_params);	/* setup function */
  void (*update_fn)();		/* update function */
  int n;			/* number of rules */
  beh_rule rules[10];		/* rule array */
} behavior;
  

typedef struct			/* return structure for fuzzy rules */
{
  float acc, accv;		/* acceleration activity and value */
  float turn, turnv;		/* turn activity and value */
} c_ret;

extern float 
exec_acc, exec_turn; /* total defuzzified values */

typedef struct
{
  behavior *beh;		/* behavior */
  beh_params params;		/* parameters */
  int priority;			/* priority of this closure */
  int running;			/* whether we're actually running or not */
  float activity;		/* level of activity */
  float goal;			/* level of goal achievement */
  c_ret vals;			/* evaluation values */
} beh_closure;

extern beh_closure *behaviors[]; /* behavior closure list */
extern int num_bcs;		/* current number of behaviors */


extern float c_active, c_progress, c_goal;


/*
 * positions and points
 *
 */

typedef struct
{
  float x;			/* x,y position of point relative to flakey */
  float y;
  float th;			/* point orientation */
  int type;			/* type of point */
} point;

extern point *			/* holds list of points to be updated */
pointlist[];

extern float
point_phi(point *p);		/* returns the direction to the point */

extern int			/* pointer to last entry */
pointlist_p;

extern void
add_point(point *p);

extern void
rem_point(point *p);
