/* 

to print: lptops -h -P5bp -L0bp dpmatch1.out | alw 

A two phoneme (A and B) version of the output would look like this:

+	-	A	B	+
-	0	ni(A)	ni(B)	-
A	nd(A)	ns(A,A)	ns(A,B) A
B	nd(B)	ns(B,A)	ns(B,B)	B
+	-	A	B	+

Where ni(X) is the number of insertions of phoneme X, nd(X) the number of
deletions of phoneme X and ns(X,Y) the number of substitutions of phoneme X
(the correct phoneme) with Y (the recognised phoneme).  ns(X,X) are the
number of times X was recognised correctly and with the null phoneme "-",
nd(X) = ns(X,-) and ni(X) = ns(-,X), hence the table layout).

*/

# include "Tools.h"
# include "Timit.h"
# include "RM.h"

# define DEFAULT_RPREFIX	RM1_LOCAL_TRUDIR_PATH
# define DEFAULT_RSUFFIX	".wrd"
# define DEFAULT_UPREFIX	RM1_LOCAL_PATH
# define DEFAULT_USUFFIX	".wwp"
# define DEFAULT_SPREFIX	RM1_LOCAL_TRUDIR_PATH
# define DEFAULT_SSUFFIX	".sym"
# define DEFAULT_CPREFIX	RM1_LOCAL_PATH
# define DEFAULT_CSUFFIX	".cor"

# define DEFAULT_PHNTAB		RM1_DEFAULT_PHNTAB

# define MAX_LABEL_LENGTH	MAX(32,MAX(TIMIT_LABEL_SIZE, RM_LABEL_SIZE))
# define MAX_SENT_SIZE		256
# define MAX_NSYMBOL		1000 /* MAX(TIMIT_NALLPHONE, RM_NWORDS) */
# define INS_DIST		3
# define SUB_DIST		4
# define DEL_DIST		3

enum {mcor, msub, mdel, mins};

int phnread(char*, char*, char*, char**, int*, int*);
int fgetlabel(char*, FILE*);

int main(int argc, char **argv) {
  FILE  *fp_dbl;
  int   from[MAX_SENT_SIZE][MAX_NSYMBOL];
  char  **dict;
  int   rdata[MAX_SENT_SIZE], udata[MAX_SENT_SIZE];
  int	score[MAX_SENT_SIZE][MAX_NSYMBOL], nsub[MAX_NSYMBOL][MAX_NSYMBOL];
  int   nins[MAX_NSYMBOL], ndel[MAX_NSYMBOL];
  int   i, j, verbose, rsize, usize, ndict, tnsym = 0;
  int	printsymbol, printcorrect;
  char  base_name[MAX_NAME_SIZE];
  char	*rprefix, *rsuffix, *uprefix, *usuffix;
  char	*sprefix, *ssuffix, *cprefix, *csuffix;

  if(Scan_flag(argc, argv, "-h") || argc <= 1 ||
     (fp_dbl = Std_fopen(argv[argc - 1], "r")) == NULL) {
    fprintf(stderr, "Syntax:\t%s\n", *argv);
    fprintf(stderr, "\t-verbose\n");
    fprintf(stderr, "\t-confusion\n");
    fprintf(stderr, "\t-ray\n");
    fprintf(stderr, "\t-rprefix %s\n", DEFAULT_RPREFIX);
    fprintf(stderr, "\t-rsuffix %s\n", DEFAULT_RSUFFIX);
    fprintf(stderr, "\t-uprefix %s\n", DEFAULT_UPREFIX);
    fprintf(stderr, "\t-usuffix %s\n", DEFAULT_USUFFIX);
    fprintf(stderr, "\t-sprefix %s\n", DEFAULT_SPREFIX);
    fprintf(stderr, "\t-ssuffix %s\n", DEFAULT_SSUFFIX);
    fprintf(stderr, "\t-cprefix %s\n", DEFAULT_CPREFIX);
    fprintf(stderr, "\t-csuffix %s\n", DEFAULT_CSUFFIX);
    fprintf(stderr, "\t-phntab %s\n", DEFAULT_PHNTAB);
    fprintf(stderr, "\t<-|filename file>\n");
    exit(1);
  }

  verbose = Scan_flag(argc, argv, "-verbose");
  printsymbol  = Scan_flag(argc, argv, "-printsymbol");
  printcorrect = Scan_flag(argc, argv, "-printcorrect");
  rprefix = Scan_string(argc, argv, "-rprefix", DEFAULT_RPREFIX);
  rsuffix = Scan_string(argc, argv, "-rsuffix", DEFAULT_RSUFFIX);
  uprefix = Scan_string(argc, argv, "-uprefix", DEFAULT_UPREFIX);
  usuffix = Scan_string(argc, argv, "-usuffix", DEFAULT_USUFFIX);
  sprefix = Scan_string(argc, argv, "-sprefix", DEFAULT_SPREFIX);
  ssuffix = Scan_string(argc, argv, "-ssuffix", DEFAULT_SSUFFIX);
  cprefix = Scan_string(argc, argv, "-cprefix", DEFAULT_CPREFIX);
  csuffix = Scan_string(argc, argv, "-csuffix", DEFAULT_CSUFFIX);

  if((dict = Char_2d_array(MAX_NSYMBOL, MAX_LABEL_LENGTH)) == NULL)
    Panic("%s: Can't mkdict\n", *argv);

  for(i = 0; i < MAX_NSYMBOL; i++) {
    nins[i] = ndel[i] = 0;
    for(j = 0; j < MAX_NSYMBOL; j++) nsub[i][j] = 0;
  }

  if(Scan_flag(argc, argv, "-phntab")) {
    char **tdict = Read_table(Scan_string(argc, argv, "-phntab",
					  DEFAULT_PHNTAB), &ndict);
    for(i = 0; i < ndict; i++)
      strcpy(dict[i], tdict[i]);
  }
  else ndict = 0;

  for(i = 0; i < MAX_SENT_SIZE; i++) {
    score[i][0] = INS_DIST * i;
    from[i][0]  = mins;
  }
  for(j = 1; j < MAX_SENT_SIZE; j++) {
    score[0][j] = DEL_DIST * j;
    from[0][j]  = mdel;
  }
    
  while(fscanf(fp_dbl, "%s\n", base_name) == 1) {
    if(verbose == TRUE) printf("%s\n", base_name);
    rsize = phnread(rprefix, base_name, rsuffix, dict, &ndict, rdata);
    usize = phnread(uprefix, base_name, usuffix, dict, &ndict, udata);

    for(i = 1; i <= usize; i++)
      for(j = 1; j <= rsize; j++) {
	if(udata[i - 1] == rdata[j - 1]) {
	  score[i][j] = score[i - 1][j - 1];
	  from[i][j]  = mcor;
	}
	else {
	  int poss;

	  score[i][j] = score[i - 1][j - 1] + SUB_DIST;
	  from[i][j]  = msub;
	  if((poss = score[i - 1][j] + INS_DIST) < score[i][j]) {
	    score[i][j] = poss;
	    from[i][j]  = mins;
	  }
	  if((poss = score[i][j - 1] + DEL_DIST) < score[i][j]) {
	    score[i][j] = poss;
	    from[i][j]  = mdel;
	  }
	}
      }	

    for(i = usize, j = rsize; i != 0 || j != 0;) {
      switch(from[i][j]) {
      case mcor:
	nsub[rdata[--j]][udata[--i]]++;
	if(printsymbol) {
	  Ppush(dict[udata[i]]);
	  Ppush(dict[rdata[j]]);
	}
	if(printcorrect)
	  Ppush("T");
	break;
      case msub:
	nsub[rdata[--j]][udata[--i]]++;
	if(printsymbol) {
	  Ppush(dict[udata[i]]);
	  Ppush(dict[rdata[j]]);
	}
	if(printcorrect)
	  Ppush("F");
	break;
      case mins:
	nins[udata[--i]]++;
	if(printcorrect)
	  Ppush("F");
	break;
      case mdel:
	ndel[rdata[--j]]++;
	if(printsymbol) {
	  Ppush("-");
	  Ppush(dict[rdata[j]]);
	}
	break;
      default:
	Panic("%s: Unexpected move\n", *argv);
      }
    }
    tnsym += rsize;

    if(printsymbol) {
      FILE *fsym = Panic_fopen_3bits(sprefix, base_name, ssuffix, "w");
      char* p0, *p1;

      while((p0 = Ppull()) != NULL && (p1 = Ppull()) != NULL)
	fprintf(fsym, "%s\t%s\n", p0, p1);

      Panic_fclose(fsym);
    }
    if(printcorrect) {
      FILE *fcor = Panic_fopen_3bits(cprefix, base_name, csuffix, "w");      
      char* p0;

      while((p0 = Ppull()) != NULL)
	fprintf(fcor, "%s\n", p0);

      Panic_fclose(fcor);
    }
  }

  if(Scan_flag(argc, argv, "-confusion") == TRUE) {
    printf("+\t-\t");
    for(i = 0; i < ndict; i++) printf("%s\t", dict[i]);
    printf("+\n");
    printf("-\t%d\t", 0);
    for(i = 0; i < ndict; i++) printf("%d\t", nins[i]);
    printf("+\n");
    for(i = 0; i < ndict; i++) {
      printf("%s\t%d\t", dict[i], ndel[i]);
      for(j = 0; j < ndict; j++) printf("%d\t", nsub[i][j]);
      printf("%s\n", dict[i]);
    }
    printf("+\t-\t");
    for(i = 0; i < ndict; i++) printf("%s\t", dict[i]);
    printf("+\n");
  }
  else {
    int tncor = 0, tnins = 0, tnsub = 0, tndel = 0;

    for(i = 0; i < ndict; i++) tnins += nins[i];
    for(i = 0; i < ndict; i++) tndel += ndel[i];
    for(i = 0; i < ndict; i++)
      for(j = 0; j < ndict; j++)
	if(i == j) tncor += nsub[i][j];
	else tnsub += nsub[i][j];

    printf("nsym: %d\n", tnsym);
    printf("ncor: %d\t%4.1f\n", tncor, 100.0 * tncor / tnsym);
    printf("nins: %d\t%4.1f\n", tnins, 100.0 * tnins / tnsym);
    printf("nsub: %d\t%4.1f\n", tnsub, 100.0 * tnsub / tnsym);
    printf("ndel: %d\t%4.1f\n", tndel, 100.0 * tndel / tnsym);
    printf("nerr: %d\t%4.1f\n", tnins + tnsub + tndel, 
	   100.0 * (tnins + tnsub + tndel) / tnsym);
    
    printf(" & %4.1f\\%% & %4.1f\\%% & %4.1f\\%% & %4.1f\\%% & %4.1f\\%% \\\\\n",
	   100.0 * tncor / tnsym, 100.0 * tnins / tnsym, 100.0 * tnsub / tnsym,
	   100.0 * tndel / tnsym, 100.0 * (tnins + tnsub + tndel) / tnsym);
  }
  exit(0);
}

int fgetlabel(char *label, FILE *stream) {
  char line[MAX_NAME_SIZE];
  int  start, stop;

  line[0] = '\0';  /* TFM says that a '\0' is always appended -- False */
  do fgets(line, MAX_NAME_SIZE, stream);
  while(feof(stream) == 0 && strlen(line) <= 2);
  if(sscanf(line, "%d%d%s", &start, &stop, label) == 3) return(1);
  else return(0);
}

int phnread(char *prefix, char *base_name, char *suffix, char **dict, int *pndict, int *data) {
  FILE *stream;
  int  i = 0, nitem;
  char file_name[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
  
  (void) strncpy(file_name, prefix, MAX_NAME_SIZE);
  (void) strncat(file_name, base_name, MAX_NAME_SIZE);
  (void) strncat(file_name, suffix, MAX_NAME_SIZE);

  if((stream = fopen(file_name, "r")) == NULL)
    Panic("phnread: Can't open %s\n", file_name);

  for(nitem = 0; nitem < MAX_SENT_SIZE && fgetlabel(label, stream)!=0;nitem++){
    if(strcmp(label, "j") == 0) strcpy(label, "jh"); /* TIMIT .PHN HACK */

    for(i = 0; i < *pndict && strcmp(label, dict[i]) != 0; i++);
    if(i == *pndict)  {
      strncpy(dict[*pndict], label, MAX_LABEL_LENGTH);
      (*pndict)++;
      if(*pndict == MAX_NSYMBOL)
	Panic("phnread: Overflow of dictionary from file: %s\n", file_name);
    }
    data[nitem] = i;
  }
  if(nitem == MAX_SENT_SIZE) Panic("phnread: Overflow of MAX_SENT_SIZE\n");

  fclose(stream);
  return(nitem);
}
