/*
  title: dbuffer.c
  purpose: Double buffering support (used by projectors.c).
  
  authors:  Gareth Lee.
  date:     10-02-94
  modified: 

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  10-02-94: Routines extracted from projectors.c during a code rationalization.
*/
#include <stdio.h>

/* X lib & XView headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>
#include <xview/cms.h>

typedef REAL real;

#include "projectors.h"
#include "fview.h"
#include "interface.h"

int buffer_index = 0;                  /* Double buffer index must be 0 or 1 */
unsigned long buffer_colors[16]; /* colour entries allocated for 3D graphics */
unsigned long buffer_mask0, buffer_mask1;   /* Bit plane masks for 2 buffers */

/*
  SetupDoubleBuffer: Initializes the double buffering mechanism.
    Initializes and allocates the colour maps.
*/
void SetupDoubleBuffer(void)
{
  int i, def;
  unsigned r1, g1, b1, r2, g2, b2, r3, g3, b3;            /* 3D Color values */
  unsigned long plane_mask[4];
  unsigned long color;
  XGCValues xgcv;

  /* Setup the three 3D color cells used for double buffering */
  if (!XAllocColorCells(display, DefColormap, True, plane_mask,
                        4, &color, 1))
  {
    printf("Error: Couldn't allocate %d 3D r/w colors\n", 16);
    Quit(-1);
  }

  /* Generate buffer_colors from colors by adding various plane_masks */
  for (i = 0; i < 16; i++)
    buffer_colors[i] = color;
  buffer_colors[1]  |= plane_mask[0];
  buffer_colors[2]  |= plane_mask[1];
  buffer_colors[3]  |= plane_mask[1] | plane_mask[0];
  buffer_colors[4]  |= plane_mask[2];
  buffer_colors[5]  |= plane_mask[2] | plane_mask[0];
  buffer_colors[6]  |= plane_mask[2] | plane_mask[1];
  buffer_colors[7]  |= plane_mask[2] |  plane_mask[1] | plane_mask[0];
  buffer_colors[8]  |= plane_mask[3];
  buffer_colors[9]  |= plane_mask[3] | plane_mask[0];
  buffer_colors[10] |= plane_mask[3] | plane_mask[1];
  buffer_colors[11] |= plane_mask[3] | plane_mask[1] | plane_mask[0];
  buffer_colors[12] |= plane_mask[3] | plane_mask[2];
  buffer_colors[13] |= plane_mask[3] | plane_mask[2] | plane_mask[0];
  buffer_colors[14] |= plane_mask[3] | plane_mask[2] | plane_mask[1];
  buffer_colors[15] |= plane_mask[3]|plane_mask[2]|plane_mask[1]|plane_mask[0];

  /*
    Manually set RGB values for three dimensional colors.  Note: The default
    colors have been manually optimised (on a 24bit Sun monitor) for the
    RED/GREEN glasses used during development.
  */
  def = 1;
  if (ColorString != NULL)
  {
    if (sscanf(ColorString, "%u:%u:%u:%u:%u:%u:%u:%u:%u",
               &r1, &g1, &b1, &r2, &g2, &b2, &r3, &g3, &b3) == 9)
    {
      if (r1 <= 256 && g1 <= 256 && b1 <= 256 &&
          r2 <= 256 && g2 <= 256 && b2 <= 256 &&
          r3 <= 256 && g3 <= 256 && b3 <= 256)
      {
        def = 0;
        
        dim3_defs[0].red   = r1 * 256;               /* Color0 == Background */
        dim3_defs[0].green = g1 * 256;
        dim3_defs[0].blue  = b1 * 256;
        dim3_defs[1].red   = 0;                            /* Color1 == Text */
        dim3_defs[1].green = 0;
        dim3_defs[1].blue  = 255 * 256;
        dim3_defs[2].red   = r2 * 256;           /* Color2 == Left Eye (Red) */
        dim3_defs[2].green = g2 * 256;
        dim3_defs[2].blue  = b2 * 256;
        dim3_defs[3].red   = r3 * 256;        /* Color3 == Right Eye (Green) */
        dim3_defs[3].green = g3 * 256;
        dim3_defs[4].blue  = b3 * 256;
      }
      else
      {
        printf("Error: RGB intensity exceeds 256\n");
        Quit(-1);
      }
    }
    else
    {
      printf("Error: format error in RGB string\n");
      Quit(-1);
    }
  }
  
  /* Setup default RGB colors if no others are specified */
  if (def)
  {
    dim3_defs[0].red = 0xdc00;
    dim3_defs[0].green = 0x8700;
    dim3_defs[0].blue = 0xff00;
    dim3_defs[1].red = 0x0000;
    dim3_defs[1].green = 0x0000;
    dim3_defs[1].blue = 0xff00;
    dim3_defs[2].red = 0xfa00;
    dim3_defs[2].green = 0x0100;
    dim3_defs[2].blue = 0x0000;
    dim3_defs[3].red = 0x0000;
    dim3_defs[3].green = 0xa600;
    dim3_defs[3].blue = 0x6600;
  }

  dim3_defs[0].flags = DoRed | DoGreen | DoBlue;
  dim3_defs[1].flags = DoRed | DoGreen | DoBlue;
  dim3_defs[2].flags = DoRed | DoGreen | DoBlue;
  dim3_defs[3].flags = DoRed | DoGreen | DoBlue;

  /* Set bit_plane masks for the two buffers */
  buffer_mask0 = plane_mask[1] | plane_mask[0];
  buffer_mask1 = plane_mask[3] | plane_mask[2];

  /* Setup correct graphics contexts for 3D drawing (left and right traces) */
  gc_3d = XCreateGC(display, win, 0, &xgcv);
  XCopyGC(display, gc, 0xffffffff, gc_3d);
  gc1_3d = XCreateGC(display, win, 0, &xgcv);
  XCopyGC(display, gc, 0xffffffff, gc1_3d);
}

/*****************************************************************************/

/*
  SwapBuffers: Swaps over the buffer into which graphics may be drawn.
*/
void SwapBuffers(void)
{
  /* Swap buffer index */
  if (buffer_index)
    buffer_index = 0;
  else
    buffer_index = 1;
}

/*****************************************************************************/

/*
  GetHiddenColors & GetExposedColours: Returns drawing colour indices for the
    currently hidden and currently exposed buffers respectively.
*/
void GetExposedColors(unsigned long *mask,
                      unsigned long *text, unsigned long *bg,
                      unsigned long *left, unsigned long *right)
{
  /* switch between drawing colours */
  if (buffer_index)
  {
    *mask  = ~buffer_mask1;
    *bg    = dim3_color[0].pixel;
    *text  = dim3_color[1].pixel;
    *left  = dim3_color[2].pixel;
    *right = dim3_color[3].pixel;
  }
  else
  {
    *mask  = ~buffer_mask0;
    *bg    = dim3_color[0].pixel;
    *text  = dim3_color[4].pixel;
    *left  = dim3_color[8].pixel;
    *right = dim3_color[12].pixel;
  }
}

void GetHiddenColors(unsigned long *mask,
                      unsigned long *text, unsigned long *bg,
                      unsigned long *left, unsigned long *right)
{
  /* switch between drawing colours */
  if (buffer_index)
  {
    *mask  = ~buffer_mask0;
    *bg    = dim3_color[0].pixel;
    *text  = dim3_color[4].pixel;
    *left  = dim3_color[8].pixel;
    *right = dim3_color[12].pixel;
  }
  else
  {
    *mask  = ~buffer_mask1;
    *bg    = dim3_color[0].pixel;
    *text  = dim3_color[1].pixel;
    *left  = dim3_color[2].pixel;
    *right = dim3_color[3].pixel;
  }
}

/*****************************************************************************/

/*
  SwapDisplayBuffer: Changes the buffer being displayed when operating in
    double buffered display mode.  NB. only used in three dimensional display
    mode.
*/
void SwapDisplayBuffer(void)
{
  int i;
  struct itimerval interval;                       /* timer to provide delay */
  
  /* Set pallette according the buffer mode that is selected */
  if (buffer_index)
  {
    for (i = 0; i < 16; i++)
    {
      dim3_color[i] = dim3_defs[i % 4];
      dim3_color[i].pixel = buffer_colors[i];
    }
  }
  else
  {
    for (i = 0; i < 16; i++)
    {
      dim3_color[i] = dim3_defs[i / 4];
      dim3_color[i].pixel = buffer_colors[i];
    }
  }
  XStoreColors(display, DefColormap, dim3_color, 16);
  XFlush(display);
}

/*****************************************************************************/

/* end of dbuffer.c */
