/*
  title: lpc.c
  purpose: Frontend filter to generate LPC coefficients  sequences for use
    by the fview package.

  authors:  Gareth Lee and Edmund Lai.
  date:     03-11-93
  modified: 

  changes:
  28-10-93: lpc.c based on fbank.c by Edmund Lai and Gareth Lee.
*/
#define VERSION "1.0"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <math.h>
#include <memory.h>

#include "sfheader.h"
#include "fviewio.h"

/* Prototypes for functions in lpcautor.c */
void init_lpca(int order);
void close_lpca(void);
void rlpc_autocorrelation(int datalength,double s[],double a[],double *pcor);
void rlpc_auto(int datalength, double s[], double a[], double *power);
void rlpc_auto2 (double rr[], double a[], double pcor[]);

/* defines */

#define  HDR_SIZE      (10240)

typedef enum {unspecified, lpc, parcor, cepstra} OP_FORMAT;
typedef enum {unset, square, hamming, hanning} WINDOW_TYPE;

/* Globals */

double preemp_factor;
int    debug       = 0;                                   /* debugging level */
int    order       = 0;                    /* number of feature coefficients */
int    cep_order   = 0;                   /* number of cepstral coefficients */
int    preemphasis = 0;                                    /* no preemphasis */
int    flength     = 256;                                    /* frame length */
int    flength2    = 128;                              /* always flength / 2 */
int    advs        = 0;                   /* frame advance (marked as unset) */

OP_FORMAT format = unspecified;             /* format of features to produce */
FviewSampleHeader fsh;                        /* input samples header record */
FviewFeatureHeader ffh;                     /* output features header record */
char header[HDR_SIZE];                         /* Textual header information */
WINDOW_TYPE win_type = unset;                      /* analysis window to use */

extern char *optarg;                   /* for use with the getopt() function */

/*****************************************************************************/

void parse_cmd_line(int argc, char* argv[])
{
  int opt;
  
  while ((opt = getopt(argc, argv, "a:d:f:F:l:o:p:w:")) != -1)
  {
    switch (opt)
    {
    case 'a':
      advs = atoi(optarg);
      break;
    case 'd':
      debug = atoi(optarg);
      break;
    case 'f':
      order = atoi(optarg);
      break;
    case 'F':
      cep_order = atoi(optarg);
      break;
    case 'l':
      flength = atoi(optarg);
      break;
    case 'o':
      if (strcmp("lpc", optarg) == 0 || strcmp("LPC", optarg) == 0)
      {
        format = lpc;
        break;
      }
      if (strcmp("parcor", optarg) == 0 || strcmp("PARCOR", optarg) == 0)
      {
        format = parcor;
        break;
      }
      if (strcmp("cepstra", optarg) == 0 || strcmp("CEPSTRA", optarg) == 0)
      {
        format = cepstra;
        break;
      }
    case 'p':
      preemphasis = 1;
      preemp_factor = atof(optarg);
      break;
    case 'w':
      if (strcmp("square", optarg) == 0 || strcmp("SQUARE", optarg) == 0)
      {
        win_type = square;
        break;
      }
      if (strcmp("hamming", optarg) == 0 || strcmp("HAMMING", optarg) == 0)
      {
        win_type = hamming;
        break;
      }
      if (strcmp("hanning", optarg) == 0 || strcmp("HANNING", optarg) == 0)
      {
        win_type = hanning;
        break;
      }
      break;
    case '?':
      fprintf(stderr, "lpc: command line error: '%s'.\n", optarg);
      exit(-1);
    }
  }

  /* Set default advs as a fraction of the analysis frame width */
  if (advs == 0)
    if (flength > 0)
      advs = (int) ((100.0 / 256.0) * flength);
    else
      advs = 100;

  /* Check that an output format has been specified */
  if (format == unspecified)
  {
    fprintf(stderr, "lpc: must specify a output format option (with -o)\n");
    exit(-1);
  }

  /* Check that an analysis window has been set */
  if (win_type == unset)
  {
    fprintf(stderr, "lpc: must specify an analysis window (with -w)\n");
    exit(-1);
  }

  /* Check that a feature dimension has been specified */
  if (order == 0)
  {
    fprintf(stderr, "lpc: must specify feature dimension (with -f)\n");
    exit(-1);
  }
  if (format == cepstra && cep_order == 0)
  {
    fprintf(stderr, "lpc: must specify cepstral dimension (with -F)\n");
    exit(-1);
  }
}
  
/*****************************************************************************/

void lpc_to_cep(int p, double a[], int cep, double c[])
/* p - order of LPC ; cep - order of cepstrum */
/* a - LPC co-efficients ; c - LPC cepstral coeffs */
{
        int     n,k;
 
        c[0] = a[0];
        for (n = 1; n < p; n++) {
                c[n] = a[n];
                for (k = 1; k <= n; k++) 
                    c[n] += (1-(((double)k)/(n+1))) * a[k-1] * c[n-k];
        }
        for (n = p; n < cep; n++ ) {
                c[n] = 0;
                for (k = 1; k <= p; k++)
                    c[n] += (1-(((double)k)/(n+1))) * a[k-1] * c[n-k];
        }
}

/*****************************************************************************/

void main(int argc, char* argv[])
{
  int 	  i, j, nframes, nsamples, m;
  short	  *bdata, *bdtmp;
  off_t	  datasize;
  double  *w, *sbuf;
  double  *lpc_coeffs, *parcor_coeffs, *cepstral_coeffs;     /* feature sets */
  char    *cptr;
  char	  *rindex();
  char    magic[10];
  
  parse_cmd_line(argc, argv);
    
  /* Read header information from data file and check file type */
  if (fread(&fsh, sizeof(FviewSampleHeader), 1, stdin) != 1)
  {
    fprintf(stderr, "lpc: error reading header record\n");
    exit(1);
  }
  strcpy(magic, FVIEW_SAMPLE_MAGIC);
  for (i = 0; i < 8; i++)
    if (fsh.magic[i] != magic[i])
      break;
  if (i < 8)
  {
    fprintf(stderr, "lpc: magic number incorrect (%s)\n", fsh.magic);
    exit(1);
  }
  if (fread(header, sizeof(char), fsh.info_size, stdin) != fsh.info_size)
  {
    fprintf(stderr, "lpc: error reading %d byte textual header\n",
            fsh.info_size);
    exit(1);
  }
  
  /* Read data */
  bdata = (short *) malloc(fsh.number_samples * sizeof(short));
  if (fread(bdata, sizeof(short), fsh.number_samples, stdin) !=
      fsh.number_samples)
  {
    fprintf(stderr, "lpc: data finished prematurely\n");
    exit(1);
  }
  nframes = (fsh.number_samples - flength + advs) / advs;
  
  if (preemphasis)
    preemp_short(fsh.number_samples, preemp_factor, &bdata[1]);

  /* Append extra information to textual header */
  cptr = header + strlen(header);
  sprintf(cptr,"LPC\nframes=%d\nlpc_order=%d\nframe_length=%d\n"
          "frame_advance=%d\n",
          nframes,   /* Number of observation vectors that will be generated */
          order,                         /* LPC order used for analysis */
          flength,		        /* Frame length in number of samples */
          advs);	       	  /* Advancement per frame in no. of samples */
  cptr = header + strlen(header);


  memcpy(ffh.magic, FVIEW_FEATURE_MAGIC, 8);
  ffh.number_observations = nframes;

  switch(format)
  {
  case lpc:
    sprintf(cptr, "features=lpc\n");
    ffh.vector_dimension = order;
    break;
  case parcor:
    sprintf(cptr, "features=parcor\n");
    ffh.vector_dimension = order + 1;
    break;
  case cepstra:
    sprintf(cptr, "features=cepstra\n");
    ffh.vector_dimension = cep_order + 1;
    break;
  }

  ffh.info_size = strlen(header) + 1;
  if (ffh.info_size > HDR_SIZE)
  {
    fprintf(stderr, "lpc: textual header overflow\n");
    exit(-1);
  }
  fwrite(&ffh, sizeof(FviewFeatureHeader), 1, stdout);      /* header record */
  fwrite(header, sizeof(char), ffh.info_size, stdout);        /* text header */

  /* allocate memory */
  w = (double *) malloc(flength * sizeof(double));
  sbuf = (double *) malloc(flength * sizeof(double));
  lpc_coeffs = (double *) malloc((order + 1) * sizeof(double));
  parcor_coeffs = (double *) malloc((order + 1) * sizeof(double));
  cepstral_coeffs = (double *) malloc((cep_order + 1) * sizeof(double));

  switch(win_type)
  {
  case square:
    Square(flength, w);                          /* Generate a square window */
    break;
  case hamming:
    Hamming(flength, w);                        /* Generate a Hamming window */
    break;
  case hanning:
    Hanning(flength, w);                        /* Generate a Hanning window */
    break;
  }

  if (debug > 0)
    fprintf(stderr, "lpc: Performing %d point LPC analysis ...\n", flength);

  /* Perform LPC Analysis on each frame */
  init_lpca(order);
  for (m = 0, bdtmp = bdata; m < nframes; m++, bdtmp += advs)
  {
    detrend_short(flength, bdtmp, sbuf);    /* ensure samples have zero mean */
    window(flength, w, sbuf);                   /* Perform Hamming windowing */

    rlpc_autocorrelation(flength, sbuf, lpc_coeffs, parcor_coeffs);

    if (format == cepstra)
      lpc_to_cep(order, lpc_coeffs, cep_order, cepstral_coeffs);

    /* Store coefficients into file */
    switch (format)
    {
    case lpc:
      fwrite(&lpc_coeffs[1], sizeof(double), order, stdout);
      break;
    case parcor:
      fwrite(parcor_coeffs, sizeof(double), (order + 1), stdout);
      break;
    case cepstra:
      fwrite(cepstral_coeffs, sizeof(double), (cep_order + 1), stdout);
      break;
    }
  }
  close_lpca();

  /* free memory */
  free(bdata);
  free(w);
  free(sbuf);
  free(lpc_coeffs);
  free(parcor_coeffs);
  free(cepstral_coeffs);

  /* Signal successful completion */
  switch(format)
  {
  case lpc:
    fprintf(stderr, "lpc (%s): lpc coeffs: %d samps -> %d frames\n",
            VERSION, fsh.number_samples, nframes);
    break;
  case parcor:
    fprintf(stderr, "lpc (%s): parcor coeffs: %d samps -> %d frames\n",
            VERSION, fsh.number_samples, nframes);
    break;
  case cepstra:
    fprintf(stderr, "lpc (%s): cepstral coeffs: %d samps -> %d frames\n",
            VERSION, fsh.number_samples, nframes);
    break;
  }
}

