/*
  title:    htk.c
  purpose:  Converters HTK V1.2 (& V1.4) files into a format suitable for use
  with the fview package.  Note: must either be compiled with the V12 flag
  defined (indicating version 1.2 compatibility) or V14 (for version 1.4).

  author:   Gareth Lee
  date:     01-11-93
  modified: 02-11-93

  changes:
  01-11-93: Program (loosely) based on sfconverter.c
*/
#define VERSION "1.0"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "fviewio.h"

/*****************************************************************************/

/* HTK standard header */

typedef struct {
   long nSamples;
   long sampPeriod;
   short sampSize;
   short sampKind;
} HTKhdr;

/* floating point precision used by the HTK implementation */
typedef float HTKReal;

/* HTK Data types supported */
#define WAVEFORM  (0)
#define LPC       (1)
#define LPCREFC   (2)
#define LPCEPSTRA (3)
#define LPDELCEP  (4)
#define IREFC     (5)
#define MFCC      (6)
#define FBANK     (7)

#ifdef V14
#define MELSPEC   (8)
#endif

/* Modifiers that may be applied to the basic types (from bit six upwards) */
#define TYPE_MASK   (0x3f)
#define WITH_ENERGY (0x01)

#ifdef V12
#define WITH_PITCH  (0x02)
#endif

#ifdef V14
#define WITH_NULLE  (0x02)
#define WITH_DELTA  (0x04)
#define WITH_ACCS   (0x08)
#endif

/* Byte order options */
#define UNSET_ENDIAN  (-1)
#define BIG_ENDIAN    (0)
#define LITTLE_ENDIAN (1)

typedef enum { undecided, samples, features } OP_FORMATS;

#define MAX_HDR_SIZE (1024)

/*****************************************************************************/

/* HTK type name strings  and the names of the modifiers that may be applied */
const char* data_type_names[8] = {"WAVEFORM", "LPC", "LPCREFC", "LPCEPSTRA",
                                  "LPDELCEP", "IREFC", "MFCC", "FBANK"};
#ifdef V12
const char *modifier_type_names[2] = {"_E", "_P"};
#endif

#ifdef V14
const char *modifier_type_names[4] = {"_E", "_N", "_D", "_A"};
#endif

/* Globals */
unsigned           sampKind;
unsigned           sampMods;
char               filename[1024];
int                endian = UNSET_ENDIAN;

/* Header records */
HTKhdr             hhdr;
OP_FORMATS         output_format;
FviewSampleHeader  fsh;
FviewFeatureHeader ffh;
char               header[MAX_HDR_SIZE];

/* for use with getopt */
extern char *optarg;
extern int optind, opterr;

/*****************************************************************************/

void main(int argc, char *argv[])
{
  int frame, index, opt;
  FILE *fd;
  char *cp;
  short *short_buffer;
  double *double_buffer;
  HTKReal *htk_buffer;
  char name_buffer[1024];
  
  /* Parse command line */
  while ((opt = getopt(argc, argv, "e:f:o:")) != -1)
    switch(opt) {

    case 'e':
      if (strcmp("big", optarg) == 0 || strcmp("BIG", optarg) == 0)
        endian = BIG_ENDIAN;
      else
        if (strcmp("little", optarg) == 0 || strcmp("LITTLE", optarg) == 0)
          endian = LITTLE_ENDIAN;
      break;

    case 'f':
      strcpy(filename, optarg);
      break;

    case 'o':
      output_format = undecided;
      switch (*optarg)
      {
      case 'f': case 'F':
        output_format = features;
        break;
      case 's': case 'S':
        output_format = samples;
        break;
      }
      if (output_format == undecided)
      {
        fprintf(stderr, "htk: output format must be specified (correctly)\n");
        exit(-1);
      }
      break;
    }
  
  if ((fd = fopen(filename, "rb")) == NULL) {
    fprintf(stderr, "htk: error opening file %s for reading\n", filename);
    exit(-1);
  }

  /* read in header information */
  if (fread(&hhdr, sizeof(HTKhdr), 1, fd) != 1)
  {
    fprintf(stderr, "htk: error reading HTK header\n");
    exit(-1);
  }
  sampKind = hhdr.sampKind & TYPE_MASK;
  sampMods = hhdr.sampKind >> 6;

  /* Check the HTK Header record */
  switch (output_format)
  {
  case samples:
    if (sampKind != WAVEFORM)
    {
      fprintf(stderr, "htk: file format is not WAVEFORM\n");
      exit(-1);
    }
    if (hhdr.sampSize != 2)
    {
      fprintf(stderr, "htk: can only cope with 16 bit WAVEFORM samples\n");
      exit(-1);
    }

    /* Create an Fview sample header record */
    memcpy(fsh.magic, FVIEW_SAMPLE_MAGIC, 8);
    fsh.number_samples = (unsigned) hhdr.nSamples;
    fsh.sample_freq = 10000000 / hhdr.sampPeriod;
    
    break;

  case features:
#ifdef V12
    if (sampKind < LPC || sampKind > FBANK)
#endif
#ifdef V14
    if (sampKind < LPC || sampKind > MELSPEC)
#endif
    {
      fprintf(stderr, "htk: file format is not a known feature type\n");
      exit(-1);
    }

    /* Create an Fview feature header record */
    memcpy(ffh.magic, FVIEW_FEATURE_MAGIC, 8);
    ffh.vector_dimension = hhdr.sampSize / sizeof(HTKReal);
    ffh.number_observations = (unsigned) hhdr.nSamples;
    
    break;
  }

#ifdef V12
  strcpy(name_buffer, data_type_names[sampKind]);
  if (sampMods & WITH_ENERGY)
    (void) strcat(name_buffer, modifier_type_names[0]);
  if (sampMods & WITH_PITCH)
    (void) strcat(name_buffer, modifier_type_names[1]);
#endif

#ifdef V14
  strcpy(name_buffer, data_type_names[sampKind]);
  if (sampMods & WITH_ENERGY)
    (void) strcat(name_buffer, modifier_type_names[0]);
  if (sampMods & WITH_NULLE)
    (void) strcat(name_buffer, modifier_type_names[1]);
  if (sampMods & WITH_DELTA)
    (void) strcat(name_buffer, modifier_type_names[2]);
  if (sampMods & WITH_ACCS)
    (void) strcat(name_buffer, modifier_type_names[3]);
#endif
  
  /* Create the textual header to go with the file */
  sprintf(header, "HTK1.2\nnSamples=%d\nsampPeriod=%d\nsampSize=%d\n"
          "sampKind=%s\n", hhdr.nSamples, hhdr.sampPeriod, hhdr.sampSize,
          name_buffer);

  /* Write out Fview headers to stdout */
  switch (output_format)
  {
  case samples:
    cp = header + strlen(header);
    sprintf(cp, "SampleFreq=%d\n", fsh.sample_freq);
    fsh.info_size = strlen(header) + 1;

    if (fwrite(&fsh, sizeof(FviewSampleHeader), 1, stdout) != 1)
    {
      fprintf(stderr, "htk: failed to write FviewSampleHeader\n");
      exit(-1);
    }

    fwrite(header, sizeof(char), fsh.info_size, stdout);
    break;
  case features:
    cp = header + strlen(header);
    sprintf(cp, "SampleFreq=%dHz\nDimension=%d\n",
            (10000000 / hhdr.sampPeriod), ffh.vector_dimension);
    ffh.info_size = strlen(header) + 1;

    if (fwrite(&ffh, sizeof(FviewFeatureHeader), 1, stdout) != 1)
    {
      fprintf(stderr, "htk: failed to write FviewFeatureHeader\n");
      exit(-1);
    }

    fwrite(header, sizeof(char), ffh.info_size, stdout);
    break;
  }

  /* perform conversion depending on the format of the data */
  switch(sampKind)
  {
  case WAVEFORM:

    /* copy 16 bit twos compliment samples across */
    short_buffer = (short *) malloc(hhdr.nSamples * sizeof(short));
    if (fread(short_buffer, sizeof(short), hhdr.nSamples, fd) != hhdr.nSamples)
    {
      fprintf(stderr, "htk: failed to read sample data\n");
      exit(-1);
    }
    if (fwrite(short_buffer, sizeof(short), hhdr.nSamples, stdout) !=
        hhdr.nSamples)
    {
      fprintf(stderr, "htk: failed to write sample data\n");
      exit(-1);
    }
    free(short_buffer);
    break;

  case LPC:
  case LPCEPSTRA:
  case LPDELCEP:
  case LPCREFC:
  case MFCC:
  case FBANK:
#ifdef V14
  case MELSPEC:
#endif

    /* convert from HTKReal format to double during copying */
    htk_buffer = (HTKReal *) malloc(ffh.vector_dimension * sizeof(HTKReal));
    double_buffer = (double *) malloc(ffh.vector_dimension * sizeof(double));

    for (frame = 0; frame < ffh.number_observations; frame++)
    {
      if (fread(htk_buffer, sizeof(HTKReal), ffh.vector_dimension, fd) !=
          ffh.vector_dimension)
      {
        fprintf(stderr, "htk: failed to read HTK feature data\n");
        exit(-1);
      }
      for (index = 0; index < ffh.vector_dimension; index++)
        double_buffer[index] = (double) htk_buffer[index];
      if (fwrite(double_buffer, sizeof(double), ffh.vector_dimension, stdout)
          != ffh.vector_dimension)
      {
        fprintf(stderr, "htk: failed to write HTK feature data\n");
        exit(-1);
      }
    }
    
    free(double_buffer);
    free(htk_buffer);
    break;

  case IREFC:
    
    /* Convert from integer encoded coefficients to double during copying */
    short_buffer = (short *) malloc(ffh.vector_dimension * sizeof(short));
    double_buffer = (double *) malloc(ffh.vector_dimension * sizeof(double));
    
    for (frame = 0; frame < ffh.number_observations; frame++)
    {
      if (fread(short_buffer, sizeof(short), ffh.vector_dimension, fd) !=
          ffh.vector_dimension)
      {
        fprintf(stderr, "htk: failed to read HTK (short) feature data\n");
        exit(-1);
      }
      for (index = 0; index < ffh.vector_dimension; index++)
        double_buffer[index] = (double) short_buffer[index];
      if (fwrite(double_buffer, sizeof(double), ffh.vector_dimension, stdout)
          != ffh.vector_dimension)
      {
        fprintf(stderr, "htk: failed to write HTK feature data\n");
        exit(-1);
      }
    }
    
    free(double_buffer);
    free(short_buffer);
    break;      
  }
  
  fclose(fd);
  
  /* print out some usefile information */
#ifdef V12
  fprintf(stderr, "htk1.2 (%s):file %s, frames %d\n",
          VERSION, filename, hhdr.nSamples);
#endif
#ifdef V14
  fprintf(stderr, "htk1.4 (%s):file %s, frames %d\n",
          VERSION, filename, hhdr.nSamples);
#endif
  
  fflush(stderr);
}

