/*
  title: projectors.c
  purpose: Projection routines called from within fview.  Routines are
    provided for 2 and 3 dimensional projections and sweeps (slowed
    projections).  A spectrogram projector is also provided.
  
  authors:  Gareth Lee & Nicole Duball.
  date:     16-08-93
  modified: 08-09-93

  changes:
  
  08-09-93: functions copied out of fview.c
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <signal.h>
#include <sys/time.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>

/* Header files associated with other compilation units */
#include "projectors.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"

/* Values for use in the Color Locking Tables */
#define COLOR_FREE  (0)
#define COLOR_USED  (1)

int ColorLock[MAX_COLORS];            /* Ensures unique allocation of colors */

/*****************************************************************************/

/*
  SetupColorEntries: Initialize colour allocation table.
*/
void SetupColorEntries(void)
{
  int i;
  
  for (i = 0; i < MAX_COLORS; i++)
    ColorLock[i] = COLOR_FREE;
}

/*****************************************************************************/

/*
  AllocateColorEntry: Allocate a unique color for drawing.
*/
int AllocateColorEntry(void)
{
  int i;
  
  for (i = 0; i < MAX_COLORS; i++)
    if (ColorLock[i] == COLOR_FREE)
      break;

  if (i == MAX_COLORS)
  {
    printf("Error: Failed to find a free color in which to draw\n");
    Quit(-1);
  }
  else
    ColorLock[i] = COLOR_USED;    /* reserve the color to ensure exclusivity */
  
  return i;
}

/*****************************************************************************/

/*
  FreeColorEntry: reclaim a color previously allocated.
*/
void FreeColorEntry(int color)
{
  ColorLock[color] = COLOR_FREE;
}

/*****************************************************************************/

/*
  AlarmHandler: Interrupt handler for the SIGALRM signal produced when
    drawing in sweep mode.
*/
void AlarmHandler(void)
{
}

/*****************************************************************************/

/*
  OriginAdjust: ensure the min-max range calculated for auto_scale mode
    contains the origin.
*/
void OriginAdjust(double *min, double *max)
{
  if (*min > 0.0)
    *min = 0.0;
  else
    if (*max < 0.0)
      *max = 0.0;
}

/*****************************************************************************/

/*
  DrawOrigin2: Draw an origin symbol centered at (x,y) co-ordinates.
*/
void DrawOrigin2(int x, int y)
{
  const int radius = 8;                          /* radius of circle to draw */
  
  int x1 = x - radius;
  int x2 = x + radius;
  int y1 = y - radius;
  int y2 = y + radius;

  /* Draw black cross hairs and a surrounding circle */
  XSetForeground(display, gc, traj_color[BLACK].pixel);
  XDrawLine(display, win, gc, x1,  y, x2,  y);
  XDrawLine(display, win, gc,  x, y1,  x, y2);
  XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, 23040);  
}

/*****************************************************************************/
/*
  DrawOrigin3: Draw a three dimensional origin symbol centered at (x,y,z)
    co-ordinates.
*/
void DrawOrigin3(int x, int y, int z_offs)
{
  const int radius = 8;                          /* radius of circle to draw */
  const int left_color = 1;
  const int right_color = 2;

  int x1, x2;
  int y1 = y - radius;
  int y2 = y + radius;

  if (z_offs < 5)
  {
    /* Draw cross hairs and a surrounding circle in a neutral colour */
    XSetForeground(display, gc, traj_color[BLUE].pixel);
    x1 = x - radius;
    x2 = x + radius;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x - z_offs), y1,  (x - z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, 23040);  
  }
  else
  {
    /* Draw cross hairs and a surrounding circle in both colours */
    XSetForeground(display, gc, dim3_color[left_color].pixel);
    x1 = x - radius - z_offs;
    x2 = x + radius - z_offs;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x - z_offs), y1,  (x - z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, 23040);  
    
    XSetForeground(display, gc, dim3_color[right_color].pixel);
    x1 = x - radius + z_offs;
    x2 = x + radius + z_offs;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x + z_offs), y1,  (x + z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, 23040);
  }
}

/*****************************************************************************/

/* Draw 3D projection of a single trajectory */
void draw_3d_graphics(unsigned int window_width, unsigned int window_height,
                      int label)
{
  const int left_color = 1;                      /* color for left eye trace */
  const int right_color = 2;                    /* color for right eye trace */
  
  int i, s, x, y;
  int width,height,base_line;
  int drawx1, drawx2, drawy1, drawy2;
  XPoint *left_points, *right_points;
  double xmin, xmax, ymin, ymax, zmin, zmax, proj;
  int index, zoffset;
  XSetWindowAttributes attributes;
  SpeechRecord *sd;
  
  /* Reduce drawable area slightly to allow for offsets */
  drawx1 = X_MARGIN_3D;
  drawx2 = window_width - X_MARGIN_3D;
  drawy1 = Y_MARGIN_3D;
  drawy2 = window_height - Y_MARGIN_3D;
  
  /* Automatic scaling to ensure trajectory fills projection area */
  s = utterance_index;
  sd = spdata[s];
  left_points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));
  right_points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));
  
  for (i = 0; i < sd->npoints; i++)
  {
    if (i == 0)
    {
      xmin = xmax = sd->project1[i];
      ymin = ymax = sd->project2[i];
      zmin = zmax = sd->project3[i];
      continue;
    }
    proj = sd->project1[i];
    if (proj > xmax)
      xmax = proj;
    else
      if (proj < xmin)
        xmin = proj;
    proj = sd->project2[i];
    if (proj > ymax)
      ymax = proj;
    else
      if (proj < ymin)
        ymin = proj;
    proj = sd->project3[i];
    if (proj > zmax)
      zmax = proj;
    else
      if (proj < zmin)
        zmin = proj;
  }

  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
    OriginAdjust(&zmin, &zmax);
  }

  for (i = 0; i < sd->npoints; i++) {

    /*
      draw trajectories using appropriate scaling to fill the rectangle
      specified by drawx1, drawx2, drawy1 and drawy2.
    */
    x = (short) nint((sd->project1[i] - xmin) * (drawx2 - drawx1) /
                                                       (xmax - xmin)) + drawx1;
    y = (short) nint((sd->project2[i] - ymin) * (drawy2 - drawy1) /
                                                       (ymax - ymin)) + drawy1;
    zoffset = (short) nint(DEPTH_SCALING * ((sd->project3[i] - zmin) /
                                   (zmax - zmin)));
    /* left eye projection */
    left_points[i].x = x - zoffset;
    left_points[i].y = y;

    /* right eye projection */
    right_points[i].x = x + zoffset;
    right_points[i].y = y;
  }

  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = dim3_color[0].pixel;
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);
  XClearWindow(display, win);

  XSetForeground(display, gc, dim3_color[left_color].pixel);
  XDrawLines(display, win, gc, left_points, sd->npoints, CoordModeOrigin);
  if (label)
  {
    XDrawString(display, win, gc, left_points[0].x, left_points[0].y,
                "Start", 5);
    XDrawString(display, win, gc, left_points[sd->npoints - 1].x,
                left_points[sd->npoints - 1].y, "End", 3);
  }
  XSetForeground(display, gc, dim3_color[right_color].pixel);
  XDrawLines(display, win, gc, right_points, sd->npoints, CoordModeOrigin);
  if (label)
  {
    XDrawString(display, win, gc, right_points[0].x, right_points[0].y,
                "Start", 5);
    XDrawString(display, win, gc, right_points[sd->npoints - 1].x,
                right_points[sd->npoints - 1].y, "End", 3);
  }

  /* draw remaining labels */
  if (label && display_label_mode)
  {
    XSetForeground(display, gc, traj_color[BLUE].pixel);
    base_line = window_height - 10;
    XDrawString(display, win, gc, 10, base_line, sd->file_name,
                sd->file_name_length);
  }
  
  /* print highlighted vertices */
  if (highlighted && sd->nverts)
  {
    for (i = 0; i < highlighted; i++)
    {
      index = sd->order[i];
      XSetForeground(display, gc, dim3_color[left_color].pixel);
      XDrawArc(display, win, gc, left_points[index].x - 4,
               left_points[index].y - 4, 8, 8, 0, 23040);
      XSetForeground(display, gc, dim3_color[right_color].pixel);
      XDrawArc(display, win, gc, right_points[index].x - 4,
               right_points[index].y - 4, 8, 8, 0, 23040);
    }
  }

  if (display_origin_mode)
  {
    x = (short) nint(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
    y = (short) nint(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    zoffset = (short) nint(DEPTH_SCALING * (zmin / (zmin - zmax)));
    DrawOrigin3(x, y, zoffset);
  }

  free(right_points);
  free(left_points);
}

/*****************************************************************************/

/* Slowly draw 3D projection of a single trajectory */
void draw_3d_sweep(unsigned int window_width, unsigned int window_height,
                   int label, unsigned int sleep_period)
{
  const int left_color = 1;                      /* color for left eye trace */
  const int right_color = 2;                    /* color for right eye trace */
  
  int i, j, s, x, y;
  int width,height,base_line;
  int drawx1, drawx2, drawy1, drawy2;
  int left_x1, left_y1, left_x2, left_y2;
  int right_x1, right_y1, right_x2, right_y2;
  double xmin, xmax, ymin, ymax, zmin, zmax, proj;
  int index, zoffset;
  XSetWindowAttributes attributes;
  XGCValues XGCdummy;
  GC gc1;
  int *vertex;
  unsigned long valuemask = 0x003fffff;
  SpeechRecord *sd;
  struct itimerval interval;              /* timer to regulate drawing speed */

  /* Reduce drawable area slightly to allow for offsets */
  drawx1 = X_MARGIN_3D;
  drawx2 = window_width - X_MARGIN_3D;
  drawy1 = Y_MARGIN_3D;
  drawy2 = window_height - Y_MARGIN_3D;
  
  /* Automatic scaling to ensure trajectory fills projection area */
  s = utterance_index;
  sd = spdata[s];

  for (i = 0; i < sd->npoints; i++)
  {
    if (i == 0)
    {
      xmin = xmax = sd->project1[i];
      ymin = ymax = sd->project2[i];
      zmin = zmax = sd->project3[i];
      continue;
    }
    proj = sd->project1[i];
    if (proj > xmax)
      xmax = proj;
    else
      if (proj < xmin)
        xmin = proj;
    proj = sd->project2[i];
    if (proj > ymax)
      ymax = proj;
    else
      if (proj < ymin)
        ymin = proj;
    proj = sd->project3[i];
    if (proj > zmax)
      zmax = proj;
    else
      if (proj < zmin)
        zmin = proj;
  }

  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
    OriginAdjust(&zmin, &zmax);
  }

  /* find which vertices need to be highlighted */
  vertex = (int *) malloc(sd->npoints * sizeof(int));
  for (i = 0; i < sd->npoints; i++)
  {
    /* check that utterance has order array set */
    if (sd->nverts > 0)
    {
      for (j = 0; j < highlighted; j++)
        if (sd->order[j] == i)
          break;
      vertex[i] = (j < highlighted);
    }
    else
      vertex[i] = 0;
  }
  
  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = dim3_color[0].pixel;
  XChangeWindowAttributes(display,win,CWBackPixel,&attributes);
  XClearWindow(display,win);

  /* Draw the origin if required */
  if (display_origin_mode)
  {
    x = (short) nint(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
    y = (short) nint(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    zoffset = (short) nint(DEPTH_SCALING * (zmin / (zmin - zmax)));
    DrawOrigin3(x, y, zoffset);
  }

  /* create duplicate GC to avoid XSetForeground calls */
  gc1 = XCreateGC(display, win, 0, &XGCdummy);
  XCopyGC(display, gc, valuemask, gc1);
  XSetForeground(display, gc, dim3_color[left_color].pixel);
  XSetForeground(display, gc1, dim3_color[right_color].pixel);

  /* Setup the interval timer and register a (empty) handler */
  interval.it_interval.tv_sec = 0;
  interval.it_interval.tv_usec = sleep_period;
  interval.it_value = interval.it_interval;
  setitimer(ITIMER_REAL, &interval, NULL);
  signal(SIGALRM, AlarmHandler);

  right_x1 = 0;
  right_y1 = 0;
  left_x1 = 0;
  left_y1 = 0;

  for (i = 0; i < sd->npoints; i++)
  {
    /*
      draw trajectories using appropriate scaling to fill the rectangle
      specified by drawx1, drawx2, drawy1 and drawy2.
    */
    x = (short) nint((sd->project1[i] - xmin) * (drawx2 - drawx1) /
                                                       (xmax - xmin)) + drawx1;
    y = (short) nint((sd->project2[i] - ymin) * (drawy2 - drawy1) /
                                                       (ymax - ymin)) + drawy1;
    zoffset = (short) nint(DEPTH_SCALING * ((sd->project3[i] - zmin) /
                                   (zmax - zmin)));
    /* left eye projection */
    left_x1 = left_x2;
    left_x2 = x - zoffset;
    left_y1 = left_y2;
    left_y2 = y;

    /* right eye projection */
    right_x1 = right_x2;
    right_x2 = x + zoffset;
    right_y1 = right_y2;
    right_y2 = y;

    if (i == 0)
    {
      if (label)
      {
        /* Draw "Start" labels */
        XDrawString(display, win, gc,  left_x2,  left_y2,  "Start", 5);
        XDrawString(display, win, gc1, right_x2, right_y2, "Start", 5);
      }
    }
    else
    {
      /* Draw left and right lines */
      XDrawLine(display, win, gc,  left_x1,  left_y1,  left_x2,  left_y2);
      XDrawLine(display, win, gc1, right_x1, right_y1, right_x2, right_y2);

      /* highlight vertex */
      if (vertex[i])
      {
        XDrawArc(display, win, gc,  left_x2 - 4,  left_y2 - 4,  8,8,0, 23040);
        XDrawArc(display, win, gc1, right_x2 - 4, right_y2 - 4, 8,8,0, 23040);
      }

      XFlush(display);        /* Ensures updates are displayed on the server */
    }
    (void) sigpause(0);    /* wait for timer to ellapse generating interrupt */
  }

  /* Switch off the interval timer */
  interval.it_interval.tv_sec = 0;
  interval.it_interval.tv_usec = 0;
  interval.it_value.tv_sec = 0;
  interval.it_value.tv_usec = 0;
  setitimer(ITIMER_REAL, &interval, NULL);
  signal(SIGALRM, SIG_DFL);

  if (label)
  {
    /* Draw "End" labels */
    XDrawString(display, win, gc,  left_x2,  left_y2,  "End", 3);
    XDrawString(display, win, gc1, right_x2, right_y2, "End", 3);

    if (display_label_mode)
    {
      /* Draw trace label */
      XSetForeground(display, gc, traj_color[BLUE].pixel);
      base_line = window_height - 10;
      XDrawString(display,win, gc, 10, base_line, sd->file_name,
                  sd->file_name_length);
    }
  }
  free(vertex);
}

/*****************************************************************************/

/* Draw 2D graphics */
void draw_2d_graphics(unsigned int window_width, unsigned int window_height,
                   int label)
{
  int i,s;
  int width,height,base_line;
  XPoint *points;
  double scale;
  double xmin, xmax, ymin, ymax, proj;
  int index, x, y;
  int drawx1, drawx2, drawy1, drawy2;
  XSetWindowAttributes attributes;
  SpeechRecord *sd;

  /* set screen background to white */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);

  /* Automatic scaling to ensure trajectory fills projection area */
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    for (i = 0; i < sd->npoints; i++)
    {
      if (s == 0 && i == 0)
      {
        xmin = xmax = sd->project1[i];
        ymin = ymax = sd->project2[i];
        continue;
      }
      proj = sd->project1[i];
      if (proj > xmax)
        xmax = proj;
      else
        if (proj < xmin)
          xmin = proj;
      proj = sd->project2[i];
      if (proj > ymax)
        ymax = proj;
      else
        if (proj < ymin)
          ymin = proj;
    }
  }

  /* Adjust the auto_scale range so that the origin is visable */
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
  }
  
  if (auto_scale)
  {
    drawx1 = 20;
    drawx2 = window_width - 20;
    drawy1 = 20;
    drawy2 = window_height - 20;    
  }
  else
  {
    width = window_width;
    height = window_height;
    scale = exp(((double)size) * 0.1);
  }

  XClearWindow(display, win);
  for (s = 0; s < samples; s++) {
    sd = spdata[s];
    points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));
    
    for (i = 0; i < sd->npoints; i++) {
      if (auto_scale)
      {
        /*
          draw trajectories using appropriate scaling to fill the rectangle
          specified by drawx1, drawx2, drawy1 and drawy2.
        */
        points[i].x = (short) nint((sd->project1[i] - xmin) *
                                   (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
        points[i].y = (short) nint((sd->project2[i] - ymin) *
                                   (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
      }
      else
      {
        /*
          Absolute scaling using enlargement and left-right up-down buttons
        */
        points[i].x = (short) ((sd->project1[i] - xmin) * scale) + left_right;
        points[i].y = (short) ((sd->project2[i] - ymin) * scale) + up_down;
      }
    }
    XSetForeground(display, gc, traj_color[sd->color].pixel);
    XDrawLines(display,win, gc, points, sd->npoints, CoordModeOrigin);
    if (label)
    {
      XDrawString(display, win, gc, points[0].x, points[0].y, "Start", 5);
      XDrawString(display, win, gc, points[sd->npoints - 1].x,
                  points[sd->npoints - 1].y, "End", 3);
      if (display_label_mode)
      {
        base_line = window_height - (samples * 12);
        XDrawString(display, win, gc, 10, (base_line + (s * 12)),
                    sd->file_name, sd->file_name_length);
      }
    }

    /* print highlighted vertices */
    if (highlighted && sd->nverts)
    {
      for (i = 0; i < highlighted; i++)
      {
        index = sd->order[i];
        XDrawArc(display, win, gc, points[index].x - 4, points[index].y - 4,
                 8, 8, 0, 23040);
      }
    }
    free(points);
  }

  /* Draw the origin if appropriate */
  if (display_origin_mode)
  {
    if (auto_scale)
    {
      x = nint(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
      y = nint(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    }
    else
    {
      x = nint((-xmin * scale) + left_right);
      y = nint((-ymin * scale) + up_down);
    }
    DrawOrigin2(x, y);
  }
}

/*****************************************************************************/

/* Slowly draw 2D graphics */
void draw_2d_sweep(unsigned int window_width, unsigned int window_height,
                int label, unsigned int sleep_period)
{
  int i, j, s;
  int width,height,base_line;
  int x, y, x1, y1, x2, y2;
  double scale;
  double xmin, xmax, ymin, ymax, proj;
  int points[MAX_SAMPLES], max_points;
  int drawx1, drawx2, drawy1, drawy2;
  int *vertex[MAX_SAMPLES];
  XSetWindowAttributes attributes;
  SpeechRecord *sd;
  struct itimerval interval;              /* timer to regulate drawing speed */

  /* set screen background to white */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);

  /* Automatic scaling to ensure trajectory fills projection area */
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    for (i = 0; i < sd->npoints; i++)
    {
      if (s == 0 && i == 0)
      {
        xmin = xmax = sd->project1[i];
        ymin = ymax = sd->project2[i];
        continue;
      }
      proj = sd->project1[i];
      if (proj > xmax)
        xmax = proj;
      else
        if (proj < xmin)
          xmin = proj;
      proj = sd->project2[i];
      if (proj > ymax)
        ymax = proj;
      else
        if (proj < ymin)
          ymin = proj;
    }
  }

  /* Adjust the auto_scale range so that the origin is visable */
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
  }

  /* pre-compute a list of the vertices that need to be highlighted */
  max_points = 0;
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    points[s] = sd->npoints;
    vertex[s] = (int *) malloc(points[s] * sizeof(int));
    for (i = 0; i < points[s]; i++)
    {
      /* check that utterance has order array set */
      if (sd->nverts > 0)
      {
        for (j = 0; j < highlighted; j++)
          if (sd->order[j] == i)
            break;
        vertex[s][i] = (j < highlighted);
      }
      else
        vertex[s][i] = 0;
    }
    if (points[s] > max_points)
      max_points = points[s];
  }
  
  if (auto_scale)
  {
    drawx1 = 20;
    drawx2 = window_width - 20;
    drawy1 = 20;
    drawy2 = window_height - 20;    
  }
  else
  {
    width = window_width;
    height = window_height;
    scale = exp(((double)size) * 0.1);
  }

  XClearWindow(display,win);

  /* Firstly draw the origin if appropriate */
  if (display_origin_mode)
  {
    if (auto_scale)
    {
      x = nint(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
      y = nint(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    }
    else
    {
      x = nint((-xmin * scale) + left_right);
      y = nint((-ymin * scale) + up_down);
    }
    DrawOrigin2(x, y);
  }

  /* Setup the interval timer and register a (empty) handler */
  interval.it_interval.tv_sec = 0;
  interval.it_interval.tv_usec = sleep_period;
  interval.it_value = interval.it_interval;
  setitimer(ITIMER_REAL, &interval, NULL);
  signal(SIGALRM, AlarmHandler);
  
  for (i = 0; i < max_points; i++)
  {
    for (s = 0; s < samples; s++)
    {
      sd = spdata[s];
      
      /* only consider samples that have more than i points */
      if (i < sd->npoints)
      {
        if (auto_scale)
        {
          /*
            draw trajectories using appropriate scaling to fill the rectangle
            specified by drawx1, drawx2, drawy1 and drawy2.
          */
          if (i > 0)
          {
            x1 = (short) nint((sd->project1[i-1] - xmin) * (drawx2 - drawx1) /
                              (xmax - xmin)) + drawx1;
            y1 = (short) nint((sd->project2[i-1] - ymin) * (drawy2 - drawy1) /
                              (ymax - ymin)) + drawy1;
          }
          x2 = (short) nint((sd->project1[i] - xmin) * (drawx2 - drawx1) /
                            (xmax - xmin)) + drawx1;
          y2 = (short) nint((sd->project2[i] - ymin) * (drawy2 - drawy1) /
                            (ymax - ymin)) + drawy1;
        }
        else
        {
          /*
            Absolute scaling using enlargement and left-right up-down buttons
          */
          if (i > 0)
          {
            x1 = (short) nint((sd->project1[i-1] - xmin) * scale) + left_right;
            y1 = (short) nint((sd->project2[i-1] - ymin) * scale) + up_down;
          }
          x2 = (short) nint((sd->project1[i] - xmin) * scale) + left_right;
          y2 = (short) nint((sd->project2[i] - ymin) * scale) + up_down;
        }
        
        XSetForeground(display, gc, traj_color[sd->color].pixel);
        if (i == 0)
        {
          if (label)
            XDrawString(display, win, gc, x2, y2 ,"Start",5);
        }
        else
        {
          XDrawLine(display, win, gc, x1, y1, x2, y2);
          if (vertex[s][i])
            XDrawArc(display, win, gc, x2 - 4, y2 - 4, 8, 8, 0, 23040);
          
          /* on final vertex */
          if (label && i == (points[s] - 1))
            XDrawString(display, win, gc, x2, y2, "End", 3);
        }
      }
    }
    XFlush(display);      /* ensure changes have been written to the display */
    (void) sigpause(0);    /* wait for timer to ellapse generating interrupt */
  }

  /* Switch off the interval timer */
  interval.it_interval.tv_sec = 0;
  interval.it_interval.tv_usec = 0;
  interval.it_value.tv_sec = 0;
  interval.it_value.tv_usec = 0;
  setitimer(ITIMER_REAL, &interval, NULL);
  signal(SIGALRM, SIG_DFL);

  /* label each of the traces */
  if (label && display_label_mode)
  {
    base_line = window_height - (samples * 12);
    for (s = 0; s < samples; s++)
    {
      sd = spdata[s];
      XSetForeground(display, gc, traj_color[sd->color].pixel);
      XDrawString(display, win, gc, 10, base_line, sd->file_name,
                  sd->file_name_length);
      base_line += 12;
    }
  }

  for (s = 0; s < samples; s++)
      free(vertex[s]);

  fflush(stdout);
}

/*****************************************************************************/

void draw_spectrogram(unsigned int window_width, unsigned int window_height,
                      int label)
{
  int i, t, j, s, w, x, y, index;
  int start_index, ngrams;
  int pix_height, pix_width;
  int y_base, base_line;
  char buf[100];
  SpeechRecord *sd;
  XSetWindowAttributes attributes;
  double d, max, min;
  int *vertex[MAX_SAMPLES];

  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = spec_color[0].pixel;
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);
  XClearWindow(display, win);

  start_index = utterance_index;
  ngrams = window_height / 220;

  if (!spectrogram_stretch_mode)
  {
    /* find a common pix_width across all the utterances being displayed */
    pix_width = 99999;
    for (i = 0; i < ngrams; i++)
    {
      s = start_index + i;
      if (s == samples)
        break;
      w = window_width / spdata[s]->npoints;
      if (w < pix_width)
        pix_width = w;
    }
  }

  /* pre-compute a list of the vertices that need to be highlighted */
  for (i = 0; i < ngrams; i++)
  {
    s = start_index + i;
    if (s == samples)
      break;

    sd = spdata[s];
    vertex[i] = (int *) malloc(sd->npoints * sizeof(int));
    for (t = 0; t < sd->npoints; t++)
    {
      /* check that utterance has order array set */
      if (sd->nverts > 0)
      {
        for (j = 0; j < highlighted; j++)
          if (sd->order[j] == t)
            break;
        vertex[i][t] = (j < highlighted);
      }
      else
        vertex[i][t] = 0;
    }
  }

  /* loop to num of spectrograms that can be drawn within the window height */
  y_base = 0;
  for (i = 0; i < ngrams; i++)
  {
    s = start_index + i;
    if (s == samples)
      break;
    sd = spdata[s];

    if (spectrogram_stretch_mode)
      pix_width = window_width / sd->npoints;

    /* find range of values associated with the utterance */
    max = min = sd->point[0][0];
    for (t = 0; t < sd->npoints; t++)
      for (j = 0; j < dimension; j++)
      {
        d = sd->point[j][t];
        if (d > max)
          max = d;
        else
          if (d < min)
            min = d;
      }

    pix_height = 180 / dimension;
    
    /* draw spectrogram */
    x = 0;
    for (t = 0; t < sd->npoints; t++)
    {
      y = y_base;
      for (j = 0; j < dimension; j++)
      {
        index = (int) floor(((double) pix_height + 0.99) *
                            (sd->point[dimension-1-j][t] - min) / (max - min));
        assert(index >= 0 && index <= pix_height);
        XSetForeground(display, gc, spec_color[index].pixel);
        XFillRectangle(display, win, gc, x, y, pix_width, pix_height);
        y += pix_height;
      }

      if (vertex[i][t])
      {
        XSetForeground(display, gc, spec_color[15].pixel);
        XDrawLine(display, win, gc, x, y_base, x, y_base + 180);
      }
      x += pix_width;
    }
    
    /* Draw trace label */
    XSetForeground(display, gc, spec_color[15].pixel);
    base_line = y_base + 200;
    XDrawString(display, win, gc, 10, base_line, sd->file_name,
                sd->file_name_length);

    y_base += 220;
  }  

  for (i = 0; i < ngrams; i++)
      free(vertex[i]);
}

/*****************************************************************************/

/* end of projectors.c */
