/*
  title:   callbacks.c (callback routines for the XView widgets).
  purpose: To display two and three dimensional projections of speech
    trajectories through a high dimensional filter-band space.  With
    bindings to an Xview interface defined in interface.c.

  author:   Gareth Lee.
  date:     16-08-93
  modified: 01-11-93

  changes:
  07-10-93: The SpecifyWindow has been rationalized with the removal of
    several SpecifyXXXCallback() routines.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>
#include <assert.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>

/* Header files associated with other compilation units */
#include "projectors.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"

/* in entropy_seg.c */
extern double* prediction_error;
extern int entropy_verbose;

extern XFontStruct *font_info;

/*****************************************************************************/

void SamplesCallback(void)
{
  xv_set(SampleFrame, XV_SHOW, TRUE, NULL);
  xv_set(ControlSamplesButton, PANEL_INACTIVE, TRUE, NULL);
}

/*****************************************************************************/

/*
  SampleDismissCallback: Dismiss button within the sample pop-up window.
*/
void SampleDismissCallback(void)
{
  xv_set(SampleFrame, XV_SHOW, FALSE, NULL);
  xv_set(ControlSamplesButton, PANEL_INACTIVE, FALSE, NULL);
}

/*****************************************************************************/

/*
  SampleListnameCallback: called when the listname field is changed. Searches
    along the SearchPath[] to see if the specified list file can be
    located.  If it can the actual path and filename are passed to
    Initialize otherwise an error notice is generated and the new name is
    not accepted.
*/
void SampleListnameCallback(Panel_item item, Event *event)
{
  int index;
  char *new_name, *new_path;
  FILE *fd;
  char path_name[1024];

  new_name = (char *)xv_get(SampleListnameText, PANEL_VALUE);
  if (!strcmp(list_fname, new_name))
    return;                    /* If name not really changed then do nothing */
  else
  {
    /* Check that the file can be read from some location along the path */
    for (index = 0; SearchPath[index] !=  NULL; index++)
    {
      (void) sprintf(path_name, "%s/%s", SearchPath[index], new_name);
      if ((fd = fopen(path_name, "r")) != NULL)
        break;
    }

    /* If it cannot be located then generate an error */
    if (fd == NULL)
    {
      (void)
        notice_prompt(SamplePanel, NULL,
                      NOTICE_FOCUS_XY, event_x(event), event_y(event),
                      NOTICE_MESSAGE_STRINGS,
                        "Error:",
                        "Cannot open list file.",
                        "New file name ignored.",
                        NULL,
                      NOTICE_BUTTON_NO, "Continue",
                      NULL);

      /* restore the previous (valid) listfile name */
      xv_set(SampleListnameText,
             PANEL_VALUE, list_fname,
             NULL);
      fclose(fd);
      return;
    }
    else
    {
      fclose(fd);
      new_path = SearchPath[index];
    }
  }
  
  /*
    free the previous sample filename list and remove all entries from the
    PANEL_LIST widget
  */
  for (index = 0; index < list_entries; index++)
  {
    free(list_names[index]);
    list_names[index] = NULL;
  }
  xv_set(SampleFileList,
         PANEL_LIST_DELETE_ROWS, 0, list_entries,
         NULL);
  
  /* Free the spdata list entries */
  for (index = 0; index < samples; index++)
  {
    /* free the color that the trajectory used and deallocate the structure */
    FreeColorEntry(spdata[index]->color);
    FreeSpeechRecord(spdata[index]);
    free(spdata[index]);
    spdata[index] = NULL;
  }

  /* load the list file and display the first entry etc. . */
  Initialize(new_name, new_path);
}

/*****************************************************************************/

/*
  SampleOptionsCallback: called by the check box within the Samples menu and
    traps the features option to see whether the Samples window should be
    expanded or contracted.  Adjusts the geometry accordingly.
*/
void SampleOptionsCallback(Panel_item item, int flags, Event *event)
{
  if (flags & 0x0002)
  {
    /* Expand the window */
    xv_set(SampleFrame,
           XV_LABEL, "List File & Front Ends",
           XV_WIDTH, 500,
           NULL);
    
    xv_set(SampleFormatList, XV_SHOW, TRUE, NULL);
    xv_set(SampleFrontendList, XV_SHOW, TRUE, NULL);
    xv_set(SampleUpdateButton, XV_SHOW, TRUE, NULL);
    xv_set(SampleParamsButton, XV_SHOW, TRUE, NULL);
  }
  else
  {
    /* Contract the window */
    xv_set(SampleFormatList, XV_SHOW, FALSE, NULL);
    xv_set(SampleFrontendList, XV_SHOW, FALSE, NULL);
    xv_set(SampleUpdateButton, XV_SHOW, FALSE, NULL);
    xv_set(SampleParamsButton, XV_SHOW, FALSE, NULL);

    xv_set(SampleFrame,
           XV_LABEL, "List File",
           XV_WIDTH, 260,
           NULL);
  }
}

/*****************************************************************************/

/*
  SampleListCallback:  Called when a the user selects deselects any items
    within the SampleList scrolling list widget.
*/  
int SampleListCallback(Panel_item item, char *string, Xv_opaque client_data,
                        Panel_list_op op, Event *event, int row)
{
  int i, index;
  int flags, info;
  char buf[1024];
  SpeechRecord *sd;

  flags = (int) xv_get(SampleCheckBox, PANEL_VALUE);
  info = flags & 0x0001;
  
  /* Four types of callback result depending on the user action */
  switch(op)
  {
  case PANEL_LIST_OP_SELECT:
    /*
      Add an extra utterance to the spdata list and increment samples.
      Refresh the display to include the new utterance.
    */
    if (samples == MAX_SAMPLES)
    {
      /* cannot select any more items from the list */
      xv_set(SampleFileList, PANEL_LIST_SELECT, row, FALSE, NULL);

      (void)
        notice_prompt(SamplePanel, NULL,
                      NOTICE_FOCUS_XY, event_x(event), event_y(event),
                      NOTICE_MESSAGE_STRINGS,
                        "Warning:",
                        "Unable to display more trajectories.",
                        "Selection ignored.",
                        NULL,
                      NOTICE_BUTTON_YES, "Continue",
                      NULL);
    }
    else
    {
      SetCursors(BusyCursor);
      sd = (SpeechRecord *) malloc(sizeof(SpeechRecord));
      if (OpenDataFile(list_names[row], sd, info))
      {
        /* A load error has occurred so proceed no further */
        free(sd);

        /* deselect item from the list */
        xv_set(SampleFileList, PANEL_LIST_SELECT, row, FALSE, NULL);
        
        (void)
          notice_prompt(SamplePanel, NULL,
                        NOTICE_FOCUS_XY, event_x(event), event_y(event),
                        NOTICE_MESSAGE_STRINGS,
                        "Error:",
                        "Unable to load utterance.",
                        "See concole output for more details.",
                        NULL,
                        NOTICE_BUTTON_YES, "Continue",
                        NULL);
        SetCursors(BasicCursor);
        return XV_OK;
      }

      /* If all samples previously unloaded then recreate vectors & widgets */
      if (samples == 0 && dimension > 0)
      {
        CreateGlobalVectors();
        ChangeBandWidgets();
        DisplayRandomizeCallback();
      }

      /* find the location to insert it within the spdata list */
      for (index = 0; index < samples; index++)
        if (spdata[index]->row > row)
          break;

      /* shift all other spdata entries up to make space for the insertion */
      samples++;
      for (i = samples; i > index; i--)
        spdata[i] = spdata[i-1];
        
      spdata[index] = sd;
      Compression(sd, compress_alpha);

      /* set additional fields in the SpeechRecord */
      sd->color = AllocateColorEntry();
      sd->row = row;

      /* Display newly loaded trajectory if in three_dimensions */
      if (display_mode == three_dims)
        utterance_index = index;
      
      Projection();
      RefreshGraphics();
    }
    break;

  case PANEL_LIST_OP_DESELECT:
    /*
      Remove an utterance to the spdata list and decrement samples.
      Refresh the display to remove the utterance.
    */
    if (samples == 1)
    {
      /* cannot select any fewer (zero) items from the list */
      xv_set(SampleFileList, PANEL_LIST_SELECT, row, TRUE, NULL);

      (void)
        notice_prompt(SamplePanel, NULL,
                      NOTICE_FOCUS_XY, event_x(event), event_y(event),
                      NOTICE_MESSAGE_STRINGS,
                        "Warning:",
                        "Must display at least one trajectory.",
                        "Deselection ignored.",
                        NULL,
                      NOTICE_BUTTON_YES, "Continue",
                      NULL);
    }
    else
    {
      /* find its location within the spdata list */
      for (index = 0; index < samples; index++)
        if (spdata[index]->row == row)
          break;

      /* Deallocate the structure */
      FreeColorEntry(spdata[index]->color);
      FreeSpeechRecord(spdata[index]);
      free(spdata[index]);
      spdata[index] = NULL;
      
      samples--;
      for (i = index; i < samples; i++)
        spdata[i] = spdata[i+1];

      /*
        Ensure that freshly deleted utterance was not being displayed in
        three_dimensions display mode.
      */
      if (display_mode == three_dims && utterance_index == index)
        utterance_index = 0;
      
      Projection();
      RefreshGraphics();
    }
    break;

  case PANEL_LIST_OP_VALIDATE:
    break;
  case PANEL_LIST_OP_DELETE:
    break;
  }

  SetCursors(BasicCursor);
  return XV_OK;
}


/*****************************************************************************/

/*
  SampleFormatCallback: Called when the user makes a new format selection from
    the scrolling list in the "Samples" window.
*/  
int SampleFormatCallback(Panel_item item, char* string, Xv_opaque client_data,
                        Panel_list_op op, Event* event, int row)
{
  FilterRecord *fptr;
  int Params_XV_SHOW;
  
  /* Four types of callback result depending on the user action */
  switch(op)
  {
  case PANEL_LIST_OP_SELECT:

    FormatFilter = (FilterRecord *) client_data;
      
    /* Ensure the selection does not clash with the current Frontend */
    if (FormatFilter->format != FrontendFilter->format)
    {
      /*
        Change FrontendFilter to the first entry that matches the current
        output format.
      */
      for (fptr = &Frontends; fptr != NULL; fptr = fptr->next)
        if (fptr->format == FormatFilter->format)
          break;

      if (fptr != NULL)
      {
        xv_set(SampleFrontendList, PANEL_LIST_SELECT, fptr->index, TRUE, NULL);
        FrontendFilter = fptr;
        XBell(display, 0);
      }
      else
      {
        printf("Error: failed to find any frontend accepting '%c' format\n",
               FormatFilter->format);
        Quit(-1);
      }
    }

    /*
      Copy the default list file name across into the Text widget but
      wait for the user to modify (or confirm by pressing return).
    */
    if (FormatFilter->def_list != NULL)
      xv_set(SampleListnameText, PANEL_VALUE, FormatFilter->def_list, NULL);

    /* Refresh the Params window if appropriate */
    Params_XV_SHOW = (int) xv_get(ParamsFrame, XV_SHOW);
    ParamsDismissCallback();   /* reset the widgets within the Params window */
    if (Params_XV_SHOW == TRUE)
      SampleParamsCallback();      /* reinitialise the Params window widgets */
    break;

  case PANEL_LIST_OP_DESELECT:
    break;
  case PANEL_LIST_OP_VALIDATE:
    break;
  case PANEL_LIST_OP_DELETE:
    break;
  }

  return XV_OK;
}

/*****************************************************************************/

/*
  SampleFrontendCallback: Called when the user makes a new front-end selection
    from the scrolling list in the "Samples" window.
*/  
int SampleFrontendCallback(Panel_item item, char* string,
                           Xv_opaque client_data, Panel_list_op op,
                           Event* event, int row)
{
  FilterRecord *fptr;
  int Params_XV_SHOW;
  
  /* Four types of callback result depending on the user action */
  switch(op)
  {
  case PANEL_LIST_OP_SELECT:
    fptr = (FilterRecord *) client_data;

    /* Ensure compliance with the frontend output format */
    if (fptr->format != FormatFilter->format)
    {
      xv_set(SampleFrontendList,
             PANEL_LIST_SELECT, FrontendFilter->index, TRUE,
             NULL);
      XBell(display, 0);
    }
    else
      FrontendFilter = fptr;

    /* Refresh the Params window if appropriate */
    Params_XV_SHOW = (int) xv_get(ParamsFrame, XV_SHOW);
    ParamsDismissCallback();        /* ensure the params window is dismissed */
    if (Params_XV_SHOW == TRUE)
      SampleParamsCallback();      /* reinitialise the Params window widgets */
    break;

  case PANEL_LIST_OP_DESELECT:
    break;
  case PANEL_LIST_OP_VALIDATE:
    break;
  case PANEL_LIST_OP_DELETE:
    break;
  }

  return XV_OK;
}

/*****************************************************************************/

/*
  SampleUpdateCallback: called by the "Update" button in the Samples window.
  Registers the work SampleUpdateWorkRroc procedure and arranges for
  xv_main_loop() to terminate.
*/
void SampleUpdateCallback()
{
  ExecWorkProc = SampleUpdateWorkProc;   /* register the WorkProc to execute */
  notify_stop();
}

/*****************************************************************************/

/*
  SampleUpdateWorkproc: work procedure associated with SampleUpdateCallback
    Resets SampleListnameText to the value currently being used (overwriting
    any speculative changes) and proceeds to reload all the selected entries
    in the SampleFileList but using the newly specified filter combination
    (or parameters).
*/
void SampleUpdateWorkProc(void)
{
  int s, i;
  int flags, info, row, dims;
  double *b1, *b2, *b3;
  SpeechRecord *spr[MAX_SAMPLES];

  /*
    Reset the list_fname since no change is required and set read only
    during the course of updates.  Also disable the FileList and UpdateButton
    widgets and others that could be dangerous during the update procedure.
  */
  xv_set(SampleListnameText,
         PANEL_VALUE, list_fname,
         PANEL_INACTIVE, TRUE,
         NULL);
  xv_set(SampleUpdateButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(DisplayRandomizeButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlAnimateButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlSweepButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFileList, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFormatList, PANEL_INACTIVE, TRUE, NULL);
  xv_set(SampleFrontendList, PANEL_INACTIVE, TRUE, NULL);
  XFlush(display);

  /* Check whether to display header informaion during reload */
  flags = (int) xv_get(SampleCheckBox, PANEL_VALUE);
  info = flags & 0x0001;

  /* Create a local copy of the current basis vectors */
  dims = dimension;
  b1 = (double *) malloc(dims * sizeof(double));
  b2 = (double *) malloc(dims * sizeof(double));
  b3 = (double *) malloc(dims * sizeof(double));
  if (b1 == NULL || b2 == NULL || b3 == NULL)
  {
    printf("Error: failed to malloc local basis vectors (b1/2/3)\n");
    Quit(-1);
  }
  for (i = 0; i < dimension; i++)
  {
    b1[i] = w1[i];
    b2[i] = w2[i];
    b3[i] = w3[i];
  }
  
  /* Allow the dimension to change when reloading */
  if (dimension != 0)
  {
    FreeGlobalVectors();
    dimension = 0;
  }

  /* Reload each of the current utterances */
  for (s = 0; s < samples; s++)
  {
    FreeSpeechRecord(spdata[s]);
    if (OpenDataFile(list_names[spdata[s]->row], spdata[s], info))
    {
      /* Load error occurred in data file so mark SpeechRecord */
      spdata[s]->npoints = 0;
    }
    else
      Compression(spdata[s], compress_alpha);

    spdata[s]->nverts = 0;               /* reset highlight previous results */

    notify_dispatch();                   /* activates a single notifier pass */
  }

  /*
    Recreate Global vectors and update widgets if *any* of the loads succeeded
  */
  if (dimension > 0)
  {
    CreateGlobalVectors();
    ChangeBandWidgets();
  }

  /* Check for any load errors */
  for (s = 0; s < samples; s++)
    spr[s] = spdata[s];
  for (s = 0, i = 0; s < samples; s++)
  {
    if (spr[s]->npoints != 0)
      spdata[i++] = spr[s];
    else
    {
      /*
        Deallocate the SpeechRecord. Note: The dynamic fields within a
        SpeechRecord are automatically freed by OpenDataFile() on error.
        It is therefore not necessary to call FreeSpeechRecord.
      */
      FreeColorEntry(spr[s]->color);
      xv_set(SampleFileList, PANEL_LIST_SELECT, spr[s]->row, FALSE, NULL);
      free(spr[s]);
    }
  }
  samples = i;

  /* dimension should never be zero whilst there are samples loaded */
  if (samples > 0 && dimension == 0)
  {
    printf("Error: samples = %d, dimension = %d\n", samples, dimension);  /**/
    Quit(-1);
  }
  assert(!(samples > 0 && dimension == 0));

  /* Initiate redisplay with previous basis vectors if dimension unchanged */
  if (samples > 0)
  {
    if (dims == dimension)
      for (i = 0; i < dimension; i++)
      {
        w1[i] = b1[i];
        w2[i] = b2[i];
        w3[i] = b3[i];
      }
    else
      Randomize();

    Projection();
  }
  RefreshGraphics();

  free(b1);
  free(b2);
  free(b3);

  /* Reinstate the widgets deactivated during updating */
  xv_set(SampleListnameText, PANEL_INACTIVE, FALSE, NULL);
  xv_set(DisplayRandomizeButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlAnimateButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlSweepButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleUpdateButton, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFileList, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFormatList, PANEL_INACTIVE, FALSE, NULL);
  xv_set(SampleFrontendList, PANEL_INACTIVE, FALSE, NULL);

  /* Ensure that utterance_index remains within range */
  if (utterance_index >= samples)
    utterance_index = samples - 1;
}

/*****************************************************************************/

/*
  SampleParamsCallback: called by the Params button within the Samples window
    to expose the Filter Parameters window.  Prior to expose it initialises
    the windows fields depending on the FormatFilter and FrontendFilter
    selections within the Samples window.
*/
void SampleParamsCallback(void)
{
  /* set FormatFilter and FrontendFilter name fields */
  xv_set(ParamsFormatNameText, PANEL_VALUE, FormatFilter->name, NULL);
  xv_set(ParamsFormatFilterText, PANEL_VALUE, FormatFilter->program, NULL);
  xv_set(ParamsFrontendNameText, PANEL_VALUE, FrontendFilter->name, NULL);
  xv_set(ParamsFrontendFilterText, PANEL_VALUE, FrontendFilter->program, NULL);

  /*
    Fill up the two scrolling lists with options from the OptionsRecord
    linked lists and setup defaults selections and the Value text fields.
  */
  if (FormatFilter->options != NULL)
  {
    SampleFillScrollList(ParamsFormatList, FormatFilter->options);
    
    /* Select the first field and setup the text widget for editing */
    xv_set(ParamsFormatList, PANEL_LIST_SELECT, 0, TRUE, NULL);
    xv_set(ParamsFormatValueText,
           PANEL_VALUE, FormatFilter->options->value,
           PANEL_CLIENT_DATA, FormatFilter->options,
           PANEL_READ_ONLY, FALSE,
           NULL);
  }
  if (FrontendFilter->options != NULL)
  {
    SampleFillScrollList(ParamsFrontendList, FrontendFilter->options);
    
    /* Select the first field and setup the text widget for editing */
    xv_set(ParamsFrontendList, PANEL_LIST_SELECT, 0, TRUE, NULL);
    xv_set(ParamsFrontendValueText,
           PANEL_VALUE, FrontendFilter->options->value,
           PANEL_CLIENT_DATA, FrontendFilter->options,
           PANEL_READ_ONLY, FALSE,
           NULL);
  }
  
  /* Expose the window */
  xv_set(ParamsFrame, XV_SHOW, TRUE, NULL);
  xv_set(SampleParamsButton, PANEL_INACTIVE, TRUE, NULL);
}

/*****************************************************************************/

/*
  UpdateOptionRecord: Update the value field of an OptionRecord to the new
    value given.
*/
void UpdateOptionRecord(Xv_opaque list, OptionRecord *optr, Xv_opaque text)
{
  char *cptr, buf[1024];
  
  /* Copy the new value back into an OptionRecord */
  free(optr->value);
  cptr = (char *) xv_get(text, PANEL_VALUE);
  optr->value = StringDup(cptr);

  /* Update the entry within the scrolling list */
  (void) sprintf(buf, "%s = %s", optr->name, optr->value);
  xv_set(list, PANEL_LIST_STRING, optr->index, buf, NULL);
}

/*****************************************************************************/

/*
  ParamsFormatCallback: Called when the user makes a format option selection
    from the scrolling list.
*/  
int ParamsFormatCallback(Panel_item item, char* string,
                         Xv_opaque client_data, Panel_list_op op,
                         Event* event, int row)
{
  OptionRecord *optr;
  
  /* Four types of callback result depending on the user action */
  switch(op)
  {
  case PANEL_LIST_OP_SELECT:
    /* Save previous value back into the OptionRecord */
    optr = (OptionRecord *) xv_get(ParamsFormatValueText, PANEL_CLIENT_DATA);
    assert(optr != NULL);
    UpdateOptionRecord(ParamsFormatList, optr, ParamsFormatValueText);

    /* Load a new value into the widget */
    optr = (OptionRecord *) client_data;
    xv_set(ParamsFormatValueText,
           PANEL_VALUE, optr->value,
           PANEL_CLIENT_DATA, optr,
           PANEL_READ_ONLY, FALSE,
           NULL);
    break;

  case PANEL_LIST_OP_DESELECT:
    break;
  case PANEL_LIST_OP_VALIDATE:
    break;
  case PANEL_LIST_OP_DELETE:
    break;
  }

  return XV_OK;
}

/*****************************************************************************/

/*
  ParamsFrontendCallback: Called when the user makes a front-end option
    selection from the scrolling list.
*/  
int ParamsFrontendCallback(Panel_item item, char* string,
                         Xv_opaque client_data, Panel_list_op op,
                         Event* event, int row)
{
  OptionRecord *optr;
  
  /* Four types of callback result depending on the user action */
  switch(op)
  {
  case PANEL_LIST_OP_SELECT:
    /* Save previous value back into the OptionRecord */
    optr = (OptionRecord *) xv_get(ParamsFrontendValueText, PANEL_CLIENT_DATA);
    assert(optr != NULL);
    UpdateOptionRecord(ParamsFrontendList, optr, ParamsFrontendValueText);

    optr = (OptionRecord *) client_data;
    xv_set(ParamsFrontendValueText,
           PANEL_VALUE, optr->value,
           PANEL_CLIENT_DATA, optr,
           PANEL_READ_ONLY, FALSE,
           NULL);
    break;

  case PANEL_LIST_OP_DESELECT:
    break;
  case PANEL_LIST_OP_VALIDATE:
    break;
  case PANEL_LIST_OP_DELETE:
    break;
  }

  return XV_OK;
}

/*****************************************************************************/

void ParamsFormatTextCallback(Panel_item item, Event event)
{
  OptionRecord *optr;

  optr = (OptionRecord *) xv_get(ParamsFormatValueText, PANEL_CLIENT_DATA);
  assert(optr != NULL);
  UpdateOptionRecord(ParamsFormatList, optr, ParamsFormatValueText);
}

/*****************************************************************************/

void ParamsFrontendTextCallback(Panel_item item, Event event)
{
  OptionRecord *optr;

  optr = (OptionRecord *) xv_get(ParamsFrontendValueText, PANEL_CLIENT_DATA);
  assert (optr != NULL);
  UpdateOptionRecord(ParamsFrontendList, optr, ParamsFrontendValueText);
}

/*****************************************************************************/

/*
  ParamsDismissCallback: called when the Filter Parameters window is dismissed.
*/
void ParamsDismissCallback(void)
{
  /* Hide the window, enable the exposure button */
  xv_set(ParamsFrame, XV_SHOW, FALSE, NULL);
  xv_set(SampleParamsButton, PANEL_INACTIVE, FALSE, NULL);

  /* Empty the two scrolling lists now that the window is hidden */
  SampleEmptyScrollList(ParamsFormatList);
  SampleEmptyScrollList(ParamsFrontendList);

  /* Set the "Value:" text widgets as empty and read only */
  xv_set(ParamsFormatValueText,
         PANEL_VALUE, "",
         PANEL_READ_ONLY, TRUE,
         NULL);
  xv_set(ParamsFrontendValueText,
         PANEL_VALUE, "",
         PANEL_READ_ONLY, TRUE,
         NULL);
}

/*****************************************************************************/

/*
  DisplaySliderCallback: Allows the degree of compression applied to the traces
    to be controlled.  At one extreme they are displayed as linear power,
    whilst at the other limit they display logged power.
*/
void DisplaySliderCallback(Panel_item item, int value, Event *event)
{
  int s;
  
  compress_alpha = 1.0 - ((double) value / 100.0);

  printf("Compression: alpha = %f\n", compress_alpha);
  fflush(stdout);

  if (!compress_raw_mode)
  {
    /* must recompress the data points then redisplay */
    for (s = 0; s < samples; s++)
      Compression(spdata[s], compress_alpha);
    
    Projection();
    RefreshGraphics();
  }
}

/*****************************************************************************/

/*
  DisplayCheckBoxCallback: Monitors changes in the display check box.  Updates
    global flags when any changes occur.
*/
void DisplayCheckBoxCallback(void)
{
  int s, flags;

  flags = (int) xv_get(DisplayCheckBox, PANEL_VALUE);

  /* Check Box options depend on display_mode */
  if (display_mode == spectrogram)
  {
    compress_raw_mode = flags & 0x0001;
    spectrogram_stretch_mode = flags & 0x0002;
  }
  else
  {
    compress_raw_mode = flags & 0x0001;
    compress_scale_mode = flags & 0x0002;
    display_label_mode = flags & 0x0004;
    display_origin_mode = flags & 0x0008;
  }

  /* must recompress the data points then redisplay */
  for (s = 0; s < samples; s++)
    Compression(spdata[s], compress_alpha);

  Projection();
  RefreshGraphics();
}

/*****************************************************************************/

/*
  DisplayRandomizeCallback: callback routine on Randomise window vectors
    button in the Display window.
*/
void DisplayRandomizeCallback(void)
{
  Randomize();
  Projection();
  RefreshGraphics();
}

/*****************************************************************************/

/*
  DisplayStoreRecallCallback: Callback routine for the View menu in the Display
    window.  Saves current viewing angle and allows it to be restored later.
*/
void DisplayStoreRecallCallback(Panel_item item, Event *event)
{
  static int dims = 0;

  int i;
  int id;

  /* ensure that dimension is set */
  if (dimension == 0)
    return;
  
  id = (int) xv_get(item, PANEL_CLIENT_DATA);
  switch (id)
  {
  case 0:
    /* Store Button */
    dims = dimension;
    printf("Storing Basis Vectors:\n");  /**/
    for (i = 0; i < dimension; i++)
    {
      stw1[i] = w1[i];
      stw2[i] = w2[i];
      stw2[i] = w2[i];
      printf("%02d %10.4f %10.4f %10.4f\n", i, w1[i], w2[i], w3[i]);  /**/
    }
    putchar('\n');   /**/
    fflush(stdout);  /**/
    break;
    
  case 1:
    /* Recall Button */
    if (dims != dimension)
    {
      (void)
        notice_prompt(MainPanel, NULL,
                      NOTICE_MESSAGE_STRINGS,
                        "Warning:",
                        "Unable to recall",
                        NULL,
                      NOTICE_BUTTON_YES, "Continue",
                      NULL);

      printf("Warning: \n");
      break;
    }
    
    printf("Recalling Basis Vectors:\n");  /**/
    for (i = 0; i < dimension; i++)
    {
      w1[i] = stw1[i];
      w2[i] = stw2[i];
      w2[i] = stw2[i];
      printf("%02d %10.4f %10.4f %10.4f\n", i, w1[i], w2[i], w3[i]);   /**/
    }
    putchar('\n');    /**/
    fflush(stdout);   /**/

    Projection();
    RefreshGraphics();
    break;
  }
}

/*****************************************************************************/

/*
  DisplayTrajectoryControls: Manages the widget changes needed when moving
    from trajectory display modes (2D or 3D) to the spectrogram display mode.
    NB: parameter is true when leaving spectrogram mode (false when entering).
*/
void DisplayTrajectoryControls(int out_of_spect_mode)
{
  int s, flags;

  if (out_of_spect_mode)
  {
    /* Make buttons active whilst not in spectrogram mode */
    xv_set(DisplayStoreButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(DisplayRecallButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(DisplayRandomizeButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(ControlAutoScaleButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(ControlAnimateButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(ControlSweepButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(ControlSpecifyButton, PANEL_INACTIVE, FALSE, NULL);
  }
  else
  {
    /* Make buttons inactive whilst in spectrogram mode */
    xv_set(DisplayStoreButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(DisplayRecallButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(DisplayRandomizeButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(ControlAutoScaleButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(ControlAnimateButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(ControlSweepButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(ControlSpecifyButton, PANEL_INACTIVE, TRUE, NULL);
  }

  /* Handle DisplayCheckBox widget */
  flags = 0x0000;
  if (out_of_spect_mode)
  {
    if (compress_raw_mode)               /* Trajectory display check options */
      flags |= 0x0001;
    if (compress_scale_mode)
      flags |= 0x0002;
    if (display_label_mode)
      flags |= 0x0004;
    if (display_origin_mode)
      flags |= 0x0008;
    xv_set(DisplayCheckBox,
           PANEL_CHOICE_STRINGS, "Raw", "Scaled", "Labels", "Origin", NULL,
           PANEL_VALUE, flags,
           NULL);
  }
  else
  {
    if (compress_raw_mode)              /* Spectrogram display check options */
      flags |= 0x0001;
    if (spectrogram_stretch_mode)
      flags |= 0x0002;
    xv_set(DisplayCheckBox,
           PANEL_CHOICE_STRINGS, "Raw", "Stretched", NULL,
           PANEL_VALUE, flags,
           NULL);
  }
}

/*****************************************************************************/

void DisplayArrowCallback(Panel_item item, Event *event)
{
  int id;

  id = (int) xv_get(item, PANEL_CLIENT_DATA);

  switch (id)
  {
  case 0:
    utterance_index--;
    if (utterance_index < 0)
    {
      utterance_index = 0;
      XBell(display, 0);
    }
    else
      RefreshGraphics();
    return;

  case 1:
    utterance_index++;
    if (utterance_index >= samples)
    {
      utterance_index = samples - 1;
      XBell(display, 0);
    }
    else
      RefreshGraphics();
    return;
  }
}

/*****************************************************************************/

/*
  DisplayDismissCallback: Temporary dismiss button within the display window.
*/
void DisplayDismissCallback(void)
{
  xv_set(DisplayFrame, XV_SHOW, FALSE, NULL);
}

/*****************************************************************************/

/*
  DisplayFormatCallback: Callback routine for the Format menu in the Display
    window.  Selects between two and three dimensional displays.
*/
void DisplayFormatCallback(Panel_item item, int value, Event *event)
{
  switch (value)
  {
  case 0:
    /* Entering two dimensional display mode */
    if (display_mode == two_dims)
      break;

    if (display_mode == spectrogram)
      DisplayTrajectoryControls(TRUE);

    display_mode = two_dims;

    xv_set(ControlAutoScaleButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(DisplayDecrButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(DisplayIncrButton, PANEL_INACTIVE, TRUE, NULL);

    RefreshGraphics();
    break;

  case 1:
    /* Entering three dimensional display mode */
    if (display_mode == three_dims)
      break;

    if (display_mode == spectrogram)
      DisplayTrajectoryControls(TRUE);
    if (display_mode == two_dims)
      utterance_index = 0;

    display_mode = three_dims;

    xv_set(ControlAutoScaleButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set(DisplayDecrButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(DisplayIncrButton, PANEL_INACTIVE, FALSE, NULL);

    /* Must return to auto_scale display mode for three dimensions */
    if (!auto_scale)
      AutoCallback();
    else
      RefreshGraphics();
    
    break;

  case 2:
    /* Entering spectrogram display mode */
    if (display_mode == spectrogram)
      break;
    else
      DisplayTrajectoryControls(FALSE);
    if (display_mode == two_dims)
      utterance_index = 0;

    display_mode = spectrogram;

    xv_set(DisplayDecrButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set(DisplayIncrButton, PANEL_INACTIVE, FALSE, NULL);

    RefreshGraphics();
    break;
  }
}

/*****************************************************************************/

/*
  Rotation[12]Callback: Obtains information about rotation planes.  Ensures
  rotational planes are kept independent of one and other.
*/
void Rotation1Callback(Panel_item item, int value, Event* event)
{
  if (value == rotation_dim2)
  {
    xv_set(item, PANEL_VALUE, rotation_dim1, NULL);
    XBell(display, 0);               /* indicate that an error has been made */
  }
  else
    rotation_dim1 = value;
}

void Rotation2Callback(Panel_item item, int value, Event* event)
{
  if (value == rotation_dim1)
  {
    xv_set(item, PANEL_VALUE, rotation_dim2, NULL);
    XBell(display, 0);               /* indicate that an error has been made */
  }
  else
    rotation_dim2 = value;
}

/*****************************************************************************/

/*
  SpecifyToggleCallback: ensures that at least one of the buttons within each
    toggle widget is always set. Note: can be called by any of the three
    widgets so uses previously attached PANEL_CLIENT_DATA to decide.
*/
void SpecifyToggleCallback(Panel_item item, int value, Event* event)
{
  static unsigned int flag1 = 1;  /* cf. values used in interface.c  */
  static unsigned int flag2 = 2;  /* xv_create() calls for              */
  static unsigned int flag3 = 4;  /* SpecifySelector[123] widgets.      */

  /* ascertain which of the Selector widgets has initiated the callback */
  switch ((int) xv_get(item, PANEL_CLIENT_DATA))
  {
  case 1:
    if (value == 0)
    {
      xv_set(SpecifySelector1, PANEL_VALUE, flag1, NULL);
      XBell(display, 0);
    }
    else
      flag1 = value;
    break;

  case 2:
    if (value == 0)
    {
      xv_set(SpecifySelector2, PANEL_VALUE, flag2, NULL);
      XBell(display, 0);
    }
    else
      flag2 = value;
    break;

  case 3:
    if (value == 0)
    {
      xv_set(SpecifySelector3, PANEL_VALUE, flag3, NULL);
      XBell(display, 0);
    }
    else
      flag3 = value;
    break;
  }
}

/*****************************************************************************/

void SpecifyDismissCallback(void)
{
  xv_set(SpecifyFrame, XV_SHOW, FALSE, NULL);
  xv_set(ControlSpecifyButton, PANEL_INACTIVE, FALSE, NULL);
}

/*****************************************************************************/

/*
  SpecifyProjectCallback: Finish button within the Specify pop-up window and
    dismiss handler for the entire window.
*/
void SpecifyProjectCallback(void)
{
  int i;

  /* set basis vector w1 */
  if (xv_get(SpecifyRandEq1Choice, PANEL_VALUE))
    RandomBasis(SpecifySelector1, w1);
  else
    DiagonalBasis(SpecifySelector1, w1);
  
  /* set basis vector w2 */
  if (xv_get(SpecifyRandEq2Choice, PANEL_VALUE))
    RandomBasis(SpecifySelector2, w2);
  else
    DiagonalBasis(SpecifySelector2, w2);
  
  /* set basis vector w3 */
  if (xv_get(SpecifyRandEq1Choice, PANEL_VALUE))
    RandomBasis(SpecifySelector3, w3);
  else
    DiagonalBasis(SpecifySelector3, w3);

  /*
    Ensures all basis vectors are orthogonal else generate a pop-up error
      window.
  */
  OrthogonalizeBases();
  
  Projection();
  RefreshGraphics();
  
  printf("New projection bases:\n");
  for (i = 0; i < dimension; i++) 
    printf("%f %f %f\n",w1[i], w2[i], w3[i]);
}

/*****************************************************************************/

/*
  AngleSliderCallback: Sets step angle for subsequent animation sequences.
*/
void AngleSliderCallback(Panel_item item, int value, Event *event)
{
  const double MinAngle = 0.01;   /* radians */
  const double MaxAngle = 0.10;

  angle = MinAngle + (((double) value) / 1000.0 * (MaxAngle - MinAngle));
}

/*****************************************************************************/

void IterationSliderCallback(Panel_item item, int value, Event *event)
{
  iterations = value;
}

/*****************************************************************************/

/*
  AnimateCallback:  Generate an animated sequence of projections of the
    trajectory rotating around dimensions dim1 and dim2.  Refuse to scan
    when no samples are loaded.
*/
void AnimateCallback(void)
{
  int i, j, t;
  int frames;
  double *y1, *y2;
  double **matrix;

  if (samples == 0)
    return;

  SetCursors(BusyCursor);
  
  /* Allocate dynamic variables */
  y1 = (double *) malloc(dimension * sizeof(double));
  y2 = (double *) malloc(dimension * sizeof(double));
  matrix = (double **) malloc(dimension * sizeof(double *));
  for (i = 0; i < dimension; i++)
    matrix[i] = (double *) malloc(dimension * sizeof(double));
  
  /* Ensure the drawing window is at the front before starting animation */
  xv_set(DisplayFrame, XV_SHOW, TRUE, NULL);
  
  frames = nint((M_PI * 2.0 * (double) iterations) / (angle * 1000.0));
  printf("Animation of %d frames: ", frames);

  /* identity matrix */
  for (i = 0; i < dimension; i++)
    for (j = 0; j < dimension; j++)
      matrix[i][j] = (i == j) ? 1.0 : 0.0;

  matrix[rotation_dim1][rotation_dim1] = cos(angle);
  matrix[rotation_dim2][rotation_dim2] = cos(angle);
  matrix[rotation_dim1][rotation_dim2] = sin(angle);
  matrix[rotation_dim2][rotation_dim1] = -sin(angle);

  for (t = 0; t < frames; t++)
  {
    /* Calculates new window vectors */
    for (i = 0; i < dimension; i++)
    {
      y1[i] = 0.0;
      y2[i] = 0.0;
      for (j = 0; j < dimension; j++)
      {
        y1[i] += matrix[i][j] * w1[j];
        y2[i] += matrix[i][j] * w2[j];
      }
    }
    for (i = 0; i < dimension; i++) {
      w1[i] = y1[i];
      w2[i] = y2[i];
    }

    Projection();

    /* draw without labelling except on final iteration */
    switch (display_mode)
    {
    case two_dims:
      draw_2d_graphics(width, height, t == (frames-1));
      break;

    case three_dims:
      draw_3d_graphics(width, height, t == (frames-1));
      break;

    case spectrogram:
      /*
        This situation should never occur since the button is set inactive
        whilst display_mode == spectrogram.
        */
      assert(display_mode != spectrogram);
      break;
    }

    if ((t % 50) == 0)
      printf("%d ", t);
  }
  printf(": done.\n");

  for (i = 0; i < dimension; i++)
    free(matrix[i]);
  free(matrix);
  free(y2);
  free(y1);

  SetCursors(BasicCursor);
}

/*****************************************************************************/

void ViewSliderCallback(void)
{
  size = (int) xv_get(EnlargeShrinkSlider, PANEL_VALUE) + SIZE_OFFSET;
  up_down = UP_DOWN_OFFSET - (int) xv_get(UpDownSlider, PANEL_VALUE);
  left_right = (int) xv_get(LeftRightSlider, PANEL_VALUE) + LEFT_RIGHT_OFFSET;

  RefreshGraphics();
}

/*****************************************************************************/

/*
  AutoCallback: called by the "AustoScale" button in the control menu.
    Refuses to operate when no samples are loaded.
*/
void AutoCallback(void)
{
  if (samples == 0)
    return;
  
  if (auto_scale)
  {
    auto_scale = 0;
    xv_set(ciips_icon, XV_SHOW, FALSE, NULL);
    xv_set(UpDownSlider, XV_SHOW, TRUE, NULL);
    xv_set(LeftRightSlider, XV_SHOW, TRUE, NULL);
    xv_set(EnlargeShrinkSlider, XV_SHOW, TRUE, NULL);
  }
  else
  {
    auto_scale = 1;
    xv_set(UpDownSlider, XV_SHOW, FALSE, NULL);
    xv_set(LeftRightSlider, XV_SHOW, FALSE, NULL);
    xv_set(EnlargeShrinkSlider, XV_SHOW, FALSE, NULL);
    xv_set(ciips_icon, XV_SHOW, TRUE, NULL);
  }
  RefreshGraphics();
}

/*****************************************************************************/

void SpecifyCallback(void)
{
  xv_set(SpecifyFrame, XV_SHOW, TRUE, NULL);
  xv_set(ControlSpecifyButton, PANEL_INACTIVE, TRUE, NULL);
}

/*****************************************************************************/

void HighlightCallback(void)
{
  xv_set(HighlightFrame, XV_SHOW, TRUE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, TRUE, NULL);
}

/*****************************************************************************/

void HighlightSliderCallback(Panel_item item, int value, Event *event)
{
  /* obtain new number of vertices to be highlighted */
  highlighted = value;

  RefreshGraphics();
}

/*****************************************************************************/

/*
  HighlightSelectorCallback: called by the "First", "Second", "Third" order
    selector in Highlight window.  Activates the appropriate predictive
    entropy function and copies the results back into the SpeechRecord.
*/
void HighlightSelectorCallback(Panel_item item, int value, Event *event)
{
  int s, i, j;
  unsigned int mode, direction, velocity, verbose;
  SpeechRecord *sd;

  /* extract mode information from the Options Check Box */
  mode = (unsigned int) xv_get(HighlightCheckBox, PANEL_VALUE);
  direction = mode & 0x0001;
  velocity = mode & 0x0002;
  verbose = mode & 0x0004;

  entropy_verbose = verbose;            /* set external to limit diagnostics */
  
  /* for each of the traces */
  SetCursors(BusyCursor);
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    
    switch (value)
    {
    case 0:
      sd->nverts =
        find_segmentation(sd->npoints, dimension, sd->data, sd->order,
                          direction, velocity);
      break;

    case 1:
      sd->nverts =
        find_diff_segmentation(sd->npoints, dimension, sd->data, sd->order,
                               direction, velocity);
      break;

    case 2:
      sd->nverts =
        find_diff2_segmentation(sd->npoints, dimension, sd->data, sd->order,
                                direction, velocity);
      break;
    }
    memcpy(sd->dist, prediction_error, sd->nverts*sizeof(double));
  }

  if (max_vertices == 0)
  {
    max_vertices = spdata[0]->nverts;
    for (s = 1; s < samples; s++)
    {
      sd = spdata[s];
      if (sd->npoints < max_vertices)
        max_vertices = sd->nverts;
    }
    printf("Maximum vertices that can be displayed = %d\n", max_vertices);

    highlighted = max_vertices / 10;    /* num vertices to initially display */
  }

  /* Reset slider to initial position and set maximum range */
  xv_set(HighlightSlider,
         PANEL_MAX_VALUE, max_vertices,
         PANEL_VALUE, highlighted,
         PANEL_INACTIVE, FALSE,
         NULL);

  SetCursors(BasicCursor);
  RefreshGraphics();
}

/*****************************************************************************/

void HighlightDismissCallback(void)
{
  xv_set(HighlightFrame, XV_SHOW, FALSE, NULL);
  xv_set(ControlHighlightButton, PANEL_INACTIVE, FALSE, NULL);
}

/*****************************************************************************/

/*
  QuitCallback: Seek confirmation and then exit after clearing up.
*/
void QuitCallback(void)
{
  int reply;
  
  /* Create notice asking for confirmation */
  reply =
    notice_prompt(MainPanel, NULL,
                  NOTICE_MESSAGE_STRING, "Do you really want to quit?",
                  NOTICE_BUTTON_YES, "Yes",
                  NOTICE_BUTTON_NO,  "No",
                  NULL);

  /* If confirmation is positive then call lowest level Quit() */
  if (reply == NOTICE_YES)
    Quit(-1);
}

/*****************************************************************************/

/*
  SweepGraphicsCallback: Controls the drawing of trajectory sweeps in two and
    three dimensions.  Refuse to sweep when no samples are loaded.
*/
void SweepGraphicsCallback()
{
  const int SweepScale = 1000;        /* convert sleep time per vertex to ms */

  int sweep_rate;

  if (samples == 0)
    return;
  
  /* Ensure that the drawing window is at the front before starting sweep */
  xv_set(DisplayFrame, XV_SHOW, TRUE, NULL);

  SetCursors(BusyCursor);
  
  /* obtain Canvas dimensions */
  width = (int) xv_get(DisplayCanvas, XV_WIDTH);
  height = (int) xv_get(DisplayCanvas, XV_HEIGHT);
  sweep_rate = (int) xv_get(SweepRateSlider, PANEL_VALUE);

  /*
    Choose correct drawing function depending on three_dimensional and
    sweep_rate flags.
  */
  switch (display_mode)
  {
  case two_dims:
    if (sweep_rate)
      draw_2d_sweep(width, height, 1, sweep_rate * SweepScale);
    else
      draw_2d_graphics(width, height, 1);
    break;

  case three_dims:
    if (sweep_rate)
      draw_3d_sweep(width, height, 1, sweep_rate * SweepScale);
    else
      draw_3d_graphics(width, height, 1);
    break;

  case spectrogram:
    /*
      This situation should never occur since the button is set inactive
      whilst display_mode == spectrogram.
    */
    assert(display_mode != spectrogram);
    break;
  }
  SetCursors(BasicCursor);
}

/*****************************************************************************/

/* end of callbacks.c */
