/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int UTdecOption (const char *argv[], int N, const char *OptTable[],
                   const char *OptArg[], int *incr)

Purpose:
  Decode command line options

Description:
  This routine decodes command line options, returning any option values.  It
  looks at the first item addressed by argv to check if it matches a table of
  option keywords.  If the option requires a value and the value follows in the
  item addressed by the second element of argv, this item is also used.  The
  table of key words contains entries which can specify the minimum length
  needed for a match with an optional '*' character.
    "-d" -	specifies that an exact match to "-d" is required.  The '-'
		character is just treated as an ordinary character for the
		match.
    "--d*elay" - specifies that the option must match the first two characters
		and optionally match additional characters.  For instance,
		"--d" matches as does "--del", but not "--dx".
  An second '*' in the keyword string can be used to indicate that characters
  in the option string after that point need not match.

  The need for an option value is signalled in two ways.
    "-d**:" -	The presence of the trailing "**:" indicates that the option
		can either directly follow the option as in "-d30" or in the
		next argument as in "-d" followed by "30" in the next argument.
    "--d*elay=* - The presence of the trailing "=*" indicates that the option
		can either follow the option, separated by a '=' character as
		in "--del=30" or in the next argument as in "--del" followed by
		"30" in the next argument.

Parameters:
  <-  int
  UTdecOption -	Status code,
		-2, Option error, value required
		 0, Not a recognized option
		>= 1, Option number
  ->  const char * []
  argv -	Pointers to option/argument strings
  ->  int
  N -		Number of argument strings in argv.  This value is assumed to
                be at least one.
  ->  const char * []
  OptTable -	Pointers to the option keywords.  The end of the keyword table
		is signalled by a NULL pointer.
  <-  const char * []
  OptArg -	Argument/option string.  For an argument or an option taking a
		value, this is a pointer to null terminated string in argv.  As
		such it should be treated as read only.  If decoded option does
		not take a value, this pointer will be NULL.
  <-  int *
  incr -	Number of arguments processed.  This value is normally one.  It
		is two if the option has a value in the next argument.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.11 $  $Date: 1994/02/24 20:55:16 $

-------------------------------------------------------------------------*/

static char rcsid[]="$Id: UTdecOption.c 1.11 1994/02/24 AFsp-V1R2 $";

#include <stddef.h>		/* NULL */
#include <string.h>
#include <libtsp.h>
#include <libtsp/UTnucleus.h>

#ifndef __STDC__
#define	const
#endif

#define ST_NOMATCH	0
#define	ST_MVAL		-2

int
UTdecOption (argv, N, OptTable, OptArg, incr)

     const char *argv[];
     int N;
     const char *OptTable[];
     const char *OptArg[];
     int *incr;

{
  int n;
  int imbe;
  char *pe;
  char *token;
  int nc;
  int m;

/* Look for a match to an option */
  n = STkeyMatch (argv[0], OptTable);

/* Look for an imbedded '=' */
  imbe = 0;
  pe = strchr(argv[0], '=');
  if (pe != NULL)
    imbe = 1;

/* Second try, match to the string before the '=' */
  if (n < 0 && imbe != 0) {
    nc = pe - argv[0];
    token = (char *) UTmalloc (nc + 1);
    strncpy (token, argv[0], nc);
    token[nc] = '\0';
    n = STkeyMatch (token, OptTable);
    UTfree (token);
  }

  *incr = 1;
  ++n;		/* No match is now n == 0 */
  if (n > 0) {

/* Found an option match */
/* Check if the option needs an argument */
    m = strlen (OptTable[n-1]);
    if (m >= 3 && strcmp ("**:", &OptTable[n-1][m-3]) == 0) {

/* Short option with a value */
      nc = m - 3;
      if (strlen (argv[0]) > (unsigned) nc) {
	*OptArg = &argv[0][nc];
      }
      else if (N >= 2) {
	*OptArg = argv[1];
	*incr = 2;
      }
      else {
	*OptArg = NULL;
	n = ST_MVAL;
      }
    }

    else if (m >= 2 && strcmp ("=*", &OptTable[n-1][m-2]) == 0) {

/* Long option with a value */
      if (imbe != 0) {
	*OptArg = pe + 1;
      }
      else if (N >= 2) {
	*OptArg = argv[1];
	*incr = 2;
      }
      else {
	*OptArg = NULL;
	n = ST_MVAL;
      }
    }

    else {

/* No value allowed */
      *OptArg = NULL;
      if (imbe != 0)	/* Reject arguments with an embedded '=' */
	n = ST_NOMATCH;
    }
  }

  return n;
}
