/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int UTgetOption (int *State, int argc, const char *argv[],
                   const char *OptTable[], const char *OptArg[])

Purpose:
  Decode command line options and arguments

Description:
  This routine decodes command line options, returning any option values and
  other (non-option) argument strings.  Typically this routine is used to
  decode Unix style command options and to return argument strings, as in the
  following example.
    command -x -f name --delay=30 argA argB.
  Each of the items in the command line, including the command itself, appears
  in an array of strings.  This routine compares the elements of the command
  line after the command name for a match to a table of option keywords.
  Options normally have a leading '-' character.

  The table of keywords contains entries which can specify the minimum length
  needed for a match with an optional '*' character.
    "-d" -	specifies that an exact match to "-d" is required.  The '-'
		character is just treated as an ordinary character for the
		match.
    "--d*elay" - specifies that the option must match the first two characters
		and optionally match additional characters.  For instance,
		"--d" matches as does "--del", but not "--dx".
  An second '*' in the keyword string can be used to indicate that characters
  in the option string after that point need not match.

  The need for an option value is signalled in two ways.
    "-d**:" -	The presence of the trailing "**:" indicates that the option
		can either directly follow the option as in "-d30" or in the
		next argument as in "-d" followed by "30" in the next argument.
    "--d*elay=* - The presence of the trailing "=*" indicates that the option
		can either follow the option, separated by a '=' character as
		in "--del=30" or in the next argument as in "--del" followed by
		"30" in the next argument.


  Errors detected by this routine result in an error message being printed and
  an error code being returned.
  - An invalid option:  If the first option in the list of option keywords
    starts with a '-' character, then any element with a leading '-' and not
    matching any of the options causes an error condition.
  - Missing option value.  If an option requires a value, and none is supplied,
    an error is reported.  Such an error can only happen if the option is the
    last item in the list of arguments, since in other cases the item after
    the option will be picked up as the option value.

  The results of the option scan are communicated to the calling program via
  the pointer OptArg.  OptArg is a pointer to a pointer to a null terminated
  strings (substrings of the strings from argv).  The interpretation of these
  strings is given by the return code.  When this routine is called in a loop,
  the OptArg strings are returned in the same order as the options and
  arguments appeared in argv.  The OptArg pointer can be NULL.  This is the
  case for an option with no option value.  Each call to UTgetOption returns
  at most one option or argument.

  The return codes are;
    -2 - Error detected
    -1 - Normal end of processing
     0 - Argument string, not associated with an option.  The argument string
	 is pointed to by OptArg.
     n - Option specified by the keyword n'th keyword, n >= 1.  The option
	 value string is pointed to by OptArg, which will be NULL if no option
	 value is allowed, and pointer to the option value string otherwise.

  A typical use of this routine is as follows.
  void main (int argc, const char *argv[])
  {
    static const char *OptTable[] = {"-d**:", "--d*elay=*", NULL};
    const char *OptArg;
    int State;
    int n;

    State = 0;
    while (1) {
      n = UTgetOption (&State, argc, argv, OptTable, &OptArg);
      if (n == -1)
        break;
      switch (n) {
      case 0:
        ... process non-option OptArg
        break;
      case 1:
        ... process option 1
      case 2:
	... process option 2
	break;
      default:
	... error
      }
    }
  }

Parameters:
  <-  int
  UTgetOptions - Status code,
		-2, Error
		-1, End of arguments/options
		 0, Argument value
		>=1, Option code
  <-> int *
  State -	Integer used internally for saving the state of the search.
		This value should be saved from call to call.
  ->  int
  argc -	Number of argument strings in argv
  ->  const char * []
  argv -	Pointers to option/argument strings
  ->  const char * []
  OptTable -	Pointers to the option keywords.  The end of the keyword table
		is signalled by a NULL pointer.  Note that with ANSI C, if
		OptTable is not declared to have the const attribute, an
		explicit cast to (const char **) is required.
  <-  const char * []
  OptArg -	Argument/option string.  For an argument or an option taking a
		value, this is a pointer to null terminated string in argv.  As
		such it should be treated as read only.  If decoded option does
		not take a value, this pointer will be NULL.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.11 $  $Date: 1994/02/03 02:53:04 $

-------------------------------------------------------------------------*/

static char rcsid[]="$Id: UTgetOption.c 1.11 1994/02/03 AFsp-V1R2 $";

#include <stddef.h>		/* NULL */
#include <libtsp.h>
#include <libtsp/UTnucleus.h>

#ifndef __STDC__
#define	const
#endif

#define	ST_MVAL		-2
#define	RET_ERROR	-2
#define	RET_END		-1
#define	RET_ARG		0

int
UTgetOption (State, argc, argv, OptTable, OptArg)

     int *State;
     const int argc;
     const char *argv[];
     const char *OptTable[];
     const char *OptArg[];

{
  int n;
  int status;
  int ip;
  int incr;

  if (*State <= 0)
    ip = 1;
  else
    ip = *State;

/* Check the index, return if finished processing arguments from argv */
  if (ip <= argc-1) {

    /* Decode the option */
    n = UTdecOption (&argv[ip], argc-ip, OptTable, OptArg, &incr);

    if (n >= 1) {
      /* Found an option match */
      status = n;
    }

    else if (n == ST_MVAL) {
      /* Error - missing value */
      UTwarn ("UTgetOption: Value expected for option %s", argv[ip]);
      status = RET_ERROR;
    }

    else {
      /* Not an option */
      if (OptTable[0] != NULL && OptTable[0][0] == '-' && argv[ip][0] == '-') {
	/* The argument has a leading '-' */
	UTwarn ("UTgetOption: Invalid option: %s", argv[ip]);
	*OptArg = NULL;
	status = RET_ERROR;
      }
      else {
	/* Ordinary argument */
	*OptArg = argv[ip];
	status = RET_ARG;
      }
    }
    
    /* Update for next call */
    ip = ip + incr;
    *State = ip;
  }

/* Finished processing */
  else {
    *OptArg = NULL;
    status = RET_END;
  }

  return status;

}
