/*------------- Telecommunications & Signal Processing Lab -------------
                         McGill University

Module:
  void FLpathList (const char Fi[], const char Dlist[], char Fo[])

Purpose:
  Fill in a directory for a file name from a list of paths

Description:
  This routine fills in a directory path name for a file name from a list of
  directories.  The intended purpose of this routine is to fill in the
  directory path from a list of defaults prior to opening the file.  If the
  input file name specifies any directory components, the output name is set
  to the input name.  If the input file name is a simple file name (no
  directory components), each path from the list is used in turn as a directory
  path.  A check is made to see if the corresponding file can be found.  The
  file name of the first file found is returned.  If the search for such a file
  is unsuccessful, the input name is returned.  Note that this behaviour
  effectively puts the current directory at the end of the list of paths.  The
  user can specify a search of the current directory in some other order, by 
  including the current directory specification "." in the list of paths.

  This routine performs the additional service of expanding home directory
  specifications of the form "~" or "~USER" which appear as the first
  component of the path name in the input name or in one of the paths in the
  list of paths.  In the case of the input name containing a home directory
  specification, the resulting expanded name will then have a directory
  component and no further search using the list of paths will occur.  If
  the home directory cannot be found (unknown user, for instance), the "~"
  specification is treated as an ordinary filename string.

  The path name list is a white-space or colon separated list of paths.  For
  example, "./:/usr/users/usera/:~userb".  The path names can have an optional
  trailing / character.

  The list of path names can be specified directly in the Dlist variable, or
  Dlist can specify the name of an environment variable (signalled by a leading
  $ character) which contains the list.

Parameters:
  ->  const char []
  Fi - 		Character string containing the file name
  ->  const char []
  Dlist -	Input character string containing a list of directories, or the
		name of an environment variable (including a leading $).  If
		Dlist specifies an environment variable, the value of this
		variable supplies the list of directories.
  <- char []
  Fo -		Character string containing the file name.  At most
		FILENAME_MAX characters not including the terminating null are
		returned.  This string can overlay the input string if desired.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.7 $  $Date: 1994/02/17 00:35:22 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: FLpathList.c 1.7 1994/02/17 AFsp-V1R2 $";

#include <stdlib.h>	/* getenv prototype */
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libtsp.h>

#ifndef __STDC__
#define	const
#endif

void
FLpathList (Fi, Dlist, Fo)

     const char Fi[];
     const char Dlist[];
     char Fo[];

{
  struct stat Fstat;
  char Fexp[FILENAME_MAX+1];
  char Fdir[FILENAME_MAX+1];
  char Ftrial[FILENAME_MAX+1];
  char *p;
  int nd;
  FILE *fp;

/* Check if the input name already has any directory specified. */
  FLexpHome (Fi, Fexp);
  FLdirName (Fexp, Fdir);
  if (Fdir[0] != '\0') {
    STcopyMax (Fexp, Fo, FILENAME_MAX);
    return;
  }

/* Check for an environment variable */
  if (Dlist[0] == '$') {
    p = getenv (&Dlist[1]);
    if (p == NULL)
      p = "";
  }
  else
    p = (char *) Dlist;

/* Zero length strings */
  if (*p == '\0') {
    STcopyMax (Fexp, Fo, FILENAME_MAX);
    return;
  }

/* Loop over the list of directories */
  while (p != NULL) {
    p = STfindToken (p, ":", "", Ftrial, 1, FILENAME_MAX);

    /* Construct a trial name; supply a / for the directory if needed */
    FLexpHome (Ftrial, Fdir);
    nd = strlen (Fdir);
    if (nd > 0 && Fdir[nd-1] != '/')
      STcatMax ("/", Fdir, FILENAME_MAX);
    FLdefName (Fexp, Fdir, Ftrial);

/* Use stat to determine if the file exists (no check to see if it is a regular
   file or not, or whether it is readable). */
    if (stat (Ftrial, &Fstat) == 0) {
      STcopyMax (Ftrial, Fo, FILENAME_MAX);
      return;
    }
  }

/* No luck in finding a file, return the input name */
  STcopyMax (Fexp, Fo, FILENAME_MAX);

}
