/*------------- Telecommunications & Signal Processing Lab -------------
                          McGill University

Module:
  void FLfullName (const char Fname[], char Fullname[])

Purpose:
  Expand a file name path

Description:
  This routine takes an input file path and tries to expand it to form an
  absolute path.  The typical use of this routine is to log the full file name
  of a file that has already been opened, and hence for which the path is well
  defined and readable.  The path is resolved to try to follow symbolic links.
  The basic strategy is to take the path and try to change the current working
  directory to this path.  If this is unsuccessful, the rightmost component of
  the path is removed and the process is repeated.  When (if) successful, the
  leftmost components are replaced by the current working directory.  This
  results in a resolved, cleaned up path name.  The rightmost components are
  appended to this to form the full path name.  Note that the rightmost
  components may not been resolved, either because the directories specified
  therein do not exist on the current host or are unreadable.

Parameters:
  -> const char []
  Fname -	Input character string with the path name
  <- char []
  Fullname -	Output string with the resolved path name.  This string is at
		most FILENAME_MAX characters long not including the terminating
		null character.  This string will be the input file name if the
		path cannot be resolved.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.9 $  $Date: 1994/02/16 17:28:51 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: FLfullName.c 1.9 1994/02/16 AFsp-V1R2 $";

#include <string.h>
#include <unistd.h>	/* getcwd, chdir prototypes */
#include <libtsp.h>
#include <libtsp/FLnucleus.h>

#ifndef __STDC__
#define	const
#endif

void
FLfullName (Fname, Fullname)

     const char Fname[];
     char Fullname[];

{
  char tname[FILENAME_MAX+1];
  char Bname[FILENAME_MAX+1];
  char Dname[FILENAME_MAX+1];
  char Dsave[FILENAME_MAX+1];
  char ddname[FILENAME_MAX+1];
  char *p;
  int status;
  int n;

/* Return an empty string if the input string is empty */
  if (Fname[0] == '\0') {
    Fullname[0]= '\0';
    return;
  }

  /* Initial separation into directory string and null file name string */
  Bname[0] = '\0';
  STcopyMax (Fname, Dname, FILENAME_MAX);

/* Save the current working directory and make sure we can get back here */
  if (getcwd (Dsave, FILENAME_MAX) == NULL || chdir (Dsave) != 0) {
    STcopyMax (Dname, Fullname, FILENAME_MAX);
    return;
  }

/* 
   Try to "resolve" the path name by doing a chdir to the directory and then
   getting the resulting path name with a getcwd.  In this way, we follow
   symbolic links.  The resulting directory string from getcwd hopefully has
   been cleaned up of things like imbedded "./" and "../" strings.

   If the directory is not readable (or does not exist), an error results.  In
   that case, peel off one level of directory and try again.  This should
   eventually give a readable directory: the path name has a leading "/" or
   ends up being ".".  Note that with these peeled off directories we do not
   get the benefit of having getcwd clean up the directory string.
*/

  ddname[0] = '\0';
  while (Dname[0] != '\0' && (status = chdir (Dname)) != 0 &&
	 strcmp (Dname, ddname) != 0) {
    STcopyMax (Dname, ddname, FILENAME_MAX);
    FLbaseName (ddname, tname);
    FLjoinNames (tname, Bname, Bname);
    FLdirName (ddname, Dname);
  }

/* Found a directory with a successful chdir */
  if ((Dname[0] == '\0' || status == 0) &&
      getcwd (Dname, FILENAME_MAX) != NULL) {

    /* On systems which use automounted disks, the returned current working
       directory may contain a /tmp_mnt/ directory component.  Yet the user can
       refer to the directory without that component.  Here we look for such a
       component.  If found, we remove it and verify that a chdir to that
       modified directory refers to the same place.
    */
    p = strstr (Dname, "/tmp_mnt/");
    if (p != NULL) {
      n = p - Dname + 1;
      STcopyNMax (Dname, ddname, n, FILENAME_MAX);
      STcatMax (&Dname[n+8], ddname, FILENAME_MAX);
      if (chdir (ddname) == 0 && getcwd (tname, FILENAME_MAX) != NULL
	  && strcmp (Dname, tname) == 0)
	STcopyMax (ddname, Dname, FILENAME_MAX);
    }

    /* Form the output name */
    FLjoinNames (Dname, Bname, Fullname);

  }

  else
    /* No success, all we can do is return the input string */
    STcopyMax (Fname, Fullname, FILENAME_MAX);

/* Restore the original working directory */
  if (chdir (Dsave) != 0)
    UTwarn ("FLfullName: Unable to reset current working directory");

}
