/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void FIreadFilt (const char FiltFile[], int Maxcof, float h[], int *Ncof,
		   FILE *fpout)

Purpose:
  Read a filter coefficient file

Description:
  The procedure reads filter coefficients from a filter coefficient file.  The
  first record indicates the type of filter.
    !FIR	- FIR filter, direct form
    !IIR	- IIR filter, cascade of biquad sections
    !ALL	- All-pole filter, direct form
    !WIN	- Window coefficients, direct form
    !CAS	- Cascade analog biquad sections
  Subsequent records contain filter coefficients.  Comment records ('!' in the
  first position of the record) can be interspersed amongst the data.  Data
  records are free form, with data values separated by white-space (as defined
  by isspace).  Commas can also be used to separate data values, but only
  within records, i.e. a comma should not appear at the end a record.  Records
  are limited to 1024 characters.

Parameters:
  ->  const char {}
  FiltFile -	Filter file name
  ->  int
  Maxcof -	Maximum number of coefficients to be returned
  <-  float []
  h -		Array of Ncof output filter coefficients
  <-  int *
  Ncof -	Number of filter coefficients returned
  ->  FILE *
  fpout  -	File pointer for printing filter file information.  If fpout is
		not NULL, information about the filter file is printed on the
		stream selected by fpout.


Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.9 $  $Date: 1994/02/23 23:56:36 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: FIreadFilt.c 1.9 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/STnucleus.h>
#include <libtsp/FIparms.h>

#ifndef __STDC__
#define	const
#endif

#define	MXSTRING	1025

int
FIreadFilt (FiltFile, Maxcof, h, Ncof, fpout)

     const char FiltFile[];
     int Maxcof;
     float h[];
     int *Ncof;
     FILE *fpout;

{
  FILE *fp;
  char string[MXSTRING];
  int Rec, N;
  char *p;
  int nc, Nvr, Nv;
  int FiltType;
  char FullName[FILENAME_MAX+1];

/* Open the filter coefficient file */
  fp = fopen (FiltFile, "r");
  if (fp == NULL)
    UTsysErrHalt ("FIreadFilt: Cannot open file \"%s\"", FiltFile);
  
/* Read the coefficients until end-of-file */
  FiltType = FI_UNDEF;
  Rec = 0;
  N = 0;

  while (1) {

    /* Read a line of input */
    string[0] = '\0';
    p = fgets (string, MXSTRING, fp);
    nc = strlen (string);
    if (p == NULL) {
      if (! feof (fp))
	UTsysErrHalt ("FIreadFilt: Read error in filter file");
      if (nc == 0)
	break;
    }

    /* Check the record length */
    if (nc == MXSTRING && string[nc-1] != '\n')
      UTerrorHalt ("FLreadFilt: Filter file record too long");

    ++Rec;
    Nvr = Maxcof - N;

    /* Process comments */
    if (string[0] == '!') {
      if (Rec == 1)
	FiltType = STkeyMatch (string, FItab) + 1;
    }

    /* Decode data records */
    else if (Nvr > 0) {
      if (STdecFloats (string, 0, Nvr, &h[N], &Nv))
	UTerrorHalt ("FIreadFilt: Data format error in filter file");
      N = N + Nv;
    }

    else {
      if (Nvr == 0 && *(STtrimIws (string)) != '\0')
	UTerrorHalt ("FIreadFilt: Too many records in filter file");
    }
  }

/* Error checks */
  if (FiltType == FI_IIR) {
    if (N % 5 != 0)
      UTerrorHalt ("FIreadFilt: No. IIR coefficients must be a multiple of 5");
  }

/* Print file information */
  if (fpout != NULL) {
    FLfullName (FiltFile, FullName);
    fprintf (fpout, " Filter file: %s\n", FullName);
    fprintf (fpout, "   %s  %s\n", FItype[FiltType], FLfileDate(fp, 3));
    if (FiltType == FI_IIR) {
      if (N == 5)
	fprintf (fpout, "   Number of coefficients: %d  (%d section)\n",
		 N, N / 5);
      else
	fprintf (fpout, "   Number of coefficients: %d  (%d sections)\n",
		 N, N / 5);
    }
    else
      fprintf (fpout, "   Number of coefficients: %d\n", N);
  }

/* Close the file */
  fclose (fp);

/* Return the values */
  *Ncof = N;
  return FiltType;
}
