/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFwriteMulaw (FILE *fp, float Fbuff[], int Nval)

Purpose:
  Write 8-bit mu-law data to an audio file (float input values)

Description:
  This routine writes a specified number of 8-bit mu-law samples to an audio
  file.  The input to this routine is a buffer of float values. The file must
  have been opened using subroutine AFopenWrite.

Parameters:
  ->  FILE *
  fp -		File pointer for a file opened by AFopenWrite
  ->  float []
  Fbuff -	Array of floats with the samples to be written
  ->  int
  Nval -	Number of samples to be written

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.6 $  $Date: 1994/02/23 22:48:17 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFwriteMulaw.c 1.6 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFtypes.h>

#define LW		1
#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define ABSV(x)		(((x) < 0) ? -(x) : (x))
#define NBBUF		8192
#define NLEV		256

/*
   Conversion to mu-law is carried out using a quantization operation.  Given
   an array of (ordered) break points, the interval containing the input value
   is determined.  However, there is an ambiguity at the quantizer break-
   points.  This is of concern if the input data takes on integer values and
   the break points themselves are integers.  The goal is to mimic the
   operation of a standard 16-bit to mu-law conversion procedure.  This is
   usually carried out with a sign-magnitude representation, with break-points
   belonging to the interval below them in magnitude.  For the routine SPquant,
   the intervals are defined as Xq[i-1] < x <= Xq[i].  Here we use SPquant to
   quantize the magnitude, and handle the sign separately.

   Mu-law data is stored in sign-magnitude bit-complemented format. The
   complement was originally specified for telecommunications applications to
   increase the density of 1 bits.  Before complementing the bits, the byte is
   in sign-segment-mantissa format.
   mu-law data  complement  segment step-size       value
   0 000 xxxx   1 111 yyyy     7      256     -8031, ... , -4191 
   0 001 xxxx   1 110 yyyy     6      128     -3999, ... , -2079
   0 010 xxxx   1 101 yyyy     5       64     -1983, ... , -1023
   0 011 xxxx   1 100 yyyy     4       32      -975, ... , -495
   0 100 xxxx   1 011 yyyy     3       16      -471, ... , -231
   0 101 xxxx   1 010 yyyy     2        8      -219, ... , -99
   0 110 xxxx   1 001 yyyy     1        4       -93, ... , -33
   0 111 xxxx   1 000 yyyy     0        2       -30, ... ,  0
   1 000 xxxx   0 111 yyyy     7      256      8031, ... , 4191
   1 001 xxxx   0 110 yyyy     6      128      3999, ... , 2079
   1 010 xxxx   0 101 yyyy     5       64      1983, ... , 1023
   1 011 xxxx   0 100 yyyy     4       32       975, ... , 495
   1 100 xxxx   0 011 yyyy     3       16       471, ... , 231
   1 101 xxxx   0 010 yyyy     2        8       219, ... , 99
   1 110 xxxx   0 001 yyyy     1        4        93, ... , 33
   1 111 xxxx   0 000 yyyy     0        2        30, ... , 0

   SPquant is used to find a bin number for the absolute value of the samples.
   The bin numbers directly correspond to the (uncomplemented) mu-law
   representation.  The remaining steps are to change the sign bit for negative
   values, and to complement the bits.

   bin no.    signal value    bit pattern   complement
    0-15      0, ... , 30     0 000 xxxx   1 111 yyyy
   16-31     33, ... , 93     0 001 xxxx   1 110 yyyy
   32-47     99, ... , 217    0 010 xxxx   1 101 yyyy
   48-63    231, ... , 471    0 011 xxxx   1 100 yyyy
   64-79    495, ... , 975    0 100 xxxx   1 011 yyyy
   80-95   1023, ... , 1983   0 101 xxxx   1 010 yyyy
   96-111  2079, ... , 3999   0 110 xxxx   1 001 yyyy
  112-127  4191, ... , 8031   0 111 xxxx   1 000 yyyy

  Note that the array of break points has 126 values.  There is an additional
  break point at zero created by taking the absolute value.  The total number
  of break points is 255, giving 256 bins.  There are two bins, [-1,0) and
  [0, 1], which have zero as a representative value.
*/

void
AFwriteMulaw (fp, Fbuff, Nval)

     FILE *fp;
     float Fbuff[];
     int Nval;

{
  struct AF_parms *Fparms;
  int is;
  int N;
  int i;
  long int offs;
  uint1 Buf[NBBUF /LW];
  float Fv;
  float Av;
  unsigned char ibin;
  static float Xq[NLEV/2-2] = {
       1.,    3.,    5.,    7.,    9.,   11.,   13.,   15.,
      17.,   19.,   21.,   23.,   25.,   27.,   29.,   31.,
      35.,   39.,   43.,   47.,   51.,   55.,   59.,   63.,
      67.,   71.,   75.,   79.,   83.,   87.,   91.,   95.,
     103.,  111.,  119.,  127.,  135.,  143.,  151.,  159.,
     167.,  175.,  183.,  191.,  199.,  207.,  215.,  223.,
     239.,  255.,  271.,  287.,  303.,  319.,  335.,  351.,
     367.,  383.,  399.,  415.,  431.,  447.,  463.,  479.,
     511.,  543.,  575.,  607.,  639.,  671.,  703.,  735.,
     767.,  799.,  831.,  863.,  895.,  927.,  959.,  991.,
    1055., 1119., 1183., 1247., 1311., 1375., 1439., 1503.,
    1567., 1631., 1695., 1759., 1823., 1887., 1951., 2015.,
    2143., 2271., 2399., 2527., 2655., 2783., 2911., 3039.,
    3167., 3295., 3423., 3551., 3679., 3807., 3935., 4063.,
    4319., 4575., 4831., 5087., 5343., 5599., 5855., 6111.,
    6367., 6879., 7135., 7391., 7647., 7903.
  };

/* Get the file format parameters */
  Fparms = AFgetFormat (fp);

  offs = Fparms->End;
  is = 0;

/* Write data to the audio file */
  while (is < Nval) {
    N = MINV (NBBUF / LW, Nval - is);
    for (i = 0; i < N; ++i) {
      Fv = Fparms->ScaleF * Fbuff[i+is];
      Av = ABSV (Fv);
      ibin = SPquant (&Av, Xq, NLEV/2-1);
      if (Fv < 0.0)
	ibin = ibin | 128;	/* Change the sign bit */
      Buf[i] = ~ibin;           /* Complement */
    }
    AFwriteFile (fp, offs, Buf, LW, N);
    is = is + N;
    offs = offs + LW * N;
  }

  /* Update the file position */
  Fparms->End = offs;
}
