/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFsetInfoAU (FILE *fp, const char Fname[], int Format, long int Nchan,
		    float *Sfreq, FILE *fpout)

Purpose:
  Write header information to an AFsp (or Sun/NeXT) audio file

Description:
  This routine writes header information to an AFsp format audio file.  The
  file is assumed to have been opened with routine AFOpenWrite.  Optionally,
  the file header information is printed.

  AFsp audio file header:
      Bytes     Type    Contents
     0 ->  3    int    File identifier
     4 ->  7    int    Header size H (bytes)
     8 -> 11    int    Audio data length (bytes)
    12 -> 15    int    Data encoding format
    16 -> 19    int    Sample rate (samples per second)
    20 -> 23    int    Number of interleaved channels
    24 -> 27    int    AFsp identifier
    28 -> H-1   --     Additional header information
  For AFsp audio files, 8-bit mu-law, 16-bit linear and 32-bit IEEE floating
  point data formats are supported.

  Conversion parameters for subsequent write operations are set up to scale
  data as follows.
    8-bit mu-law       -->  [ -32124 , +32124 ]
    16-bit linear      -->  [ -32768 , +32767 ]
    32-bit IEEE float  -->  scaled by 1/32768  (file data values nominally
                            between -1 and +1)

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  ->  int
  Format -	Audio file data format code, evaluated as
		Format = Dformat + 256 * NH + 512 * Swapb
		Dformat - data format
		  1, mu-law 8-bit data
		  2, linear twos complement 16-bit data
		  3  32-bit float data
                NH - header flag
                  0, AFsp file header (data in big-endian byte order)
		  1, no header
		Swapb - swap flag.  For AFsp files, the byte swap flag must
		  result in big-endian data.
		  0, big-endian byte order (illegal for AFsp files)
		  1, little-endian byte order
		  2, native (no swap) byte order
		  3, byte-swapped
  ->  long int *
  Nchan -	Number of channels
  ->  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.10 $  $Date: 1994/02/12 20:37:08 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFsetInfoAU.c 1.10 1994/02/12 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AUparms.h>

#ifndef __STDC__
#define	const
#endif

#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

void
AFsetInfoAU (fp, Fname, Format, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     int Format;
     long int Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct AU_head Fhead;
  struct AF_parms Fparms;
  int Dformat;
  int NH;
  int Dencod;
  int Ninfo;
  int Swapb;
  float ScaleF;
  float Sfreqx;
  char *Filetype;
  long int Srate;
  char *h;
  int Lhead;
  char *Filedata;
  char FullName[FILENAME_MAX+1];

/* Decode the format into data format, noheader flag, and swap flag*/
  Dformat = Format % FD_NH;
  NH = (Format / FD_NH) % 2;
  Swapb = Format / FD_BO;
  switch (Swapb) {
  case DS_EB:
    Swapb = (AFbyteOrder () == SF_EB) ? DS_NATIVE : DS_SWAP;
    break;
  case DS_EL:
    Swapb = (AFbyteOrder () == SF_EL) ? DS_NATIVE : DS_SWAP;
    break;
  case DS_NATIVE:
  case DS_SWAP:
    break;
  default:
    UTerrorHalt ("AFsetInfoAU: Invalid byte swap option");
    break;
  }

/* Set up the encoding parameters */
  switch (Dformat) {
  case FD_MULAW8:
    Dencod = AU_MULAW8;
    Filedata = "8-bit mu-law";
    ScaleF = 0.25;
    break;
  case FD_LIN16:
    Dencod = AU_LIN16;
    Filedata = "16-bit integer";
    ScaleF = 1.;
    break;
  case FD_FLOAT32:
    if (AFcheckIEEE ())
      UTerrorHalt ("AFsetInfoAU: Host does not use IEEE float format");
    Dencod = AU_FLOAT32;
    Filedata = "32-bit float";
    ScaleF = 1./32768.;
    break;
  default:
    UTerrorHalt ("AFsetInfoAU: Unsupported data encoding");
    break;
  }
  Sfreqx = *Sfreq;
  Srate = Sfreqx + 0.5;

  if (NH == 0) {

/* Create the header information */
    Filetype = "AFsp";

    /* Get the header information string */
    h = AFgetUinfo (&Sfreqx, &Ninfo);
    Lhead = AU_LHMIN + Ninfo;

    /* Set up the fixed header information */
    Fhead.Magic = FM_SUN[0];
    Fhead.Lhead = Lhead;
    Fhead.Ldata = AU_NOSIZE;
    Fhead.Dencod = Dencod;
    Fhead.Srate = Srate;
    Fhead.Nchan = Nchan;
    Fhead.AFsp = FM_AFSP[0];

    /* Check if the data should be byte-swapped */
    if ((AFbyteOrder () == SF_EB ? DS_NATIVE : DS_SWAP) != Swapb)
      UTerrorHalt ("AFsetInfoAU: Incompatible byte order");
    if (Swapb == DS_SWAP)
      AFswapBytes (&Fhead, U4, AU_LHMIN / U4);

    /* Write out the header */
    AFwriteFile (fp, 0, &Fhead, U4, AU_LHMIN / U4);
    AFwriteFile (fp, AU_LHMIN, h, 1, Ninfo);
  }

  else {

/* Headerless file */
    Lhead = 0;
    Filetype = "Headerless";
  }

/* Print the header information */
  if (fpout != NULL) {
    FLfullName (Fname, FullName);
    fprintf (fpout, " %s audio file: %s\n", Filetype, FullName);
    fprintf (fpout, "   Sampling frequency: %.6g Hz\n", Sfreqx);
    fprintf (fpout, "   Number of channels: %d (%s)\n", Nchan, Filedata);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_WO;
  Fparms.Format = Dformat;
  Fparms.Start = Lhead;
  Fparms.End = Lhead;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = ScaleF;
  AFsetFormat (fp, &Fparms);

}
