/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int AFreadMulaw (FILE *fp, long int offs, float Fbuff[], int Nreq)

Purpose:
  Read 8-bit mu-law data from an audio file (return float values)

Description:
  This routine reads a specified number of 8-bit mu-law samples from an audio
  file.  The data in the file is converted to float format on output. The file
  must have been opened using subroutine AFopenRead.

Parameters:
  <-  int
  AFreadMulaw -	Number of data values transferred from the file.  On reaching
		reaching the end of the file, this value may be less than Nreq.
  ->  FILE *
  fp -		File pointer for a file opened by AFopenRead
  ->  long int
  offs -	Offset into the file in samples.  The parameter offs must be
                non-negative.
  <-  float []
  Fbuff -	Array of floats to receive the samples
  ->  int
  Nreq -	Number of samples requested.  Nreq may be zero.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.9 $  $Date: 1994/02/23 22:48:00 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFreadMulaw.c 1.9 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFtypes.h>

#define LW		1
#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define NBBUF		8192

/* Mu-law data is stored in sign-magnitude bit-complemented format. The
   complement was originally specified for telecommunications applications to
   increase the density of 1 bits.  After complementing the bits, the byte is
   in sign-segment-mantissa format.
   mu-law data  complement  segment step-size       value
   0 000 xxxx   1 111 yyyy     7      256     -8031, ... , -4191 
   0 001 xxxx   1 110 yyyy     6      128     -3999, ... , -2079
   0 010 xxxx   1 101 yyyy     5       64     -1983, ... , -1023
   0 011 xxxx   1 100 yyyy     4       32      -975, ... , -495
   0 100 xxxx   1 011 yyyy     3       16      -471, ... , -231
   0 101 xxxx   1 010 yyyy     2        8      -219, ... , -99
   0 110 xxxx   1 001 yyyy     1        4       -93, ... , -33
   0 111 xxxx   1 000 yyyy     0        2       -30, ... ,  0
   1 000 xxxx   0 111 yyyy     7      256      8031, ... , 4191
   1 001 xxxx   0 110 yyyy     6      128      3999, ... , 2079
   1 010 xxxx   0 101 yyyy     5       64      1983, ... , 1023
   1 011 xxxx   0 100 yyyy     4       32       975, ... , 495
   1 100 xxxx   0 011 yyyy     3       16       471, ... , 231
   1 101 xxxx   0 010 yyyy     2        8       219, ... , 99
   1 110 xxxx   0 001 yyyy     1        4        93, ... , 33
   1 111 xxxx   0 000 yyyy     0        2        30, ... , 0

  The following table implements the conversion, including the complementing
  operation.
*/

static float fmutab[256] = {
  -8031., -7775., -7519., -7263., -7007., -6751., -6495., -6239.,
  -5983., -5727., -5471., -5215., -4959., -4703., -4447., -4191.,
  -3999., -3871., -3743., -3615., -3487., -3359., -3231., -3103.,
  -2975., -2847., -2719., -2591., -2463., -2335., -2207., -2079.,
  -1983., -1919., -1855., -1791., -1727., -1663., -1599., -1535.,
  -1471., -1407., -1343., -1279., -1215., -1151., -1087., -1023.,
   -975.,  -943.,  -911.,  -879.,  -847.,  -815.,  -783.,  -751.,
   -719.,  -687.,  -655.,  -623.,  -591.,  -559.,  -527.,  -495.,
   -471.,  -455.,  -439.,  -423.,  -407.,  -391.,  -375.,  -359.,
   -343.,  -327.,  -311.,  -295.,  -279.,  -263.,  -247.,  -231.,
   -219.,  -211.,  -203.,  -195.,  -187.,  -179.,  -171.,  -163.,
   -155.,  -147.,  -139.,  -131.,  -123.,  -115.,  -107.,   -99.,
    -93.,   -89.,   -85.,   -81.,   -77.,   -73.,   -69.,   -65.,
    -61.,   -57.,   -53.,   -49.,   -45.,   -41.,   -37.,   -33.,
    -30.,   -28.,   -26.,   -24.,   -22.,   -20.,   -18.,   -16.,
    -14.,   -12.,   -10.,    -8.,    -6.,    -4.,    -2.,     0.,
   8031.,  7775.,  7519.,  7263.,  7007.,  6751.,  6495.,  6239.,
   5983.,  5727.,  5471.,  5215.,  4959.,  4703.,  4447.,  4191.,
   3999.,  3871.,  3743.,  3615.,  3487.,  3359.,  3231.,  3103.,
   2975.,  2847.,  2719.,  2591.,  2463.,  2335.,  2207.,  2079.,
   1983.,  1919.,  1855.,  1791.,  1727.,  1663.,  1599.,  1535.,
   1471.,  1407.,  1343.,  1279.,  1215.,  1151.,  1087.,  1023.,
    975.,   943.,   911.,   879.,   847.,   815.,   783.,   751.,
    719.,   687.,   655.,   623.,   591.,   559.,   527.,   495.,
    471.,   455.,   439.,   423.,   407.,   391.,   375.,   359.,
    343.,   327.,   311.,   295.,   279.,   263.,   247.,   231.,
    219.,   211.,   203.,   195.,   187.,   179.,   171.,   163.,
    155.,   147.,   139.,   131.,   123.,   115.,   107.,    99.,
     93.,    89.,    85.,    81.,    77.,    73.,    69.,    65.,
     61.,    57.,    53.,    49.,    45.,    41.,    37.,    33.,
     30.,    28.,    26.,    24.,    22.,    20.,    18.,    16.,
     14.,    12.,    10.,     8.,     6.,     4.,     2.,     0.
};

int
AFreadMulaw (fp, offs, Fbuff, Nreq)

     FILE *fp;
     long int offs;
     float Fbuff[];
     int Nreq;

{
  struct AF_parms *Fparms;
  long int offsb;
  int Nd;
  int is;
  int n;
  int i;
  uint1 Buf[NBBUF / LW];
  uint1 *B;
  int Nval;

/* Get the file format parameters */
  Fparms = AFgetFormat (fp);

  offsb = Fparms->Start + LW * offs;
  Nd = MINV (Nreq, (Fparms->End - offsb) / LW);
  is = 0;

/* Read data from the audio file */
  while (1) {
    n = MINV (NBBUF / LW, Nd);
    if (n <= 0)
      break;
    Nval = AFreadFile (fp, offsb, Buf, LW, n);
    if (Nval != n)
      UTerrorHalt ("AFreadMulaw: Unexpected end-of-file");

    /* Byte swap and convert to float */
    B = Buf;
    for (i = 0; i < Nval; ++i) {
      Fbuff[i+is] = Fparms->ScaleF * fmutab[*B];
      ++B;
    }
    is = is + Nval;
    Nd = Nd - Nval;
    offsb = offsb + LW * Nval;
  }

  return is;
}
