/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int AFgetRecSP (const char name[], int type, void *data, int errcont,
		   const char head[], int N)

Purpose:
  Find a named record in a NIST SPHERE audio file header

Description:
  This routine searches through records in the header of an NIST SPHERE audio
  file, looking for a record with a given name field.  The data value for that
  record is returned.  The data value is an integer value, a double value or a
  string.  For strings, the returned string is null terminated.  This routine
  should only be used for retrieving strings which are known not to contain
  imbedded null characters or imbedded newline characters.  Except for these
  limitations, this routine returns values from correctly formed records.  It
  does not attempt to verify full syntactic correctness of records.

Parameters:
  <-  int
  AFgetRecSP -	Returned error status, 0 for no error, 1 for error
  ->  const char []
  name - 	Record name to be matched
  ->  int
  type -	Data type, 0 for integer, 1 for double, 2 for string
  <-  void *
  data -	Pointer to the where the data is to be written.  For integer
		data, this should be a pointer to an int value.  Similarly for
		double, data, this should be a pointer to a double value.  For
		strings, this should be a pointer to a character string with
		room for 81 characters (80 plus a terminating null).
  ->  int
  errcont -	Error control flag.  The parameter errcnt controls how this
		routine reacts if a matching name is not found or the decoding
		error is detected for a record.
		0 - Return to the calling program.  No error messages are
		    printed.
		1 - Return to the calling program.  Error messages are printed.
		2 - Terminate execution if an error is detected.  Error mesages
		    are printed.
  ->  const char []
  head -	Character string (N characters) with the header records
  ->  int
  N -		Length of the header string

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.3 $  $Date: 1994/01/23 03:02:32 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFgetRecSP.c 1.3 1994/01/23 AFsp-V1R2 $";

#include <string.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>

#ifndef __STDC__
#define	const
#endif

#define MXRECORD	1024
#define MXSTRING	80

#define T_INTEGER	0
#define T_REAL		1
#define T_STRING	2

int
AFgetRecSP (name, type, data, errcont, head, N)

     const char name[];
     int type;
     void *data;
     int errcont;
     const char head[];
     int N;

{
  char *p;
  int d_len;
  char dataf[MXRECORD+1];
  char *valuef;
  int Iv;
  int *Ip;
  double Dv;
  double *Dp;
  int Ncs;

/* Search for a matching record */
  p = AFsepRecSP (name, head, N);
  if (p == NULL) {
    if (errcont == 2)
      UTerrorHalt ("AFgetRecSP: No \"%s\" record in header", name);
    else if (errcont == 1)
      UTwarn ("AFgetRecSP: No \"%s\" record in header", name);
    return 1;
  }

/* Copy the type and value fields to temporary storage */
  d_len = (char *) memchr (p, '\n', N - (p - head)) - p;
  if (d_len > MXRECORD)
    UTerrorHalt ("AFgetRecSP: Record too long"); /* impossible for N <= 1024 */
  memcpy (dataf, p, d_len);
  dataf[d_len] = '\0';
  
/* Find the end of the type field */
  p = strchr (dataf, ' ');
  if (p != NULL) {
    *p = '\0';
    valuef = p + 1;
  }
  else
    valuef = &dataf[d_len];

/* Decode the type field */
  if (type == T_INTEGER) {
    if (strcmp (dataf, "-i") == 0 && sscanf (valuef, "%d", &Iv) == 1) {
      Ip = data;
      *Ip = Iv;
      return 0;
    }
  }
  else if (type == T_REAL) {
    if (strcmp (dataf, "-r") == 0 && sscanf (valuef, "%lg", &Dv) == 1) {
      Dp = data;
      *Dp = Dv;
      return 0;
    }
  }
  else if (type == T_STRING) {
    if (strncmp (dataf, "-s", 2) == 0
	&& sscanf (&dataf[2], "%d", &Ncs) == 1
	&& Ncs > 0 && Ncs <= MXSTRING) {
      STcopyNMax (valuef, (char *) data, Ncs, MXSTRING);
      return 0;
    }
  }

/* Error messages */
  if (errcont == 2)
    UTerrorHalt ("AFgetRecSP: Invalid field in \"%s\" record", name);
  else if (errcont == 1)
    UTwarn ("AFgetRecSP: Invalid field in \"%s\" record", name);
  return 1;
}
