/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoSP (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get header information from a NIST SPHERE audio file

Description:
  This routine reads header information from a NIST SPHERE audio file.
  Optionally, the file header information is printed.

  NIST SPHERE audio file:
     Bytes      Type    Contents
    0 ->  7     char   NIST_1A file identifier
    8 -> 15     char   Header length in ASCII (1024)
   16 -> H-1    struct Object-oriented fields
  For NIST SPHERE audio files, 16-bit linear data encoding is supported.

  Conversion parameters for subsequent read operations are set up to scale data
  as follows.
    16-bit linear      -->  [ -32768 , +32767 ]

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.13 $  $Date: 1994/02/23 22:47:33 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoSP.c 1.13 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>

#ifndef __STDC__
#define	const
#endif

#define LHEAD		1024

/* NIST SPHERE file identifier as a character string */
#define SP_LI	((sizeof SP_ident) - 1)
#define SP_LH	((sizeof SP_hsize) - 1)
static const char SP_ident[] = "NIST_1A\n";
static const char SP_hsize[] = "   1024\n";

#define SP_LR	(LHEAD - SP_LH - SP_LI)

#define T_INTEGER	0
#define T_REAL		1
#define T_STRING	2
#define MXSTRING	80

void
AFgetInfoSP (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct AF_parms Fparms;
  char Fhead_ident[SP_LI];
  char Fhead_hsize[SP_LH];
  char Fhead_rec[SP_LR]; 
  long int Nbytes;
  int Srate;
  long int Nsampx;
  long int Nchanx;
  int Lw;
  int nbis;
  int Format;
  int Swapb;
  int Ldata;
  float ScaleF;
  char FullName[FILENAME_MAX+1];
  char ByteFormat[MXSTRING+1];
  char Date[MXSTRING+1];

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < LHEAD)
    UTerrorHalt ("AFgetInfoSP: File header too short");

/* Read in the fixed portion of the header */
  (void) AFreadFile (fp, 0, Fhead_ident, 1, SP_LI);
  (void) AFreadFile (fp, SP_LI, Fhead_hsize, 1, SP_LH);

/* Check the file magic */
  if (memcmp (Fhead_ident, SP_ident, SP_LI) != 0)
    UTerrorHalt ("AFgetInfoSP: Invalid file identifier");
  if (memcmp (Fhead_hsize, SP_hsize, SP_LH) != 0)
    UTerrorHalt ("AFgetInfoSP: Invalid header length specifier");

/* Read the remaining part of the header */
  (void) AFreadFile (fp, SP_LI + SP_LH, Fhead_rec, 1, SP_LR);

/* Sampling frequency */
  AFgetRecSP ("sample_rate", T_INTEGER, &Srate, 2, Fhead_rec, SP_LR);

/* Number of samples and number of channels */
  AFgetRecSP ("sample_count", T_INTEGER, &Nsampx, 2, Fhead_rec, SP_LR);
  AFgetRecSP ("channel_count", T_INTEGER, &Nchanx, 2, Fhead_rec, SP_LR);

/* Data format */
  AFgetRecSP ("sample_n_bytes",T_INTEGER, &Lw, 2, Fhead_rec, SP_LR);
  AFgetRecSP ("sample_sig_bits", T_INTEGER, &nbis, 2, Fhead_rec, SP_LR);
  AFgetRecSP ("sample_byte_format", T_STRING, ByteFormat, 2, Fhead_rec, SP_LR);
  if (Lw == 2 && nbis == 16) {
    Format = FD_LIN16;
    ScaleF = 1.;
    if (strcmp (ByteFormat, "10") != 0)
      UTerrorHalt ("AFgetInfoSP: Unsupported byte format");
    Swapb = (AFbyteOrder () == SF_EB) ? DS_NATIVE : DS_SWAP;
  }
  else
    UTerrorHalt ("AFgetInfoSP: Unsupported data format");

/* Warnings, error checks */
  Ldata = Nsampx * Lw;
  if (Ldata < 0 || Ldata > Nbytes - LHEAD)
    UTerrorHalt ("AFgetInfoSP: Invalid data length");
  if (Ldata != (Lw * ((Nbytes - LHEAD) / Lw)))
    UTwarn ("AFgetInfoSP: Data length does not match the file length");
  if ((Ldata % (Lw * Nchanx)) != 0)
    UTwarn ("AFgetInfoSP: No. samples inconsistent with no. channels");

/* Set the return parameters */
  *Nsamp = Nsampx;
  *Nchan = Nchanx;
  *Sfreq = Srate;

/* Print the header information */
  if (fpout != NULL) {
    STcopyMax (FLfileDate (fp, 3), Date, 30);
    AFgetRecSP ("recording_date", T_STRING, Date, 0, Fhead_rec, SP_LR);
    FLfullName (Fname, FullName);
    fprintf (fpout, " NIST SPHERE audio file: %s\n", FullName);
    fprintf (fpout, "   Number of samples : %d  %.30s\n", Nsampx, Date);
    fprintf (fpout, "   Sampling frequency: %d Hz\n", Srate);
    fprintf (fpout, "   Number of channels: %d (16-bit integer)\n", Nchanx);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = Format;
  Fparms.Start = LHEAD;
  Fparms.End = LHEAD + Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = ScaleF;
  AFsetFormat (fp, &Fparms);
}
