/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoSF (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get header information from an IRCAM SoundFile audio file

Description:
  This routine reads header information from an IRCAM SoundFile audio file.
  Optionally, the file header information is printed.

  IRCAM SoundFile header:
      Bytes     Type    Contents
     0 ->  3    int    File identifier
     4 ->  7    float  Sampling Frequency
     8 -> 11    int    Number of Channels
    12 -> 15    int    Data Type
    16 -> 1023  --     Additional header information
  1024 -> ...   --     Audio data
  For IRCAM SoundFiles, 16-bit linear and 32-bit IEEE floating point data
  formats are supported.

  Conversion parameters for subsequent read operations are set up to scale data
  as follows.
    16-bit linear      -->  [ -32768 , +32767 ]
    32-bit IEEE float  -->  unity scale factor

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.10 $  $Date: 1994/02/23 22:46:58 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoSF.c 1.10 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFtypes.h>

#ifndef __STDC__
#define	const
#endif

#define LHEAD		1024
#define LHMIN		(sizeof (struct SF_head))

/* File magic values, as read in by a {big-endian, little-endian} machine */
static const uint4 FM_SF[2]   = {0x0001a364, 0x64a30100};

enum {
  SF_LIN16	= 2,	/* linear twos complement 16-bit data */
  SF_FLOAT32	= 4	/* IEEE 32-bit float data (-1. to +1.) */
};

struct SF_head {
  uint4 Magic;		/* File magic */
  float4 Sfreq;		/* Sampling frequency */
  uint4 Nchan;		/* Number of channels */
  uint4 Dencod;		/* Encoding type */
};

void
AFgetInfoSF (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct SF_head Fhead;
  struct AF_parms Fparms;
  int ml;
  int Lw;
  long int Nbytes;
  long int Ldata;
  long int Nsampx;
  int Format;
  char *Filedata;
  float ScaleF;
  char FullName[FILENAME_MAX+1];
  int Swapb;

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < LHEAD)
    UTerrorHalt ("AFgetInfoSF: File header too short");

/* Read in the fixed portion of the header */
  (void) AFreadFile (fp, 0, &Fhead, U4, LHMIN / U4);

/* Check the file magic */
  ml = AFbyteOrder ();
  if (Fhead.Magic != FM_SF[ml])
    UTerrorHalt ("AFgetInfoSF: Invalid file identifier");

/* Check if the data should be byte-swapped */
  Swapb = DS_NATIVE;
  if (ml != SF_EB) {
    Swapb = DS_SWAP;
    AFswapBytes (&Fhead, U4, LHMIN / U4);
  }

/* Set up the decoding parameters */
  switch (Fhead.Dencod) {
  case SF_LIN16:
    Lw = 2;
    Format = FD_LIN16;
    Filedata = "16-bit integer";
    ScaleF = 1.;
    break;
  case SF_FLOAT32:
    Lw = 4;
    Format = FD_FLOAT32;
    Filedata = "32-bit float";
    ScaleF = 1.;
    break;
  default:
    UTerrorHalt ("AFgetInfoSF: Unsupported data encoding");
    break;
  }

/* Warnings, error checks */
  Ldata = Nbytes - LHEAD;
  if ((Ldata % Lw) != 0) {
    UTwarn ("AFgetInfoSF: Non-integral number of samples");
    Ldata = Lw * (Ldata / Lw);
  }
  if ((Ldata % (Lw * Fhead.Nchan)) != 0)
    UTwarn ("AFgetInfoSF: No. samples inconsistent with no. channels");

/* Set the return parameters */
  Nsampx = Ldata / Lw;
  *Nsamp = Nsampx;
  *Nchan = Fhead.Nchan;
  *Sfreq = Fhead.Sfreq;

/* Print the header information */
  if (fpout != NULL) {
    FLfullName (Fname, FullName);
    fprintf (fpout, " SoundFile: %s\n", FullName);
    fprintf (fpout, "   Number of samples : %d  %s\n", Nsampx,
	     FLfileDate (fp, 3));
    fprintf (fpout, "   Sampling frequency: %.5g Hz\n", Fhead.Sfreq);
    fprintf (fpout, "   Number of channels: %d (%s)\n", Fhead.Nchan, Filedata);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = Format;
  Fparms.Start = LHEAD;
  Fparms.End = LHEAD + Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = ScaleF;
  AFsetFormat (fp, &Fparms);

}
