/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoNH (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get file information for a headerless audio file

Description:
  This routine gets file information for an audio file with a non-standard
  header or with no header.  File format information is passed to this routine
  by calling AFSetNH before calling this routine.  Optionally, the file format
  information is printed.

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.10 $  $Date: 1994/02/23 22:46:40 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoNH.c 1.10 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>

#ifndef __STDC__
#define	const
#endif

void
AFgetInfoNH (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct AF_NH *Fdef;
  struct AF_parms Fparms;
  long int Nsampx;
  long int Nbytes;
  long int Ldata;
  int Lw;
  int Swapb;
  char *Filedata;
  char FullName[FILENAME_MAX+1];

/* Get the file data parameters */
  Fdef = AFgetNH ();

/* Check if the data should be swapped */
  Swapb = Fdef->Swapb;
  switch (Swapb) {
  case DS_EB:
    Swapb = (AFbyteOrder () == SF_EB) ? DS_NATIVE : DS_SWAP;
    break;
  case DS_EL:
    Swapb = (AFbyteOrder () == SF_EL) ? DS_NATIVE : DS_SWAP;
    break;
  case DS_NATIVE:
  case DS_SWAP:
    break;
  default:
    UTerrorHalt ("AFgetInfoNH: Invalid byte swap option");
    break;
  }

/* Set up the decoding parameters */
  switch (Fdef->Format) {
  case FD_MULAW8:
    Lw = 1;
    Filedata = "8-bit mu-law";
    break;
  case FD_LIN16:
    Lw = 2;
    Filedata = "16-bit integer";
    break;
  case FD_FLOAT32:
    Lw = 4;
    Filedata = "32-bit float";
    break;
  case FD_UNDEF:
    UTerrorHalt ("AFgetInfoNH: Default data format not defined");
    break;
  default:
    UTerrorHalt ("AFgetInfoNH: Unsupported data encoding");
    break;
  }

/* Warnings, error checks */
  Nbytes = AFfileSize (fp);
  if (Fdef->Start < 0 || Fdef->Start > Nbytes)
    UTerrorHalt ("AFgetInfoNH: Invalid data offset value: %d", Fdef->Start);
  Ldata = Nbytes - Fdef->Start;
  if ((Ldata % Lw) != 0) {
    UTwarn ("AFgetInfoNH: Non-integral number of samples");
    Ldata = Lw * (Ldata / Lw);
  }
  if ((Ldata % (Lw * Fdef->Nchan)) != 0)
    UTwarn ("AFgetInfoNH - No. samples inconsistent with no. channels");

/* Set the return parameters */
  Nsampx = Ldata / Lw;
  *Nsamp = Nsampx;
  *Nchan = Fdef->Nchan;
  *Sfreq = Fdef->Sfreq;

/* Print the header information */
  if (fpout != NULL) {
    FLfullName (Fname, FullName);
    fprintf (fpout, " Audio file: %s\n", FullName);
    fprintf (fpout, "   Number of samples : %d  %s\n", Nsampx,
	     FLfileDate (fp, 3));
    fprintf (fpout, "   Sampling frequency: %.5g Hz\n", Fdef->Sfreq);
    fprintf (fpout, "   Number of channels: %d (%s)\n", Fdef->Nchan, Filedata);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = Fdef->Format;
  Fparms.Start = Fdef->Start;
  Fparms.End = Fdef->Start + Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = Fdef->ScaleF;
  AFsetFormat (fp, &Fparms);

}
