/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoIN (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get header information from an INRS-Telecommunications audio file

Description:
  This routine reads header information from an INRS-Telecommunications audio
  file.  Optionally, the file header information is printed.

  INRS-Telecommunications audio file:
      Bytes     Type    Contents
     0 ->  3    float  Sampling Frequency (VAX float format)
     6 -> 25    char   Creation time (e.g. Jun 12 16:52:50 1990)
    26 -> 29    int    Number of speech samples in the file
  The number of speech samples is checked against the total extent of the file.
  If unspecified, the file size is used to determine the number of samples.
  The data in an INRS-Telecommunications audio file is in 16-bit integer
  format.

  Conversion parameters for subsequent read operations are set up to scale data
  as follows.
    16-bit linear      -->  [ -32768 , +32767 ]

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.12 $  $Date: 1994/02/23 22:45:42 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoIN.c 1.12 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFtypes.h>

#ifndef __STDC__
#define	const
#endif

#define LHEAD		512
#define LHMIN		(sizeof (struct IN_head))
#define LW		2

#define SWAPBV(x)	(AFswapBytes (&(x), sizeof (x), 1))

/*
   Common sampling frequencies in VAX floating point format, as read in by a
   {big-endian, little-endian} machine.  The sampling frequencies recognized
   are 6500, 20000/3, 8000, 10000, 12000, 16000, and 20000 Hz.
*/

#define NINRS	7
static const uint4 FM_INRS[2][NINRS] = {
  {0xcb460020, 0xd0465555, 0xfa460000, 0x1c470040, 0xc5b8ff7f, 0x7a470000,
     0x9c470040},
  {0x200046cb, 0x555546d0, 0x000046fa, 0x4000471c, 0x7fffb8c5, 0x0000477a,
     0x4000479c}
};
static const float VSfreq[NINRS] = {
  6500., 20000./3., 8000., 10000., 12000., 16000., 20000.
};

void
AFgetInfoIN (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  uint4 FHVSfreq;
  char FHDatetime[21];
  uint4 FHNsamp;
  struct AF_parms Fparms;
  int ml;
  long int Nbytes;
  long int Nchanx;
  long int Ldata;
  int Swapb;
  int iSF;
  char Datetime[21];
  char FullName[FILENAME_MAX+1];
  char *p;

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < LHEAD)
    UTerrorHalt ("AFgetInfoIN: File header too short");

/* Read in portions of the header */
  (void) AFreadFile (fp, 0, &FHVSfreq, U4, 1);
  (void) AFreadFile (fp, 6, FHDatetime, 1, 20);
  FHDatetime[21]='\0';
  (void) AFreadFile (fp, 26, &FHNsamp, U4, 1);

/* Use the sampling frequency as a file magic value */
  ml = AFbyteOrder ();
  for (iSF = 0; iSF < NINRS; iSF++) {
    if (FHVSfreq == FM_INRS[ml][iSF])
      break;
  }
  if (iSF >= NINRS)
    UTerrorHalt ("AFgetInfoIN: Invalid file identifier");

/* Check if the data should be byte-swapped */
  if (ml == SF_EL)
    Swapb = DS_NATIVE;
  else {
    Swapb = DS_SWAP;
    SWAPBV (FHNsamp);
  }

/* Notes:
  - Very old INRS-Telecom audio files have ~0 values for unwritten bytes in
    the header.
  - Old INRS-Telecom audio file used the FHNsamp field.  These files were an
    integral number of disk blocks long (512 bytes per disk block), with
    possibly part of the last block being unused.
  - Old INRS-Telecom audio files have only an 18 byte date and time, followed
    by two 0 or two ~0 bytes.
*/

/* Warnings, error checks */
  Ldata = Nbytes - LHEAD;
  if (FHNsamp == 0 || FHNsamp == ~0L) {
    FHNsamp = Ldata / LW;
    if (Ldata != LW * FHNsamp) {
      UTwarn ("AFgetInfoIN: Non-integral number of samples");
      Ldata = LW * (Ldata / LW);
    }
  }
  else if (MSiCeil (Ldata, 512) != MSiCeil (LW * FHNsamp, 512)) {
    UTwarn ("AFgetInfoIN: File header specifies an invalid number of samples");
    FHNsamp = Ldata / LW;
  }

/* Set the return parameters */
  *Nsamp = FHNsamp;
  Nchanx = 1;
  *Nchan = Nchanx;
  *Sfreq = VSfreq[iSF];

/* Print the header information */
  if (fpout != NULL) {
    p = strchr (FHDatetime, '\377');
    if (p != NULL)
      *p = '\0';
    FLfullName (Fname, FullName);
    fprintf (fpout, " INRS-Telecom audio file: %s\n", FullName);
    fprintf (fpout, "   Number of samples : %d  %s\n", FHNsamp, FHDatetime);
    fprintf (fpout, "   Sampling frequency: %.5g Hz\n", VSfreq[iSF]);
    fprintf (fpout, "   Number of channels: %d (16-bit integer)\n", Nchanx);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = FD_LIN16;
  Fparms.Start = LHEAD;
  Fparms.End = LHEAD + Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = 1.0;
  AFsetFormat (fp, &Fparms);

}
