/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoES (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get header information from an ESPS sampled data feature file

Description:
  This routine reads header information from an ESPS sampled data feature file.
  Optionally, the file header information is printed.

  ESPS sampled data feature file header:
      Bytes     Type    Contents
     8 -> 11    --     Header size (bytes)
    12 -> 15    int    Sampled data record size
    16 -> 19    int    File identifier
    40 -> 65    char   File creation date
   124 -> 127   int    Number of samples (may indicate zero)
   132 -> 135   int    Number of doubles in a data record
   136 -> 139   int    Number of floats in a data record
   140 -> 143   int    Number of longs in a data record
   144 -> 147   int    Number of shorts in a data record
   148 -> 151   int    Number of chars in a data record
   160 -> 167   char   User name
   333 -> H-1   --     Generic header items, including "record_freq"
     H -> ...   --     Audio data
  For ESPS sampled data feature files, 16-bit linear and 32-bit IEEE floating
  point data formats are supported.

  Conversion parameters for subsequent read operations are set up to scale data
  as follows.
    16-bit linear      -->  [ -32768 , +32767 ]
    32-bit IEEE float  -->  unity scale factor

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.15 $  $Date: 1994/02/12 20:37:27 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoES.c 1.15 1994/02/12 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/ESparms.h>
#include <libtsp/STnucleus.h>

#ifndef __STDC__
#define	const
#endif

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define SWAPBV(x)	(AFswapBytes (&(x), sizeof (x), 1))

void
AFgetInfoES (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct ES_preamb Fpreamb;
  struct ES_head Fhead;
  struct AF_parms Fparms;
  long int Nbytes;
  int offs, Swapb, Lw;
  long int Nsampx, Nchanx;
  int Format;
  char *Filedata;
  float ScaleF;
  long int Ldata;
  int Ninfo;
  char Cinfo[ES_MXINFO+1];
  char *p;
  double8 FSfreq;
  char FullName[FILENAME_MAX+1];

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < ES_LHMIN)
    UTerrorHalt ("AFgetInfoES: File header too short");

/* Read in the fixed portions of the header */
  (void) AFreadFile (fp, 0, &Fpreamb, U4, sizeof (Fpreamb) / U4);
  offs = sizeof (Fpreamb);
  (void) AFreadFile (fp, offs, &Fhead, U4, sizeof (Fhead) / U4);

/* Check the file magic */
  if (Fpreamb.Magic == FM_ESPS[0])
    Swapb = DS_NATIVE;
  else if (Fpreamb.Magic == FM_ESPS[1])
    Swapb = DS_SWAP;
  else
    UTerrorHalt ("AFgetInfoES: Invalid file identifier");
  if (Fpreamb.Magic != Fhead.Magic)
    UTerrorHalt ("AFgetInfoES: File identifier values do not match");

/* Byte swap the header items */
  if (Swapb == DS_SWAP) {
    AFswapBytes (&Fpreamb, U4, sizeof (Fpreamb) / U4);
    SWAPBV (Fhead.Ndouble);
    SWAPBV (Fhead.Nfloat);
    SWAPBV (Fhead.Nlong);
    SWAPBV (Fhead.Nshort);
    SWAPBV (Fhead.Nchar);
  }

/* Get the data format */
  if (Fhead.Nshort != 0) {
    Lw = 2;
    Nchanx = Fhead.Nshort;
    Format = FD_LIN16;
    Filedata = "16-bit integer";
    ScaleF = 1.;
  }
  else if (Fhead.Nfloat != 0) {
    Lw = 4;
    Nchanx = Fhead.Nfloat;
    Format = FD_FLOAT32;
    Filedata = "32-bit float";
    ScaleF = 1.;
  }
  else {
    UTerrorHalt ("AFgetInfoES: Unsupported data format");
  }
  if (Fpreamb.Record_size != Lw * Nchanx)
    UTerrorHalt ("AFgetInfoES: Unsupported data encoding");

/* Warnings, error checks */
  if (Fpreamb.Data_offset < ES_LHMIN || Fpreamb.Data_offset > Nbytes)
    UTerrorHalt ("AFgetInfoES: Invalid data offset value");
  if (Fhead.Ndrec > 0)
    Ldata = Fhead.Ndrec * Lw * Nchanx;
  else
    Ldata = Nbytes - Fpreamb.Data_offset;
  if (Ldata < 0 || Ldata > Nbytes - Fpreamb.Data_offset)
    UTerrorHalt ("AFgetInfoES: Invalid data length");
  if (Ldata != Nbytes - Fpreamb.Data_offset)
    UTwarn ("AFgetInfoES: Data length does not match the file length");
  if ((Ldata % Lw) != 0) {
    UTwarn ("AFgetInfoES: Non-integral number of samples");
    Ldata = Lw * (Ldata / Lw);
  }
  if ((Ldata % (Lw * Nchanx)) != 0)
    UTwarn ("AFgetInfoES: No. samples inconsistent with no. channels");

/*
   Get the sampling frequency from a "record_freq" record 
   Alignment problems prevent us from setting up a structure to match the
   file data - the double value may not end up on an 8 byte boundary.
*/
  Ninfo = MINV (ES_MXINFO, Fpreamb.Data_offset - ES_LHMIN);
  (void) AFreadFile (fp, ES_LHMIN, Cinfo, 1, Ninfo);
  p = STstrstrNM (Cinfo, "record_freq", Ninfo, 12);
  if (p != NULL) {
    offs = ES_LHMIN + (p - Cinfo) - 4;	/* Beginning of record */
    (void) AFreadFile (fp, offs + 22, &FSfreq, sizeof (FSfreq), 1);
    if (Swapb == DS_SWAP)
      SWAPBV (FSfreq);
  }
  else {
    UTerrorHalt ("AFgetInfoES: Unable to determine sampling frequency");
  }

/* Set the return parameters */
  Nsampx = Ldata / Lw;
  *Nsamp = Nsampx;
  *Nchan = Nchanx;
  *Sfreq = FSfreq;

/* Print the header information */
  if (fpout != NULL) {
    FLfullName (Fname, FullName);
    fprintf (fpout, " ESPS audio file: %s\n", FullName);
    fprintf (fpout, "   Number of samples : %d  %.26s\n", Nsampx,
	     Fhead.Datetime);
    fprintf (fpout, "   Sampling frequency: %.5g Hz\n", FSfreq);
    fprintf (fpout, "   Number of channels: %d (%s)\n", Nchanx, Filedata);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = Format;
  Fparms.Start = Fpreamb.Data_offset;
  Fparms.End = Fpreamb.Data_offset + Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = ScaleF;
  AFsetFormat (fp, &Fparms);

}
