/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFgetInfoAU (FILE *fp, const char Fname[], long int *Nsamp,
		    long int *Nchan, float *Sfreq, FILE *fpout)

Purpose:
  Get header information from an AFsp or Sun/DEC/NeXT audio file

Description:
  This routine reads header information from an AFsp, Sun, DEC or NeXT
  format audio file.  The AFsp audio file format uses a file header which is
  compatible with a Sun/NeXT audio file header.  Optionally, information about
  the audio file can be printed.

  AFsp (also Sun/NeXT/DEC) audio file header:
      Bytes     Type    Contents
     0 ->  3    int    File identifier
     4 ->  7    int    Header size H (bytes)
     8 -> 11    int    Audio data length (bytes)
    12 -> 15    int    Data encoding format
    16 -> 19    int    Sample rate (samples per second)
    20 -> 23    int    Number of interleaved channels
    24 -> 27    int    AFsp identifier
    28 -> H-1   --     Additional header information
     H -> ...   --     Audio data
  For AFsp (Sun/NeXT/DEC) audio files, 8-bit mu-law, 16-bit linear and 32-bit
  IEEE floating point data formats are supported.

  Conversion parameters for subsequent read operations are set up to scale data
  as follows.
    8-bit mu-law       -->  [ -32124 , +32124 ]
    16-bit linear      -->  [ -32768 , +32767 ]
    32-bit IEEE float  -->  scaled by 32768  (file data values nominally
                            between -1 and +1)

Parameters:
  ->  FILE *
  fp -		File pointer for the audio file
  ->  const char []
  Fname -	File name
  <-  long int *
  Nsamp -	Total number of samples in the file (all channels)
  <-  long int *
  Nchan -	Number of channels
  <-  float *
  Sfreq -	Sampling frequency from the file header
  ->  FILE *
  fpout  -	File pointer for printing audio file information.  If fpout is
		not NULL, information about the audio file is printed on the
		stream selected by fpout.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.18 $  $Date: 1994/02/23 22:44:50 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFgetInfoAU.c 1.18 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AUparms.h>

#ifndef __STDC__
#define	const
#endif

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define ABSV(x)		(((x) < 0) ? -(x) : (x))

void
AFgetInfoAU (fp, Fname, Nsamp, Nchan, Sfreq, fpout)

     FILE *fp;
     const char Fname[];
     long int *Nsamp;
     long int *Nchan;
     float *Sfreq;
     FILE *fpout;

{
  struct AU_head Fhead;
  struct AF_parms Fparms;
  int ml;
  long int Nsampx;
  long int Nbytes;
  int Ftype;
  int Lw;
  int Format;
  int Swapb;
  float ScaleF;
  float Sfreqx;
  float Fv;
  char *Filetype;
  char *Filedata;
  int Ninfo;
  char *Hinfo;
  char *Datetime;
  char *Csf;
  char FullName[FILENAME_MAX+1];

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < AU_LHMIN)
    UTerrorHalt ("AFgetInfoAU: File header too short");

/* Read in the fixed portion of the header */
  (void) AFreadFile (fp, 0, &Fhead, U4, AU_LHMIN / U4);

/* Check the file magic */
  ml = AFbyteOrder ();
  if (Fhead.Magic == FM_SUN[ml] && Fhead.AFsp == FM_AFSP[ml]) {
    Ftype = FT_AFSP;
    Filetype = "AFsp";
  }
  else if (Fhead.Magic == FM_SUN[ml]) {
    Ftype = FT_SUN;
    Filetype = "Sun";
  }
  else if (Fhead.Magic == FM_DEC[ml]) {
    Ftype = FT_DEC;
    Filetype = "DEC";
  }
  else
    UTerrorHalt ("AFgetInfoAU: Invalid file identifier");

/* Swap the header data if necessary */
  Swapb = DS_NATIVE;
  if ( (Ftype == FT_AFSP && ml == SF_EL) ||
       (Ftype == FT_SUN && ml == SF_EL) ||
       (Ftype == FT_DEC && ml == SF_EB) ) {
    Swapb = DS_SWAP;
    AFswapBytes (&Fhead, U4, AU_LHMIN / U4);
  }

/* Set up the decoding parameters */
  switch (Fhead.Dencod) {
  case AU_MULAW8:
    Lw = 1;
    Format = FD_MULAW8;
    Filedata = "8-bit mu-law";
    ScaleF = 4.;
    break;
  case AU_LIN16:
    Lw = 2;
    Format = FD_LIN16;
    Filedata = "16-bit integer";
    ScaleF = 1.;
    break;
  case AU_FLOAT32:
    if (AFcheckIEEE ())
      UTerrorHalt ("AFgetInfoAU: Host does not use IEEE float format");
    Lw = 4;
    Format = FD_FLOAT32;
    Filedata = "32-bit float";
    ScaleF = 32768.;
    break;
  default:
    UTerrorHalt ("AFgetInfoAU: Unsupported data encoding");
    break;
  }

/* Warnings, error checks */
  if (Fhead.Lhead < AU_LHMIN || Fhead.Lhead > Nbytes)
    UTerrorHalt ("AFgetInfoAU: Invalid header length");
  if (Fhead.Ldata == AU_NOSIZE)
    Fhead.Ldata = Nbytes - Fhead.Lhead;
  if (Fhead.Ldata > Nbytes - Fhead.Lhead)
    UTerrorHalt ("AFgetInfoAU: Invalid data length");
  if ((Fhead.Ldata % Lw) != 0) {
    UTwarn ("AFgetInfoAU: Non-integral number of samples");
    Fhead.Ldata = Lw * (Fhead.Ldata / Lw);
  }
  if (Fhead.Ldata != (Lw * ((Nbytes - Fhead.Lhead) / Lw)))
    UTwarn ("AFgetInfoAU: Data length does not match the file length");
  if ((Fhead.Ldata % (Lw * Fhead.Nchan)) != 0)
    UTwarn ("AFgetInfoAU: No. samples inconsistent with no. channels");

/* Scan an AFsp header for a "sample_rate:" record */
  Sfreqx = Fhead.Srate;
  if (Ftype == FT_AFSP) {
    Hinfo = AFgetHinfo (fp, &Ninfo);
    Csf = AFgetRecAU ("sample_rate:", Hinfo, Ninfo);
    if (Csf != NULL) {
      if (STdecFloat1 (Csf, &Fv))
	UTwarn ("AFgetInfoAU: Invalid AFsp sample_rate record");
      else if (ABSV(Fv - Fhead.Srate) <= 0.5)
	Sfreqx = Fv;
      else
	UTwarn ("AFgetInfoAU: AFsp sample rate mismatch, %g : %g",
		Sfreqx, Fv);
    }
  }

/* Set the return parameters */
  Nsampx = Fhead.Ldata / Lw;
  *Nsamp = Nsampx;
  *Nchan = Fhead.Nchan;
  *Sfreq = Sfreqx;

/* Print the header information */
  if (fpout != NULL) {

    /* Scan the header for a "date:..." string */
    if (Ftype == FT_AFSP)
      Datetime = AFgetRecAU ("date:", Hinfo, Ninfo);
    else
      Datetime = NULL;
    if (Datetime == NULL)
      Datetime = FLfileDate (fp, 3);

    FLfullName (Fname, FullName);
    fprintf (fpout, " %s audio file: %s\n", Filetype, FullName);
    fprintf (fpout, "   Number of samples : %d  %.30s\n", Nsampx, Datetime);
    fprintf (fpout, "   Sampling frequency: %.6g Hz\n", Sfreqx);
    fprintf (fpout, "   Number of channels: %d (%s)\n", Fhead.Nchan, Filedata);
  }

/* Set the parameters for file access */
  Fparms.Op = FO_RO;
  Fparms.Format = Format;
  Fparms.Start = Fhead.Lhead;
  Fparms.End = Fhead.Lhead + Fhead.Ldata;
  Fparms.Swapb = Swapb;
  Fparms.ScaleF = ScaleF;
  AFsetFormat (fp, &Fparms);

}
