/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int AFfileType (FILE *fp)

Purpose:
  Identify the format of an audio file

Description:
  This routine determines the format for an audio file.  This routine checks
  the header of the file to determine the format of the file.  If the file
  has a header which is recognized, the file is declared to be a file of the
  corresponding type.  If the file header is not recognized, and a default file
  format has been declared (with AFSetNH), the file is declared to be a
  headerless audio file.  Otherwise, the file type is declared to be unknown.

Parameters:
  <-  int
  AFfileType -	Integer code specifying the type of file,
		FT_UNKNOWN = 0,	unknown audio file format
		FT_NH	   = 1,	headerless (non-standard or no header) audio
		                file
		FT_AFSP	   = 2, AFsp audio file
		FT_SUN     = 3, Sun/Next audio file
		FT_DEC     = 4,	DEC audio file
		FT_SPHERE  = 5,	NIST SPHERE audio file
		FT_ESPS    = 6,	ESPS sampled data feature file
		FT_SF      = 7,	IRCAM SoundFile audio file
		FT_INRS    = 8	INRS-Telecom audio file
  ->  FILE *
  fp -		File pointer associated with the audio file

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.13 $  $Date: 1994/02/23 22:44:44 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFfileType.c 1.13 1994/02/23 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFtypes.h>

#ifndef __STDC__
#define	const
#endif

/*
   File magic values, as read in by a {big-endian, little-endian} machine.
   ESPS files can be written in either big-endian or little-endian byte order.
*/
static const uint4 FM_SUN[2]  = {0x2e736e64, 0x646e732e};	/* ".snd"  */
static const uint4 FM_DEC[2]  = {0x2e736400, 0x0064732e};	/* ".sd\0" */
static const uint4 FM_AFSP[2] = {0x41467370, 0x70734641};	/* "AFsp"  */
static const uint4 FM_SPHERE[2] = {0x4e495354, 0x5453494e};	/* "NIST"  */
static const uint4 FM_ESPS[2] = {0x00006a1a, 0x1a6a0000};
static const uint4 FM_SF[2]   = {0x0001a364, 0x64a30100};

/* Common sampling frequencies in VAX floating point format, as read in by a
   {big-endian, little-endian} machine.  The sampling frequencies recognized
   are 6500, 20000/3, 8000, 10000, 12000, 16000, and 20000 Hz. */
#define NINRS	7
static const uint4 FM_INRS[][NINRS] = {
  {0xcb460020, 0xd0465555, 0xfa460000, 0x1c470040, 0xc5b8ff7f, 0x7a470000,
     0x9c470040},
  {0x200046cb, 0x555546d0, 0x000046fa, 0x4000471c, 0x7fffb8c5, 0x0000477a,
     0x4000479c}
};

/* Minimum header lengths */
#define LHMIN_SUN	28
#define LHMIN_AFSP	LHMIN_SUN
#define LHMIN_DEC	LHMIN_SUN
#define LHMIN_SPHERE	1024
#define LHMIN_ESPS	333
#define LHMIN_SF	1024
#define LHMIN_INRS	512

/* Offset to the file magic */
#define MAGIC_OFFS_AFSP		24
#define MAGIC_OFFS_SUN		0
#define MAGIC_OFFS_DEC		0
#define MAGIC_OFFS_SPHERE	0
#define MAGIC_OFFS_ESPS		16
#define MAGIC_OFFS_SF		0
#define MAGIC_OFFS_INRS		0

int
AFfileType (fp)

     FILE *fp;

{
  uint4 Magic;
  int ml;
  long int Nbytes;
  int i;
  struct AF_NH *Fdef;

/* Get the file size and the machine byte order */
  Nbytes = AFfileSize (fp);
  ml = AFbyteOrder ();

/* Check for an AFsp or Sun/NeXT audio file */
  if (Nbytes > LHMIN_SUN) {
    (void) AFreadFile (fp, MAGIC_OFFS_SUN, &Magic, U4, 1);
    if (Magic == FM_SUN[ml]) {
      (void) AFreadFile (fp, MAGIC_OFFS_AFSP, &Magic, U4, 1);
      if (Magic == FM_AFSP[ml])
	return FT_AFSP;
      else
	return FT_SUN;
    }
  }

/* Check for a DEC audio file */
  if (Nbytes > LHMIN_DEC) {
    (void) AFreadFile (fp, MAGIC_OFFS_DEC, &Magic, U4, 1);
    if (Magic == FM_DEC[ml])
      return FT_DEC;
  }

/* Check for an NIST SPHERE audio file */
  if (Nbytes > LHMIN_SPHERE) {
    (void) AFreadFile (fp, MAGIC_OFFS_SPHERE, &Magic, U4, 1);
    if (Magic == FM_SPHERE[ml])
      return FT_SPHERE;
  }

/* Check for an ESPS sampled data feature file */
/* ESPS files can be written in either byte order */
  if (Nbytes > LHMIN_ESPS) {
    (void) AFreadFile (fp, MAGIC_OFFS_ESPS, &Magic, U4, 1);
    if (Magic == FM_ESPS[0] || Magic == FM_ESPS[1])
      return FT_ESPS;
  }

/* Check for an IRCAM SoundFile audio file */
  if (Nbytes > LHMIN_SF) {
    (void) AFreadFile (fp, MAGIC_OFFS_SF, &Magic, U4, 1);
    if (Magic == FM_SF[ml])
      return FT_SF;
  }

/* Check for an INRS-Telecommunications audio file */
  if (Nbytes > LHMIN_INRS ) {
    (void) AFreadFile (fp, MAGIC_OFFS_INRS, &Magic, U4, 1);
    for (i = 0; i < NINRS; i++) {
      if (Magic == FM_INRS[ml][i])
        return FT_INRS;
    }
  }

/* No match, see if a default format has been defined */
  Fdef = AFgetNH ();
  if (Fdef->Format != FD_UNDEF)
    return FT_NH;

/* No match, declare it an unknown file type */
  return FT_UNKNOWN;

}
