/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFsetNH (const char String[])
  struct AF_NH *AFgetNH (void)

Purpose:
  Set defaults for headerless audio files
  Get defaults for headerless audio files

Description:
  This routine sets or gets default audio file data parameters.  These
  parameters are used for files with unrecognized (non-standard) headers or
  files with no headers (raw audio files).  The AFsetNH routine must be called
  before opening the file with AFOpenRead.  The parameters for AFsetNH are
  determined from an input string which consists of a list of parameters
  separated by commas.  The form of the list is
    "Format, Start, Sfreq, Swapb, Nchan, ScaleF"
  In the following, an asterisk marks the shortest possible short form
  accepted for keyword parameters.
  Format - File data format,
	   "undef*ined" - Headerless files will be rejected
	   "m*u-law8    - 8-bit mu-law data
	   "i*nteger16" - 16-bit twos complement integer data
	   "f*loat32"   - 32-bit floating point
  Start -  byte offset to the start of data (integer value)
  Sfreq -  sampling frequency in Hz (floating point number)
  Swapb  - Data byte swap parameter
	   "native" - no byte swapping
	   "little*-endian" - file data is in little-endian byte order and will
		be swapped if the current host uses big-endian byte order
	   "big*-endian" - data is in big-endian byte order and will be swapped
	   	if the current host uses little-endian byte order
	   "swap" - swap the data bytes
  Nchan -  number of channels, the data consists of interleaved samples from
	   Nchan channels (integer value)
  ScaleF - Scaling applied to the data from the file (floating point number)
  The default values correspond to the following string
    "undefined, 0, 8000., native, 1, 1.0"

  Leading and trailing white-space (as defined by isspace) is removed from each
  item.  In the input string, any of the parameters may be omitted, in which
  case whatever value has been previously set remains in effect for that
  parameter.  The string ",512, 10000." would set the Start and Sfreq
  parameters and leave the other parameters undisturbed.

  If the input string contains has a leading '$', the string is assumed to
  specify the name of an environment variable after the '$'.  This routine
  uses the value of this environment variable to determine the parameters.  If
  this routine is called as AFsetNH("$AFNHparms"), this routine would look for
  the parameter string in environment variable AFNHparms.

  The procedure AFgetNH is used internally by the audio file routines. It
  returns values in a structure of type AF_NH.  The structure and the format
  codes for the structure elements are defined in the header file AFparms.h.

Parameters:
  AFsetNH:
  ->  const char []
  String -	String containing the list of parameters for headerless files
		or the name of an environment variable (with a leading $)

  AFgetNH:
  <-  struct AF_NH *
  AFgetNH -	Structure with file data parameters
		int      Format - Data format
		int      Start  - Offset in bytes to the start of data
		float    Sfreq  - Sampling frequency
		int      Swapb  - Byte swap flag
		long int Nchan  - Number of channels
		float    ScaleF - Scale factor

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.12 $  $Date: 1994/02/12 15:26:00 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFsetNH.c 1.12 1994/02/12 AFsp-V1R2 $";

#include <stdlib.h>	/* getenv prototype */
#include <string.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>

#ifndef __STDC__
#define	const
#endif

/* Default values */
static struct AF_NH NHparms = { FD_UNDEF, 0, 8000., DS_NATIVE, 1, 1.};

void
AFsetNH (String)

     const char String[];

{
  int n;
  int k;
  int nc;
  const char *p;
  char *token;

  /* Keyword tables */
  static const char *Format_keys[] = {
    "undef*ined", "m*u-law8", "i*nteger16", "f*loat32", NULL};
  static const char *Swapb_keys[] = {
    "native", "little*-endian", "big*-endian", "swap", NULL};

  /* Values corresponding to keywords */
  static const int Format_values[] = {
    FD_UNDEF, FD_MULAW8, FD_LIN16, FD_FLOAT32};
  static const int Swapb_values[] = {
    DS_NATIVE, DS_EL, DS_EB, DS_SWAP};

/* Check for an environment variable */
  if (String[0] == '$') {
    p = getenv (&String[1]);
    if (p == NULL)
      p = "";
  }
  else
    p = String;

/* Allocate temporary storage */
  nc = strlen (p);
  token = (char *) UTmalloc (nc + 1);

/* Separate the parameters */
  k = 0;
  while (p != NULL) {
    p = STfindToken (p, ",", "", token, 1, nc);
    if (token[0] != '\0') {

      /* Decode the parameter values */
      switch (k) {
      case 0:
	n = STkeyMatch (token, Format_keys);
	if (n < 0)
	  UTwarn ("AFsetNH: Invalid format keyword, \"%.10s\"", token);
	else
	  NHparms.Format = Format_values[n];
	break;
      case 1:
	if (STdecInt1 (token, &NHparms.Start))
	  UTwarn ("AFsetNH: Invalid start value, \"%.10s\"", token);
	break;
      case 2:
	if (STdecFloat1 (token, &NHparms.Sfreq))
	  UTwarn ("AFsetNH: Invalid sampling frequency, \"%.10s\"", token);
	break;
      case 3:
	n = STkeyMatch (token, Swapb_keys);
	if (n < 0)
	  UTwarn ("AFsetNH: Invalid swap keyword, \"%.10s\"", token);
	else
	  NHparms.Swapb = Swapb_values[n];
	break;
      case 4:
	if (STdecInt1 (token, &NHparms.Nchan))
	  UTwarn ("AFsetNH: Invalid number of channels, \"%.10s\"", token);
	break;
      case 5:
	if (STdecFloat1 (token, &NHparms.ScaleF))
	  UTwarn ("AFsetNH: Invalid scaling factor, \"%.10s\"", token);
	break;

      }
    }
    ++k;
  }
  UTfree (token);
}

struct AF_NH *
AFgetNH ()

{
  return &NHparms;
}
